#### Mundim F.M & Gibson A.K. (2022). Genetic interactions and parasite diversity underlie the 
#    success of biological control of the root-knot nematode Meloidogyne arenaria with its 
#    bacterial parasite Pasteuria penetrans.  

#### Packages ####
library(dplyr)
source("glmm_funs.r")
library(lme4)
library(lmtest)
library(reshape)
library(MASS)
library(emmeans) 
library (ggplot2)
library(car)
library(gplots)
library(tidyverse)
library(grid)
library(RColorBrewer)
library(hrbrthemes)
library(viridis)

#############################################################################################
#### Experiment 1 ####
# 13 host lines combined with four parasite sources to test for GxG interactions
## importing data for experiment 1 ##
dat<-read.csv("FM&AG_2022_Exp1.csv",header=T)
summary(dat)
dat$replicate=factor(dat$replicate) # should be a factor
dat1=subset(dat,dat$use=="y") # excludes flasks without nematodes
summary(dat1)

#creating a percentage column with the rate data
dat1<-dplyr::mutate(dat1, prop.with.spores_perct = (prop.with.spores*100)) 

#set data in long form data for analysis of load
dat1_long<-reshape(dat1, direction = "long", v.names = "spores", 
                    varying = c("n1","n2","n3","n4","n5","n6","n7","n8","n9","n10","n11","n12",
                                "n13","n14","n15"))

dat1_long<-na.omit(dat1_long) # remove na's #but consider all the nematodes (with and withous spores)
dat1_long<-dplyr::mutate(dat1_long, UniqueID = row_number()) #creating a unique ID for each nematode (independent of combination or flask)

dat1_long2<-subset(dat1_long,dat1_long$spores>0) # to consider only nematodes with spores
dat1_long2<-dplyr::mutate(dat1_long2, UniqueID = row_number()) #creating a unique ID for each nematode (independent of combination or flask)

#### Experiment 1: summary stats ####
# mean number nematodes per flask
meanN1 <- dat1 %>% 
  group_by() %>% 
  summarise( 
    Total = n(), 
    Mean=mean(Origin.total),
    sem=sd(Origin.total)/sqrt(length(Origin.total)))
as.data.frame(meanN1)

# mean number hosts per host line
meanN2 <- dat1 %>% # can use flasks with no nematodes here
  group_by(Host) %>% 
  summarise( 
    Total = n(), 
    Mean=mean(Origin.total),
    sem=sd(Origin.total)/sqrt(length(Origin.total)))
as.data.frame(meanN2)

# mean attachment rate overall
meanN3<- dat1 %>%
  group_by() %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(meanN3)

# mean attachment rate per host line
meanN4 <- dat1 %>% 
  group_by(Host) %>% 
  summarise( 
    Total = n(),  
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(meanN4)

# mean attachment rate per parasite lineage
meanN5 <- dat1 %>% 
  group_by(Past.names) %>% 
  summarise( 
    Total = n(),  
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(meanN5)

# mean attachment load overall - only hosts with spores>0
meanN6 <- dat1_long2 %>% 
  group_by() %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(meanN6)

# mean load by host
meanN7 <- dat1_long2 %>% 
  group_by(Host) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(meanN7)

# mean load by parasite
meanN8 <- dat1_long2 %>% 
  group_by(Past.names) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(meanN8)

#### Experiment 1: analysis of attachment rate - Table 1a, S1a####
model0= glm(cbind(Nemat.Without.Past, Nemat.With.Past) ~ 1, data = dat1, family = "binomial")
#full model
model1 = glm(cbind(Nemat.Without.Past, Nemat.With.Past) ~ Host * Past.names, data = dat1, family = "binomial")
dp = sum(residuals(model1,type ="pearson")^2)/model1$df.residual # test of overdispersion
1- logLik(model1)/logLik(model0) 
# drop interaction
model2 = glm(cbind(Nemat.Without.Past, Nemat.With.Past) ~ Host + Past.names, data = dat1, family = "binomial")
#drop parasite
model3 = glm(cbind(Nemat.Without.Past, Nemat.With.Past) ~ Host, data = dat1, family = "binomial")
# drop host
model4 = glm(cbind(Nemat.Without.Past, Nemat.With.Past) ~ Past.names, data = dat1, family = "binomial")

#model selection table
f.aic<-c(AIC(model1),AIC(model2),AIC(model3),AIC(model4)) 
delAIC<-f.aic-min(f.aic)
relLik <- exp(-0.5 * delAIC)
aicweight <- relLik/sum(relLik)
aic.table<-data.frame(AIC = f.aic, delAIC = delAIC, relLik = relLik, 
                      weight = aicweight)
rownames(aic.table) <- c("host*parasite","host+parasite","host","parasite")
aic.table # Table 1a
BIC(model1, model2, model3, model4)
Anova(model1,type=2) # Table S1a
summary(model1)

#### Experiment 1: analysis of attachment load - Table 1 ####
# uses longform data with all hosts with >0 endospores

#full model
load1 = glmer(spores ~ Host * Past.names + (1|UniqueID), data=dat1_long2, family="poisson", control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
# drop interaction
load2 = glmer(spores ~ Host + Past.names + (1|UniqueID), data=dat1_long2, family="poisson", control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
# drop host
load4 = glmer(spores ~ Past.names + (1|UniqueID), data=dat1_long2, family="poisson", control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
# drop parasite
load3 = glmer(spores ~ Host + (1|UniqueID), data=dat1_long2, family="poisson", control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
# model selection table
f.aic<-c(AIC(load1),AIC(load2),AIC(load3),AIC(load4)) 
delAIC<-f.aic-min(f.aic)
relLik <- exp(-0.5 * delAIC)
aicweight <- relLik/sum(relLik)
aic.table<-data.frame(AIC = f.aic, delAIC = delAIC, relLik = relLik, 
                      weight = aicweight)
rownames(aic.table) <- c("host*parasite","host+parasite","host","parasite")
aic.table # table 1b
BIC (load1, load2, load3, load4)
Anova(load1,type=2) #table s1b
summary(load1)


###################################################################################
#### Experiment 2 ####
# four host lines with 8 sources of parasite (low diversity) and their combination (high diversity)

# data input
dat2<-read.csv("FM&AG2022_Exp2.csv",header=T)
summary(dat2)
dat2$replicate=factor(dat2$replicate)

#creating a percentage column with the rate data
dat2<-dplyr::mutate(dat2, prop.with.spores_perct = (prop.with.spores*100)) 

#long form data for load analysis
dat2_long<-reshape(dat2, direction = "long", v.names = "spores", 
                   varying = c("n1","n2","n3","n4","n5","n6","n7","n8","n9","n10","n11","n12",
                               "n13","n14","n15","n16","n17","n18","n19","n20","n21","n22","n23",
                               "n24","n25","n26","n27","n28","n29","n30"))

dat2_long<-na.omit(dat2_long) # remove na's #but consider all the nematodes (with and withous spores)
dat2_long<-dplyr::mutate(dat2_long, UniqueID = row_number()) #creating a unique ID for each nematode (independent of combination or flask)

dat2_long2<-subset(dat2_long,dat2_long$spores>0) # to consider only nematodes with spores
dat2_long2<-dplyr::mutate(dat2_long2, UniqueID = row_number()) #creating a unique ID for each nematode (independent of combination or flask)

#### Experiment 2: summary stats ####
# n nematodes
mean1 <- dat2 %>% 
  group_by() %>% 
  summarise( 
    Total = n(), 
    Mean=mean(Total_Nemat),
    sem=sd(Total_Nemat)/sqrt(length(Total_Nemat)))
as.data.frame(mean1)

# attachment rate, overall
mean2 <- dat2 %>% 
  group_by() %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean2)

# attachment rate by host
mean3 <- dat2 %>% 
  group_by(Host) %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean3)

# attachment rate by parasite
mean4 <- dat2 %>% 
  group_by(Past.names) %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean4)

# attachment rate by diversity level
mean5 <- dat2 %>% 
  group_by(Diversity) %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean5)

# attachment rate by host, high diversity
dat2_div=subset(dat2,dat2$Diversity=="HIGH")
mean6 <- dat2_div %>%
  group_by(Host) %>%
  summarise(
    Total = n(),
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean6)

# attachment rate by host, low diversity
dat2_low=subset(dat2,dat2$Diversity=="LOW")
mean6b <- dat2_low %>%
  group_by(Host) %>%
  summarise(
    Total = n(),
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean6b)

mean6c <- dat2 %>%  #Table S3a
  group_by(Host,Past.names) %>% 
  summarise( 
    Total = n(), 
    Mean=mean(prop.with.spores),
    sem=sd(prop.with.spores)/sqrt(length(prop.with.spores)))
as.data.frame(mean6c)

# mean attachment load overall - only hosts with spores>0
mean7 <- dat2_long2 %>% 
  group_by() %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean7)

# mean load by host
mean8 <- dat2_long2 %>% 
  group_by(Host) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean8)

# mean load by parasite
mean9 <- dat2_long2 %>% 
  group_by(Past.names) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean9)

# mean load by diversity level
mean10 <- dat2_long2 %>% 
  group_by(Diversity) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean10)

# mean load by host, high diversity 
dat2_long_div=subset(dat2_long2,dat2_long2$Diversity=="HIGH")
mean11 <- dat2_long_div %>% 
  group_by(Host) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean11)
 
# mean load by host, low diversity only
dat2_long_low=subset(dat2_long2,dat2_long2$Diversity=="LOW")
mean12 <- dat2_long_low %>% 
  group_by(Host) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean12)

# mean load by host*parasite Table S3b
mean13 <- dat2_long2 %>% 
  group_by(Host,Past.names) %>% # 
  summarise( 
    Total = n(),  
    Mean=mean(spores),
    sem=sd(spores)/sqrt(length(spores)))
as.data.frame(mean13)

#### Experiment 2: analysis of diversity on attachment rate - Table S2a ####
M1 = glmer(cbind(Nemat.Without.Past,Nemat.With.Past) ~ Diversity + (1|Host) + (1|Past.names) + (1|concat), data = dat2, family = "binomial") # concat id corrects for overdispersion
M0<-glmer(cbind(Nemat.Without.Past,Nemat.With.Past) ~ (1|Host) + (1|Past.names) + (1|concat), data = dat2, family = "binomial")
AIC(M0,M1)
BIC(M0,M1)
anova(M0,M1)
lrtest (M0,M1)
f.aic<-c(AIC(M1),AIC(M0)) 
delAIC<-f.aic-min(f.aic)
relLik <- exp(-0.5 * delAIC)
aicweight <- relLik/sum(relLik)
aic.table<-data.frame(AIC = f.aic, delAIC = delAIC, relLik = relLik, 
                      weight = aicweight)
rownames(aic.table) <- c("diversity","intercept")
aic.table 
summary(M1) # Table S2a

#
#### Experiment 2: Pairwise comparison for rate (high vs. low diversity sources) - Table S3a####
P1 = glm(cbind(Nemat.Without.Past,Nemat.With.Past)~ Past.names * Host, data = dat2, family = "binomial")
summary (P1) 
P1emms <- emmeans(P1, "Past.names","Host")
pairs(P1emms) #Table S3a

#
#### Experiment 2: analysis of attachment rate to test for interactions as in Experiment 1 - Table S4a####
dat2sub<-subset(dat2,dat2$Diversity!="HIGH") # include only single sources here to replicate Experiment 1
summary(dat2sub)

mod0<-glm(cbind(Nemat.Without.Past,Nemat.With.Past) ~ 1, data = dat2sub, family = "binomial")
mod1<-glm(cbind(Nemat.Without.Past,Nemat.With.Past) ~ Host*Past.names, data = dat2sub, family = "binomial")
dp = sum(residuals(mod1,type ="pearson")^2)/mod1$df.residual
1- logLik(mod1)/logLik(mod0) 

mod2<-glm(cbind(Nemat.Without.Past,Nemat.With.Past) ~ Host+Past.names, data = dat2sub, family = "binomial")
mod3<-glm(cbind(Nemat.Without.Past,Nemat.With.Past) ~ Host, data = dat2sub, family = "binomial")
mod4<-glm(cbind(Nemat.Without.Past,Nemat.With.Past) ~ Past.names, data = dat2sub, family = "binomial")
f.aic<-c(AIC(mod1),AIC(mod2),AIC(mod3),AIC(mod4)) 
delAIC<-f.aic-min(f.aic)
relLik <- exp(-0.5 * delAIC)
aicweight <- relLik/sum(relLik)
aic.table<-data.frame(AIC = f.aic, delAIC = delAIC, relLik = relLik, 
                      weight = aicweight)
rownames(aic.table) <- c("host*parasite","host+parasite","host","parasite")
aic.table # Table S4a
BIC (mod1, mod2, mod3, mod4)
summary(mod1)

####Experiment 2: analysis of attachment load by diversity level - Table S2b####
L1 = glmer(spores~ Diversity + (1|Host) + (1|Past.names) + (1|concat), data = dat2_long2, family = "poisson")
overdisp_fun(L1)
L0 = glmer(spores~ (1|Host) + (1|Past.names) + (1|concat), data = dat2_long2, family = "poisson")
AIC(L1,L0)
summary(L1)
# table s2b

#
#### Experiment 2: Pairwise comparison for rate (high vs. low diversity sources) - Table S3b####
P2 = glm(spores ~ Past.names * Host, data = dat2_long2, family = "poisson")
summary (P2) 
P2emms <- emmeans(P2, "Past.names","Host")
pairs(P2emms) #Table S3b

#### Experiment 2: analysis of attachment load to test for interactions as in Experiment 1 - Table S4b####
dat2_long2_sub<-subset(dat2_long2,dat2_long2$Diversity!="HIGH") # include only single sources here to replicate Experiment 1
summary(dat2_long2_sub)

loadb1 = glmer(spores ~ Host * Past.names + (1|concat), data=dat2_long2_sub, family="poisson",control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
overdisp_fun(loadb1) 
loadb2 = glmer(spores ~ Host + Past.names + (1|concat), data=dat2_long2_sub, family="poisson",control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
loadb3 = glmer(spores ~ Host  + (1|concat), data=dat2_long2_sub, family="poisson",control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))
loadb4 = glmer(spores ~ Past.names  + (1|concat), data=dat2_long2_sub, family="poisson",control=glmerControl(optimizer="bobyqa",optCtrl=list(maxfun=1e5)))

f.aic<-c(AIC(loadb1),AIC(loadb2),AIC(loadb3),AIC(loadb4)) 
delAIC<-f.aic-min(f.aic)
relLik <- exp(-0.5 * delAIC)
aicweight <- relLik/sum(relLik)
aic.table<-data.frame(AIC = f.aic, delAIC = delAIC, relLik = relLik, 
                      weight = aicweight)
rownames(aic.table) <- c("host*parasite","host+parasite","host","parasite")
aic.table
BIC (loadb1, loadb2, loadb3, loadb4) #Table S4b
summary(loadb1)

#### Figures ####

#### Figure 1: A) Attachment rate by host ####
# dat<-read.csv("FM&AG_2022_Exp1.csv",header=T)
# dat1=subset(dat,dat$use=="y")
summary(dat1)
library(plyr) #for the figures#
a1 = ddply(dat1, c("Host"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(dat1, c("Host"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)

f1A = ggplot(a1, aes(x = Host, y = mean)) +
  geom_bar(stat="identity", color="darkgrey", fill = "darkgrey") +
  coord_cartesian(ylim=c(0, 105))+
  annotate("text",x=1, y=5, label="a", size=5)+
  annotate("text",x=2, y=5, label="a", size=5)+
  annotate("text",x=3, y=5, label="a", size=5)+
  annotate("text",x=4, y=12, label="a", size=5)+
  annotate("text",x=5, y=30, label="b", size=5)+
  annotate("text",x=6, y=46, label="b", size=5)+
  annotate("text",x=7, y=61.5, label="de", size=5)+
  annotate("text",x=8, y=56, label="cd", size=5)+
  annotate("text",x=9, y=60.5, label="de", size=5)+
  annotate("text",x=10, y=62, label="e", size=5)+
  annotate("text",x=11, y=85, label="f", size=5)+
  annotate("text",x=12, y=87, label="f", size=5)+
  annotate("text",x=13, y=105, label="g", size=5)
f1Aa = f1A+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
f1Aaa= f1Aa + theme_bw() + theme(panel.border = element_blank(),
                                 text = element_text(size = 20),
                                 panel.grid.major = element_blank(), 
                                 panel.grid.minor = element_blank(), 
                                 axis.line.x = element_line(colour = "black"),
                                 axis.line.y = element_line(colour = "black"),
                                 legend.position = "none")
f1Ab = f1Aaa + ylab (label =  "Attachment rate") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a)") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fig1A = f1Ab + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fig1A

#### Figure 1: B) Attachment load by host ####
summary (dat1_long2)

f1B= ggplot(dat1_long2, aes(x = Host, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  #geom_point(aes(fill = Host), size = 3, shape = 21, position = position_dodge()) +
  #geom_point(size = 2, shape = 21,  color = 'black', fill ='black', position=position_jitter(0.1)) +
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fi1B = f1B + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "Host line")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(b)") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fig1B = fi1B + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fig1B

pushViewport(viewport(layout = grid.layout(2, 1)))
print(fig1A, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(fig1B, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))


#### Figure 2: A) Attachment rate by parasite source ####
a2 = ddply(dat1, c("Past.names"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(dat1, c("Past.names"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
f2A = ggplot(a2, aes(x = Past.names, y = mean)) +
  geom_bar(stat="identity", color="darkgrey", fill = "darkgrey") +
  coord_cartesian(ylim=c(0, 105))+
  annotate("text",x=1, y=68, label="a", size=5)+
  annotate("text",x=2, y=48, label="b", size=5)+
  annotate("text",x=3, y=55, label="bc", size=5)+
  annotate("text",x=4, y=48, label="b", size=5)
f2Aa = f2A+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
f2Aa
f2Ab = f2Aa + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+
  ggtitle ("(a)") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))

fig2A= f2Ab + theme_bw() + theme(panel.border = element_blank(), 
                                 text = element_text(size = 20), 
                                 panel.grid.major = element_blank(),
                                 panel.grid.minor = element_blank(), 
                                 axis.line = element_line(colour = "black"),
                                 legend.position = "none")
fig2A

#### Figure 2: B) Attachment load by parasite source ####
f2B= ggplot(dat1_long2, aes(x = Past.names, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  #geom_point(aes(fill = Host), size = 3, shape = 21, position = position_dodge()) +
  #geom_point(size = 2, shape = 21,  color = 'black', fill ='black', position=position_jitter(0.1)) +
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fig2B =f2B +  ylab (label = "Attachment load")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20))+
  ggtitle ("(b)") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 23))+
  xlab (label = "Parasite source")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))
fig2B

pushViewport(viewport(layout = grid.layout(2, 1)))
print(fig2A, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(fig2B, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))



#### Figure 3: A) Attachment rate - Heat Map ####
# Building the file (with means) to be use for the heat map figure
# Building the file (with means) to be use for the heat map figure
means.semA <- ddply(dat1, c("Host", "Past.names"), summarise,
                    mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))

#data above saved in csv format
heat<-read.table("FM&AG2022_HeatMap.csv", sep=",", header=T) # open the original table
summary (heat)

#Heat map
heat <- heat[order(heat$Host),]
#name the rows instead of row number
row.names(heat) <- heat$Host
#Now the rows are named, and we don't need the first column anymore 
heat <- heat[,2:5]
#The data was loaded into a data frame, but it has to be a data matrix to make your heatmap
heat_matrix <- data.matrix(heat)
#making a heatmap
heat_map <- heatmap(heat_matrix, Rowv=NA, Colv=NA, col = heat.colors(200), scale="column", margins=c(5,10))
#changing the color of the heat map
coul <- colorRampPalette(brewer.pal(5, "PiYG"))(25)
colMain <- colorRampPalette(brewer.pal(5, "Blues"))(25)
heatmap(heat_matrix, Rowv=NA, Colv=NA, scale = "none", col = colMain, 
        trace = "none", density.info = "none") 
#another way to plot the heat map 3A
heatmap.2(heat_matrix,
          cellnote = heat_matrix,  # same data set for cell labels
          notecol="black",      # change font color of cell labels to black
          density.info="none",  # turns off density plot inside color legend
          trace="none",         # turns off trace lines inside the heat map
          col=colMain,          # use on color palette defined earlier
          dendrogram="none",     # turn on row clustering
          Rowv=FALSE,
          Colv=FALSE)            # turn off column clustering


#### Figure 3: B) Attachment load - Heat Map ####
#Build the file for the heat map figure
means.semB <- ddply(dat1_long2, c("Host", "Past.names"), summarise,
                    mean=mean(spores), sem=sd(spores)/sqrt(length(spores)))
means.semB

#opening the file (now in csv format) 
heatB<-read.table("FM&AG2022_HeatMap3B.csv", sep=",", header=T) # open the original table
summary (heatB)

#heat map
heatB <- heatB[order(heatB$Host),]
row.names(heatB) <- heatB$Host
heatB <- heatB[,2:5]
heat_matrixB <- data.matrix(heatB)
heat_mapB <- heatmap(heat_matrixB, Rowv=NA, Colv=NA, col = cm.colors(256), scale="column", margins=c(5,10))
coul <- colorRampPalette(brewer.pal(5, "PiYG"))(25)
colMain <- colorRampPalette(brewer.pal(5, "Blues"))(25)
heatmap(heat_matrixB, Rowv=NA, Colv=NA, scale = "none", col = colMain, 
        trace = "none", density.info = "none")
#another way to plot the heat map 3B
heatmap.2(heat_matrixB,
          cellnote = heat_matrixB,  # same data set for cell labels
          notecol="black",      # change font color of cell labels to black
          density.info="none",  # turns off density plot inside color legend
          trace="none",         # turns off trace lines inside the heat map
          col=colMain,          # use on color palette defined earlier
          dendrogram="none",     # turn on row clustering
          Rowv=FALSE,
          Colv=FALSE)            # turn off column clustering

#### Figure 4: Parasite diversity ####
#overall# (a)
DD1 = ddply(dat2, c("Diversity"), summarise,
            mean=mean(prop.with.spores_perct))
means.sem <- ddply(dat2, c("Diversity"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
f4Aa = ggplot(DD1, aes(x = Diversity, y = mean, fill = Diversity)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  # annotate("text",x=1, y=35, label="a", size=5)+
  # annotate("text",x=2, y=70, label="b", size=5)+
  scale_x_discrete(limits = rev)+
  scale_fill_manual(values=c("#999999", "#CCCCCC"))

f4Ab = f4Aa+ geom_errorbar(aes(ymax=upper,
                               ymin=lower),
                           width=.2, position=position_dodge(),
                           data=means.sem)
f4Ac= f4Ab + theme_bw() + theme(panel.border = element_blank(),
                                text = element_text(size = 20),
                                panel.grid.major = element_blank(), 
                                panel.grid.minor = element_blank(), 
                                axis.line.x = element_line(colour = "black"),
                                axis.line.y = element_line(colour = "black"),
                                legend.position = "none")
fig4A = f4Ac + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 20)) +
  ggtitle ("(a) Overall") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fig4A 

#by host# (b)
dat2$Diversity <- as.factor(dat2$Diversity)
DH <- paste(dat2$Host, dat2$Diversity)
M1 = ddply(dat2, c("DH"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(dat2, c("DH"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
f4Ba = ggplot(M1, aes(x = DH, y = mean, fill =DH)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  scale_fill_manual(values=c("#999999", "#CCCCCC", "#999999", "#CCCCCC", "#999999", "#CCCCCC", "#999999", "#CCCCCC"))
f4Bb = f4Ba+ geom_errorbar(aes(ymax=upper,
                               ymin=lower),
                           width=.2, position=position_dodge(),
                           data=means.sem)
f4Bc= f4Bb + theme_bw() + theme(panel.border = element_blank(),
                                text = element_text(size = 20),
                                panel.grid.major = element_blank(), 
                                panel.grid.minor = element_blank(), 
                                axis.line.x = element_line(colour = "black"),
                                axis.line.y = element_line(colour = "black"),
                                legend.position = "none")
fig4B = f4Bc + ylab (label = "")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 20)) +
  ggtitle ("(b)") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fig4B 

#### Supplemental figures ####
#### Fig.S1 (=3A) ####
datP1 = dat1[dat1$Past.names=="P01",] 
datP2 = dat1[dat1$Past.names=="P02",] 
datP3 = dat1[dat1$Past.names=="P03",] 
datP4 = dat1[dat1$Past.names=="P04",] 

# Pasteuria source 1 - P01 #
p1 = ddply(datP1, c("Host"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(datP1, c("Host"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)

fp1 = ggplot(p1, aes(x = Host, y = mean)) +
  geom_bar(stat="identity", color="black", fill = "white") +
  coord_cartesian(ylim=c(0.0, 105))
fp1a = fp1+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
fp1b = fp1a + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 22))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 22))+
  ggtitle ("(a) P01") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 22))
fp1c = fp1b + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fP01 = fp1c + theme_bw() + theme(panel.border = element_blank(),
                                 text = element_text(size = 24),
                                 panel.grid.major = element_blank(), 
                                 panel.grid.minor = element_blank(), 
                                 axis.line.x = element_line(colour = "black"),
                                 axis.line.y = element_line(colour = "black"),
                                 legend.position = "none")
fP01

# Pasteuria source 2 - P02 #
p2 = ddply(datP2, c("Host"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(datP2, c("Host"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)

fp2 = ggplot(p2, aes(x = Host, y = mean)) +
  geom_bar(stat="identity", color="black", fill = "white") +
  coord_cartesian(ylim=c(0.0, 105))
fp2a = fp2+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
fp2b = fp2a + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20))+
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(b) P02") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fp2c = fp2b + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fP02 = fp2c + theme_bw() + theme(panel.border = element_blank(),
                                 text = element_text(size = 24),
                                 panel.grid.major = element_blank(), 
                                 panel.grid.minor = element_blank(), 
                                 axis.line.x = element_line(colour = "black"),
                                 axis.line.y = element_line(colour = "black"),
                                 legend.position = "none")
fP02

# Pasteuria source 3 - P03 #
p3 = ddply(datP3, c("Host"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(datP3, c("Host"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)

fp3 = ggplot(p3, aes(x = Host, y = mean)) +
  geom_bar(stat="identity", color="black", fill = "white") +
  coord_cartesian(ylim=c(0.0, 105))
fp3a = fp3+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
fp3b = fp3a + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(c) P03") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fp3c = fp3b + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fP03 = fp3c + theme_bw() + theme(panel.border = element_blank(),
                                 text = element_text(size = 24),
                                 panel.grid.major = element_blank(), 
                                 panel.grid.minor = element_blank(), 
                                 axis.line.x = element_line(colour = "black"),
                                 axis.line.y = element_line(colour = "black"),
                                 legend.position = "none")
fP03

# Pasteuria source 4 - P04 #
p4 = ddply(datP4, c("Host"), summarise,
           mean=mean(prop.with.spores_perct))
means.sem <- ddply(datP4, c("Host"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)

fp4 = ggplot(p4, aes(x = Host, y = mean)) +
  geom_bar(stat="identity", color="black", fill = "white") +
  coord_cartesian(ylim=c(0.0, 105))
fp4a = fp4+ geom_errorbar(aes(ymax=upper,
                              ymin=lower),
                          width=.2, position=position_dodge(),
                          data=means.sem)
fp4b = fp4a + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 20))+ 
  xlab (label = "Host lines")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(d) P04") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
fp4c = fp4b + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
fP04 = fp4c + theme_bw() + theme(panel.border = element_blank(),
                                 text = element_text(size = 24),
                                 panel.grid.major = element_blank(), 
                                 panel.grid.minor = element_blank(), 
                                 axis.line.x = element_line(colour = "black"),
                                 axis.line.y = element_line(colour = "black"),
                                 legend.position = "none")
fP04

pushViewport(viewport(layout = grid.layout(4, 1)))
print(fP01, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(fP02, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))
print(fP03, vp = viewport(layout.pos.row = 3, layout.pos.col = 1))
print(fP04, vp = viewport(layout.pos.row = 4, layout.pos.col = 1))

#### Fig.S2 (=3B) ####
Ndat1 = dat1_long2[dat1_long2$Past.names=="P01",] 
Ndat2 = dat1_long2[dat1_long2$Past.names=="P02",] 
Ndat3 = dat1_long2[dat1_long2$Past.names=="P03",] 
Ndat4 = dat1_long2[dat1_long2$Past.names=="P04",] 

#### Load - Pasteuria source 1 - P01 #
fLP1= ggplot(Ndat1, aes(x = Host, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP1 = fLP1 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) P01") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP1 = fiLP1 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP1


#### Load - Pasteuria source 2 - P02 #
fLP2 = ggplot(Ndat2, aes(x = Host, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP2 = fLP2 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) P02") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP2 = fiLP2 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP2

#### Load - Pasteuria source 3 - P03 #
fLP3 = ggplot(Ndat3, aes(x = Host, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP3 = fLP3 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) P03") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP3 = fiLP3 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP3

#### Load - Pasteuria source 4 - P04 #
fLP4 = ggplot(Ndat4, aes(x = Host, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,55), breaks=seq(0,55,10), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP4 = fLP4 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "Host lines")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) P04") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP4 = fiLP4 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP4

pushViewport(viewport(layout = grid.layout(4, 1)))
print(figLP1, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(figLP2, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))
print(figLP3, vp = viewport(layout.pos.row = 3, layout.pos.col = 1))
print(figLP4, vp = viewport(layout.pos.row = 4, layout.pos.col = 1))

#### Fig.S3 ####
DDatH4 = dat2[dat2$Host=="H04",] 
DDatH8 = dat2[dat2$Host=="H08",] 
DDatH10 = dat2[dat2$Host=="H10",] 
DDatH13 = dat2[dat2$Host=="H13",] 

##H04##
DDH04 = ddply(DDatH4, c("Past.names"), summarise,
              mean=mean(prop.with.spores_perct))
means.sem <- ddply(DDatH4, c("Past.names"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
figH4Aa = ggplot(DDH04, aes(x = Past.names, y = mean, fill = Past.names)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  scale_fill_manual(values=c("#999999", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC","#CCCCCC", "#CCCCCC", "#CCCCCC"))

figH4Ab = figH4Aa+ geom_errorbar(aes(ymax=upper,
                                     ymin=lower),
                                 width=.2, position=position_dodge(),
                                 data=means.sem)
figH4Ac= figH4Ab + theme_bw() + theme(panel.border = element_blank(),
                                      text = element_text(size = 23),
                                      panel.grid.major = element_blank(), 
                                      panel.grid.minor = element_blank(), 
                                      axis.line.x = element_line(colour = "black"),
                                      axis.line.y = element_line(colour = "black"),
                                      legend.position = "none")
figH4 = figH4Ac + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  ggtitle ("(a) H04") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 23))
figH4 

##H08##
DDH08 = ddply(DDatH8, c("Past.names"), summarise,
              mean=mean(prop.with.spores_perct))
means.sem <- ddply(DDatH8, c("Past.names"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
figH8Aa = ggplot(DDH08, aes(x = Past.names, y = mean, fill=Past.names)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  scale_fill_manual(values=c("#999999", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC","#CCCCCC", "#CCCCCC", "#CCCCCC"))

figH8Ab = figH8Aa+ geom_errorbar(aes(ymax=upper,
                                     ymin=lower),
                                 width=.2, position=position_dodge(),
                                 data=means.sem)
figH8Ac= figH8Ab + theme_bw() + theme(panel.border = element_blank(),
                                      text = element_text(size = 23),
                                      panel.grid.major = element_blank(), 
                                      panel.grid.minor = element_blank(), 
                                      axis.line.x = element_line(colour = "black"),
                                      axis.line.y = element_line(colour = "black"),
                                      legend.position = "none")
figH8 = figH8Ac + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  ggtitle ("(b) H08") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 23))
figH8 

##H10##
DDH10 = ddply(DDatH10, c("Past.names"), summarise,
              mean=mean(prop.with.spores_perct))
means.sem <- ddply(DDatH10, c("Past.names"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
figH10Aa = ggplot(DDH10, aes(x = Past.names, y = mean, fill=Past.names)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  scale_fill_manual(values=c("#999999", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC","#CCCCCC", "#CCCCCC", "#CCCCCC"))

figH10Ab = figH10Aa+ geom_errorbar(aes(ymax=upper,
                                       ymin=lower),
                                   width=.2, position=position_dodge(),
                                   data=means.sem)
figH10Ac= figH10Ab + theme_bw() + theme(panel.border = element_blank(),
                                        text = element_text(size = 23),
                                        panel.grid.major = element_blank(), 
                                        panel.grid.minor = element_blank(), 
                                        axis.line.x = element_line(colour = "black"),
                                        axis.line.y = element_line(colour = "black"),
                                        legend.position = "none")
figH10 = figH10Ac + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  ggtitle ("(c) H10") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 23))
figH10

##H13##
DDH13 = ddply(DDatH13, c("Past.names"), summarise,
              mean=mean(prop.with.spores_perct))
means.sem <- ddply(DDatH13, c("Past.names"), summarise,
                   mean=mean(prop.with.spores_perct), sem=sd(prop.with.spores_perct)/sqrt(length(prop.with.spores_perct)))
means.sem <- transform(means.sem, lower=mean-sem, upper=mean+sem)
figH13Aa = ggplot(DDH13, aes(x = Past.names, y = mean, fill=Past.names)) +
  geom_bar(stat="identity", show.legend = FALSE) +
  coord_cartesian(ylim=c(0.0, 100))+
  scale_fill_manual(values=c("#999999", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC", "#CCCCCC","#CCCCCC", "#CCCCCC", "#CCCCCC"))

figH13Ab = figH13Aa+ geom_errorbar(aes(ymax=upper,
                                       ymin=lower),
                                   width=.2, position=position_dodge(),
                                   data=means.sem)
figH13Ac= figH13Ab + theme_bw() + theme(panel.border = element_blank(),
                                        text = element_text(size = 23),
                                        panel.grid.major = element_blank(), 
                                        panel.grid.minor = element_blank(), 
                                        axis.line.x = element_line(colour = "black"),
                                        axis.line.y = element_line(colour = "black"),
                                        legend.position = "none")
figH13 = figH13Ac + ylab (label = "Attachment rate")  + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  xlab (label = "")  + theme (axis.title.x = element_text(vjust = 1.5, angle=90, color = "black", size = 23)) +
  ggtitle ("(d) H13") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 23))
figH13

pushViewport(viewport(layout = grid.layout(4, 1)))
print(figH4, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(figH8, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))
print(figH10, vp = viewport(layout.pos.row = 3, layout.pos.col = 1))
print(figH13, vp = viewport(layout.pos.row = 4, layout.pos.col = 1))

#### Fig.S4 ####
L2DatH4 = dat2_long2[dat2_long2$Host=="H04",] 
L2DatH8 = dat2_long2[dat2_long2$Host=="H08",] 
L2DatH10 = dat2_long2[dat2_long2$Host=="H10",] 
L2DatH13 = dat2_long2[dat2_long2$Host=="H13",] 

#### Load - Host H04 #
LP1= ggplot(L2DatH4, aes(x = Past.names, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,15), breaks=seq(0,15,5), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP1 = LP1 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) H04") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP1 = fiLP1 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP1

#### Load - Pasteuria source 2 - P02 #
LP2= ggplot(L2DatH8, aes(x = Past.names, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,15), breaks=seq(0,15,5), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP2 = LP2 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) H08") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP2 = fiLP2 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP2

#### Load - Pasteuria source 3 - P03 #
LP3= ggplot(L2DatH10, aes(x = Past.names, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,15), breaks=seq(0,15,5), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP3 = LP3 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) H10") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP3 = fiLP3 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP3


#### Load - Pasteuria source 4 - P04 #
LP4= ggplot(L2DatH13, aes(x = Past.names, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=0.7, fatten = 1.5, alpha = 0.90, width = 0.8, fill = "white", outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,15), breaks=seq(0,15,5), expand = c(0, 0)) + 
  geom_point(size = 1, shape = 21, position=position_jitter(0.2)) +
  stat_summary(fun.y=mean, geom="point", shape=23, size=3, color = 'skyblue', fill = 'skyblue') +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")

fiLP4 = LP4 + ylab (label =  "Attachment load") + theme (axis.title.y = element_text(vjust = 1.5, angle=90, color = "black", size = 23))+ 
  xlab (label = "Parasite sources")+ theme (axis.title.x = element_text(vjust = -0.5, angle=0, color = "black", size = 20))+
  ggtitle ("(a) H13") + theme (plot.title= element_text(vjust = 2, hjust =0, angle=0, color = "black", size = 20))
figLP4 = fiLP4 + theme(axis.text.x=element_text(angle=0, size=20, vjust=0.5))+
  theme(axis.text.y=element_text(angle=0, size=20, vjust=0.5))
figLP4

pushViewport(viewport(layout = grid.layout(4, 1)))
print(figLP1, vp = viewport(layout.pos.row = 1, layout.pos.col = 1))
print(figLP2, vp = viewport(layout.pos.row = 2, layout.pos.col = 1))
print(figLP3, vp = viewport(layout.pos.row = 3, layout.pos.col = 1))
print(figLP4, vp = viewport(layout.pos.row = 4, layout.pos.col = 1))

### extra
dat2_long2$Diversity=factor(dat2_long2$Diversity,levels=c("LOW","HIGH"))
LP= ggplot(dat2_long2, aes(x = Diversity, y = spores)) +
  stat_boxplot(geom ='errorbar', width = 0.6) +
  geom_boxplot(lwd=1, fatten = 1.5, alpha = 0.90, width = 0.8, fill = c("#83d3d4","#2d8183"), outlier.shape = NA) +  #outlier.colour = "red") + # "lwd" for "line width"; fatten for the  median line 
  scale_y_continuous(limits=c(0,15), breaks=seq(0,15,5), expand = c(0, 0)) + 
  geom_point(size = 2, shape = 21, position=position_jitter(0.3),color="gray30") +
  theme_bw() + theme(panel.border = element_blank(), 
                     text = element_text(size = 20), 
                     panel.grid.major = element_blank(),
                     panel.grid.minor = element_blank(), 
                     axis.line = element_line(colour = "black"),
                     legend.position = "none")
LP
