%==============================================================%
%----------Skytrain TAC Data Workup---(TK Bauska)---------------%
%==============================================================%
clear all
%==============================================================%
%----------------------Switches--------------------------------%
%==============================================================%

error_input = 'Gaussian';
switch error_input
    case 'Gaussian'
         disp ('Errors in model parameters are input with Gaussian distributions')
    case 'Uniform'
          disp ('Errors in model parameters are input with Uniform distributions. Note that what errors that are labelled 1-sigma s.d. are taken to be +/- the uniform range')
    otherwise
        warning 'Please set the error input type or check your spelling :)'
end

sensitivity = 'Full';
switch sensitivity 
    case 'Full'
         disp ('Standard scenario where TAC, temperature and Vc all change')
    case 'TAC Only'
         disp ('Only the effect from changing TAC is used. temperature (and thus Vc) set to constant')
    case 'Temperature Only'
        disp ('Only the effect from temperature (ideal gas law).  TAC is set to constant and Vc depedence turned of')
    case 'Vc Only'
         disp ('Only the effect from Vc.  Temperature is varied and Vc evolves but temperature is not employed in the ideal gas correction' )
    case 'Temperature and Vc'
         disp ('TAC is set to constant.  Temperature and Vc are changing' ) 
    otherwise
        warning 'Please set the sensivity case. Check your spelling :)'
end
Vc_propogation = 'with temp';
switch Vc_propogation
    case 'with temp'
         disp ('Vc is calculated at each TAC sample based on the temperature at bubble closure')
    case 'with delta age'
        disp ('Vc is calculated at each TAC sample based on the tempeature of the ice that enclosuses the air at the time it formed (using delta age)q')
    otherwise
        warning 'Please set the the Vc propogation or check your spelling :)'
end

iso_elevation = 'Elevation Direct';
switch iso_elevation
    case 'Temperature and Lapse Rate'
         disp ('The elevation history based on isotopes is constructed using a combination of prescribed temperture sensitivity and lapse-rate')
    case 'Elevation Direct'
        disp ('The elevation history based on isotopes is constructed using a prescribed elevation sensitivity')
    otherwise
        warning 'Please set the isotope to elevatin method or check your spelling :)'
end

summary_stats_method  = 'Means and standard deviations';
switch summary_stats_method 
    case 'Means and standard deviations'
         disp ('When analysing the elevation changes before and after the jump report the means and standard deviations of the MC distributions')
    case 'Median and percentiles'
          disp ('When analysing the elevation changes before and after the jump take the medians and your chosen percentiles')
end


figures  = 'Summary';
switch figures
    case 'In Loop'
         disp ('Figures will be plotted as the code works through the loop.  Try a small number (n = 100)')
    case 'Summary'
        disp ('Just your summary figures will be plotted')
end
%==============================================================%
%---------------------Parameters-------------------------------%
%==============================================================%
iterations = 500;          %number of simulations in Monte Carlo
smooth_resolution = 100;    %number of data points in isotope/surface tempearture record to smooth
GMWL = 8;                   %DEFUNCT: global (or local) meteoric water line to convert dD to d18O 
TACerr = 0.4/100;           %error on TAC data (in percent)
TAC_bubble_correction = 4/100;    %increase in measured to TAC or bubble correctoin (in percent)
modern_temp = -26;          %modern temp at site to anchor temperature reconstruction: NOTE that if you are using temperature history at bubble close-off rather than raw isotope data this should be the same as used in that modelled history
Tc_TK = 273.15;             %celcius to kelvin
modern_isotope = -34.5;     %DEFUNCT: modern isotope at site to anchor temperature reconstrution
%modern_vc  = 0.125;         %modern Vc at site to anchor Vc: NOTE this sets the absolute value of your TAC-based elevation history!
%%modern_vc  = 0.127;         %modern Vc at site to anchor Vc: NOTE this sets the absolute value of your TAC-based elevation history!
modern_vc  = 0.132;         %modern Vc at site to anchor Vc: NOTE this sets the absolute value of your TAC-based elevation history!

Ps = 1000; 	                %standard pressure (mBar)
Ps = 1013.25; 	                %standard pressure (mBar)
Ts = 273.15;                %standard temperature (K)
%Ts = 273;                %standard temperature (K)
 
isotope_slope_mean = 0.8;   %isotope slope mean (per mil/deg C)
isotope_slope_unc = 0.2;    %isotope slope uncertainty(1-sigma s.d.)

Vc_slope_mean = 4.5e-4;     %Vc slope mean (cm3/g/deg C)
Vc_slope_unc  = 3e-4;       %Vc slope uncertainty(1-sigma s.d.)

lapse_rate_mean = 10;       %lapse rate mean(deg C/km)
lapse_rate_unc = 2;         %lapse rate uncertainty (1-sigma s.d.)(deg C/km)

isotope_elev_slope_mean = -0.8;   %isotope slope mean (per mil/per 100 meters)
isotope_elev_slope_unc = 0.2;    %isotope slope uncertainty(1-sigma s.d.)


modern_elevation = 784;     %modern elevation for tempererature/isotope method (m)

constant_elevation = 1000;  %constant elevation for TAC prediction in sensitivity experiments
%==============================================================%
%-----------------Load and Plot Data---------------------------%
%==============================================================%
%TAC_DATA = load('Generic_TAC_data.csv');
TAC_DATA = load('TAC_ST_HoloJump_Bauska.csv');
%ISO_DATA = load('Generic_d18O_data.csv');
TEMP_DATA = load('CFM_Output_ST_25k.csv');
BCO_TEMP_DATA = TEMP_DATA(:,3)-Tc_TK;    %selects temperture data for bubble close-off history
SURF_TEMP_DATA = TEMP_DATA(:,2)-Tc_TK;   %selects temperture data for surface history

switch sensitivity
    case 'Full'
         TAC_DATA(:,2) = TAC_DATA(:,2);
    case 'TAC Only'
         TAC_DATA(:,2) = TAC_DATA(:,2);
         BCO_TEMP_DATA  = modern_temp+zeros(size(BCO_TEMP_DATA,1),1);
         SURF_TEMP_DATA = modern_temp+zeros(size(SURF_TEMP_DATA,1),1);
    case 'Temperature Only'
         TAC_DATA(:,2) = mean(TAC_DATA(:,2))+zeros(size(TAC_DATA(:,2),1),1);
         Vc_slope_mean = 0;     %Vc slope mean (cm3/g/deg C)
         Vc_slope_unc  = 0;     %Vc slope uncertainty(1-sigma s.d.)
         TACerr  = 0;
    case 'Vc Only'
          TAC_DATA(:,2) = mean(TAC_DATA(:,2))+zeros(size(TAC_DATA(:,2),1),1);
          TACerr  = 0;
    case 'Temperature and Vc'
          TAC_DATA(:,2) = mean(TAC_DATA(:,2))+zeros(size(TAC_DATA(:,2),1),1);
          TACerr  = 0;
    otherwise
        TAC_DATA(:,2) = TAC_DATA(:,2);
end

TAC_DATA(:,2) = TAC_DATA(:,2)*(1+TAC_bubble_correction);
%Smooth and interpolate temperature histories onto TAC resolution (think about smoothing)
BCO_TEMP_SMOOTH_2TAC = interp1(TEMP_DATA(:,1),smooth(BCO_TEMP_DATA(:,1),1),TAC_DATA(:,1));
SURF_TEMP_SMOOTH = smooth(SURF_TEMP_DATA(:,1),smooth_resolution);
%BCO_TEMP_SMOOTH_2TAC  = interp1(TEMP_DATA(:,1),smooth(SURF_TEMP_DATA(:,1),smooth_resolution),TAC_DATA(:,1));
SURF_TEMP_DA_PROPOGATE_2TAC = interp1(TEMP_DATA(:,1),smooth(SURF_TEMP_DATA(:,1),smooth_resolution),TAC_DATA(:,5));

%Plot raw and smoothed data
figure(1)
subplot(2,1,1)
hold on
plot(TAC_DATA(:,1),TAC_DATA(:,2))
xlabel('Time (years)')
ylabel('TAC (cm3g-1)')
xlim([5000 10000])
subplot(2,1,2)
hold on
plot(TEMP_DATA(:,1),BCO_TEMP_DATA(:,1))
plot(TEMP_DATA(:,1),SURF_TEMP_SMOOTH)
plot(TAC_DATA(:,1),BCO_TEMP_SMOOTH_2TAC , 'o')
plot(TAC_DATA(:,1),SURF_TEMP_DA_PROPOGATE_2TAC  , '.')
xlabel('Time (years)')
ylabel('Temperature K')
legend('Close-off T', 'Surface (smoothed)', 'Interpolated to TAC', 'Past Surface Temperature on Gas Age')
xlim([5000 10000])
%==============================================================%
%-----------------Uncertainty Analysis--------------------------%
%==============================================================%
%Uncertainity in the calibration (Vc, isotope temperature, lapse rate)

%Preallocate data space
iso_slope = zeros(1,iterations);                             %all the isotope/temperature slopes
iso_intercept = zeros(1,iterations);                         %DEFUNCT:  all the isotope/temperature intercept
Vc_slope = zeros(1,iterations);                              %all the Vc/temperature slopes
Vc_intercept = zeros(1,iterations);                          %all the Vc/temperature intercept 
lapse_rate = zeros(1,iterations);                            %all the lapse rates 
iso_2_elevation = zeros(1,iterations);                       %all the isotope to elevation slopes
TEMP_2TAC = zeros(size(TAC_DATA,1),iterations);              %the temperature at closure for each TAC data point for each MC run
TEMP_DA_PROPOGATE_2TAC = zeros(size(TAC_DATA,1),iterations); %the temperature of the ice at bubble closure when it formed for each TAC data point for each MC run
Vc = zeros(size(TAC_DATA,1),iterations);                     %the volume at closure for each TAC data point for each MC run 
Pc = zeros(size(TAC_DATA,1),iterations);                     %the pressure at closure for each TAC data point for each MC run
Elevation_TAC = zeros(size(TAC_DATA,1),iterations);          %the elevation for each TAC data point for each MC run
FULL_TEMP = zeros(size(TEMP_DATA,1),iterations);             %all the temperature histories for the full time range 
Elevation_ISO = zeros(size(TEMP_DATA,1),iterations);         %the lapse-rate implied elevation range for the entire temperature history (NOTE this will be meaningless on most timescales)
Temperature_Elevation_TAC = zeros(size(TAC_DATA,1),iterations);  %the lapse-rate implied temperatur range for the elevation history (NOTE this is just a fun triple-check if isotope/temp scenario agrees with the TAC)
TAC_predicted = zeros(size(TAC_DATA,1),iterations);          %Space to fill with TAC predicted values if elevation was held constant

%--->Loop Start
for i = 1:iterations


switch error_input
    case 'Gaussian'
    iso_slope(i) = isotope_slope_mean+randn(1)*isotope_slope_unc;   %randomly perturb isotope slope
    Vc_slope(i) = Vc_slope_mean+randn(1)*Vc_slope_unc;              %randomly perturb Vc slope
    lapse_rate(i) = -lapse_rate_mean+randn(1)*lapse_rate_unc;         %Random Perturbation of Lapse Rate
    iso_2_elevation(i) = isotope_elev_slope_mean+randn(1)*isotope_elev_slope_unc;   %randomly perturb isotope slope

    case 'Uniform'
    iso_slope(i) = (isotope_slope_mean-isotope_slope_unc)+rand(1)*2*isotope_slope_unc;   %randomly perturb isotope slope
    Vc_slope(i) =  (Vc_slope_mean-Vc_slope_unc)+rand(1)*2*Vc_slope_unc;              %randomly perturb Vc slope
    lapse_rate(i) = (-lapse_rate_mean-lapse_rate_unc)+rand(1)*2*lapse_rate_unc;         %Random Perturbation of Lapse Rate
    iso_2_elevation(i) = (isotope_elev_slope_mean-isotope_elev_slope_unc)+rand(1)*2*isotope_elev_slope_unc;   %randomly perturb isotope slope
end

Vc_intercept(i)  = modern_vc-(modern_temp+Tc_TK)*Vc_slope(i);     %calculate intercept for each slope such that the modern temperature and Vc are anchor points

TEMP_2TAC(:,i) = (BCO_TEMP_SMOOTH_2TAC-modern_temp)*(1/iso_slope(i))+modern_temp;%+iso_intercept(i);   %isotopic temperature at TAC resolution
FULL_TEMP(:,i) = (SURF_TEMP_SMOOTH-modern_temp)*(1/iso_slope(i))+modern_temp;%+iso_intercept(i);        %isotopic temperature at full resolution
TEMP_DA_PROPOGATE_2TAC(:,i) =(SURF_TEMP_DA_PROPOGATE_2TAC-modern_temp)*(1/iso_slope(i))+modern_temp;%+iso_intercept(i);   %isotopic temperature at TAC resolution propogate with delta age

switch Vc_propogation
    case 'with temp'
         Vc(:,i) = (TEMP_2TAC(:,i)+Tc_TK)*Vc_slope(i)+Vc_intercept(i);     %Vc (volume at closure)
    case 'with delta age'
         Vc(:,i) = (TEMP_DA_PROPOGATE_2TAC(:,i)+Tc_TK)*Vc_slope(i)+Vc_intercept(i);     %Vc (volume at closure)
end

switch sensitivity
    case 'Vc Only'
    Pc(:,i) = ((Ps/Ts)*((modern_temp+Tc_TK))./Vc(:,i)).*((TAC_DATA(:,2)+TAC_DATA(:,2).*randn(size(TAC_DATA(:,2),1),1)*TACerr));  %Pc (pressure at closure)
    TAC_predicted(:,i) = Vc(:,i).*(989.1*exp(-1*constant_elevation/7588))./(modern_temp+Tc_TK)*(Ts/Ps);
    otherwise
    Pc(:,i) = ((Ps/Ts)*((TEMP_2TAC(:,i)+Tc_TK))./Vc(:,i)).*((TAC_DATA(:,2)+TAC_DATA(:,2).*randn(size(TAC_DATA(:,2),1),1)*TACerr));  %Pc (pressure at closure)
    TAC_predicted(:,i) = Vc(:,i).*(989.1*exp(-1*constant_elevation/7588))./(TEMP_2TAC(:,i)+Tc_TK)*(Ts/Ps);

end
Elevation_TAC(:,i) = -7588*log(Pc(:,i)/989.1);                    %Elevation (m) appproximation from Fegyveresi Thesis based on Stone (2000) Air pressure and cosmogenic isotope production. Journal of Geophysical Research-Solid Earth, 105(B10): 23753-23759

switch iso_elevation
    case 'Temperature and Lapse Rate'
        Elevation_ISO(:,i) = (FULL_TEMP(:,i)-modern_temp)*(1000/(lapse_rate(i)))+modern_elevation;  %Calculating Elevation Implied from Isotope/Surface Temperature History
    case 'Elevation Direct'
      Elevation_ISO(:,i) = (SURF_TEMP_SMOOTH-modern_temp)/(iso_2_elevation(i)/100)+modern_elevation;  %Calculating Elevation Implied from Isotope/Surface Temperature History
end

Temperature_Elevation_TAC(:,i) = (Elevation_TAC(:,i)-modern_elevation)/(1000/lapse_rate(i))+modern_temp; %Calculating Temperature Implied from TAC Elevation History

switch figures
    case 'In Loop'
figure(2)
subplot(4,1,1)
hold on
plot(TAC_DATA(:,1),TEMP_2TAC(:,i) , '.')  %temperature interpoled for each TAC data point versus age
plot(TEMP_DATA(:,1),FULL_TEMP(:,i))       %all temperaure data versus age
xlabel('Time (years)')
ylabel('Tempeature')

subplot(4,1,2)
hold on
plot(TAC_DATA(:,1),Vc(:,i) , '.')         %Volume at closure (Vc) calculated for each TAC data point
xlabel('Time (years)')
ylabel('Vc (cm3/g)')

subplot(4,1,3)
hold on
plot(TAC_DATA(:,1),Pc(:,i)  , '.')       %Pressure at close (Pc) calculated for each TAC data point
xlabel('Time (years)')
ylabel('Pc (mbar)')

subplot(4,1,4)
hold on
plot(TAC_DATA(:,1),Elevation_TAC(:,i)   , '.')  %Elevation caclulated for each TAC data point
plot(TEMP_DATA(:,1),Elevation_ISO(:,i))         %Elevation caclulated for each temperature data point
xlabel('Time (years)')
ylabel('Elevation (m)')
%}
        
end
end
%--->Loop End

%==============================================================%
%-----------------Plot and Analyse-----------------------------%
%==============================================================%
quantile_min = 0.16;       %Change these for different ranges 0.16 and 0.84 is 1-sigma s.d.  2-sigma might be better
quantile_max = 0.84;

afterjump_ISO = [6000 7500];   %Age range after jump for doing stats.  Note this will ultimately be your anchor for calculating the change in elevation, so ideally encompasses enough data such that it's insensitive to outliers.
beforejump_ISO = [8500 10000];  %Age range before jump for doing stats. 


afterjump_TAC = [4900 6600];   %Age range after jump for doing stats.  Note this will ultimately be your anchor for calculating the change in elevation, so ideally encompasses enough data such that it's insensitive to outliers.
%afterjump_TAC = [7850 7960];   %Age range after jump for doing stats.  Note this will ultimately be your anchor for calculating the change in elevation, so ideally encompasses enough data such that it's insensitive to outliers.

beforejump_TAC = [8500 9500];  %Age range before jump for doing stats. 

switch summary_stats_method 
    case 'Means and standard deviations'
         Elevation_TAC_export = [TAC_DATA(:,1) mean(Elevation_TAC,2) mean(Elevation_TAC,2)-std(Elevation_TAC,0,2) mean(Elevation_TAC,2)+std(Elevation_TAC,0,2)...
                                      mean(TEMP_2TAC,2) mean(TEMP_2TAC,2)-std(TEMP_2TAC,0,2) mean(TEMP_2TAC,2)+std(TEMP_2TAC,0,2)...
                                      mean(TAC_predicted,2) mean(TAC_predicted,2)-std(TAC_predicted,0,2) mean(TAC_predicted,2)+std(TAC_predicted,0,2)];

          Elevation_ISO_export = [TEMP_DATA(:,1) mean(Elevation_ISO,2) mean(Elevation_ISO,2)-std(Elevation_ISO,0,2) mean(Elevation_ISO,2)+std(Elevation_ISO,0,2)...
                                       mean(FULL_TEMP,2)  mean(FULL_TEMP,2)-std(FULL_TEMP,0,2) mean(FULL_TEMP,2)+std(FULL_TEMP,0,2)];
    case 'Median and percentiles'
         Elevation_TAC_export = [TAC_DATA(:,1) quantile(Elevation_TAC,.5,2) quantile(Elevation_TAC,quantile_max,2) quantile(Elevation_TAC,quantile_min,2)...
                                      quantile(TEMP_2TAC,0.5,2) quantile(TEMP_2TAC,quantile_max,2) quantile(TEMP_2TAC,quantile_min,2)...
                                      quantile(TAC_predicted,0.5,2) quantile(TAC_predicted,quantile_max,2) quantile(TAC_predicted,quantile_min,2)];

         Elevation_ISO_export = [TEMP_DATA(:,1) quantile(Elevation_ISO,.5,2) quantile(Elevation_ISO,quantile_max,2) quantile(Elevation_ISO,quantile_min,2)...
                                      quantile(FULL_TEMP,0.5,2) quantile(FULL_TEMP,quantile_max,2) quantile(FULL_TEMP,quantile_min,2)];
end

Parameter_export = [iterations;TACerr;modern_vc;isotope_slope_mean;isotope_slope_unc;Vc_slope_mean;Vc_slope_unc;lapse_rate_mean;lapse_rate_unc;isotope_elev_slope_mean; isotope_elev_slope_unc;modern_elevation;afterjump_TAC';beforejump_TAC';afterjump_ISO';beforejump_ISO';quantile_min;quantile_max];

figure(3)
hold on
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,4), 'b')
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,3), 'b')
plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,4), 'r')
plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,3), 'r')
xlim([0000 10000])
xlabel('Time (years)')
ylabel('Elevation (m)')
legend('TAC max','TAC min','ISO max','ISO min')

figure(4)
subplot(2,2,1)
plot(TEMP_2TAC(:,:),BCO_TEMP_SMOOTH_2TAC,'.')
ylabel('Temperature Input (C)')
xlabel('Bubble closure temperature including range (C)')
subplot(2,2,2)
plot(TEMP_2TAC(:,:),Vc(:,:),'.')
xlabel('Bubble closure temperature including range (C)')
ylabel('Vc (cm3/g)')
subplot(2,2,3)
plot(TAC_DATA(:,2),Elevation_TAC(:,:),'.')
xlabel('TAC (cm3/g)')
ylabel('Elevation (m)')
subplot(2,2,4)
plot(FULL_TEMP(:,:),Elevation_ISO(:,:),'.')
xlabel('Surface Temperature  (C)')
ylabel('Lapse-rate Elevation (m)')

figure(5)
hold on
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,6), 'k')
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,7), 'k')
plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,6), 'r')
plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,7), 'r')
xlim([5000 10000])
xlabel('Time (years)')
ylabel('Temperature (C)')
legend('Bubble close-off max','Bubble close-ff min','Surface Temperatur max','Surface Temperature min')

figure(6)
hold on
plot(TAC_DATA(:,1),TAC_DATA(:,2), 'r')
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,9), 'k')
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,10), 'k')
xlabel('Time (years)')
ylabel('TAC (cm3g-1)')
legend('TAC data input','Predicted TAC at constant elevation max','Predicted TAC at constant elevation min')
xlim([5000 10000])


%==============================================================%
%--Find Temperature/Isotope-based scenarios that are consistent with TAC--------%
%==============================================================%

%Elevation_TAC_afterjump_range = [mean(Elevation_TAC_export(find(Elevation_TAC_export(:,1)>=afterjump(1) & Elevation_TAC_export(:,1)<=afterjump(2)),3))...
%                   mean(Elevation_TAC_export(find(Elevation_TAC_export(:,1)>=afterjump(1) & Elevation_TAC_export(:,1)<=afterjump(2)),4))];
%Elevation_TAC_beforejump_range = [mean(Elevation_TAC_export(find(Elevation_TAC_export(:,1)>=beforejump(1) & Elevation_TAC_export(:,1)<=beforejump(2)),3))...
%                   mean(Elevation_TAC_export(find(Elevation_TAC_export(:,1)>=beforejump(1) & Elevation_TAC_export(:,1)<=beforejump(2)),4))];

TAC_afterjump_index  = find(Elevation_TAC_export(:,1)>=afterjump_TAC(1) & Elevation_TAC_export(:,1)<=afterjump_TAC(2));
TAC_beforejump_index = find(Elevation_TAC_export(:,1)>=beforejump_TAC(1) & Elevation_TAC_export(:,1)<=beforejump_TAC(2));


ISO_afterjump_index  = find(Elevation_ISO_export(:,1)>=afterjump_ISO(1) & Elevation_ISO_export(:,1)<=afterjump_ISO(2));
ISO_beforejump_index = find(Elevation_ISO_export(:,1)>=beforejump_ISO(1) & Elevation_ISO_export(:,1)<=beforejump_ISO(2));


switch summary_stats_method 

    case 'Means and standard deviations'
         Elevation_TAC_afterjump_range=  [mean(Elevation_TAC(TAC_afterjump_index,:), 'all'); ...
                                     mean(Elevation_TAC(TAC_afterjump_index,:), 'all')-std(Elevation_TAC(TAC_afterjump_index,:),0, 'all');...
                                     mean(Elevation_TAC(TAC_afterjump_index,:), 'all')+std(Elevation_TAC(TAC_afterjump_index,:),0, 'all')];

         Elevation_TAC_beforejump_range = [mean(Elevation_TAC(TAC_beforejump_index,:), 'all'); ...
                                     mean(Elevation_TAC(TAC_beforejump_index,:), 'all')-std(Elevation_TAC(TAC_beforejump_index,:),0, 'all');...
                                     mean(Elevation_TAC(TAC_beforejump_index,:), 'all')+std(Elevation_TAC(TAC_beforejump_index,:),0, 'all')];

         Elevation_ISO_afterjump_range=  [mean(Elevation_ISO(ISO_afterjump_index,:), 'all'); ...
                                     mean(Elevation_ISO(ISO_afterjump_index,:), 'all')-std(Elevation_ISO(ISO_afterjump_index,:),0, 'all');...
                                     mean(Elevation_ISO(ISO_afterjump_index,:), 'all')+std(Elevation_ISO(ISO_afterjump_index,:),0, 'all')];

         Elevation_ISO_beforejump_range = [mean(Elevation_ISO(ISO_beforejump_index,:), 'all'); ...
                                     mean(Elevation_ISO(ISO_beforejump_index,:), 'all')-std(Elevation_ISO(ISO_beforejump_index,:),0, 'all');...
                                     mean(Elevation_ISO(ISO_beforejump_index,:), 'all')+std(Elevation_ISO(ISO_beforejump_index,:),0, 'all')];
        
         diff_allscenarios_TAC = mean(Elevation_TAC(TAC_beforejump_index,:))...
                                 -mean(Elevation_TAC(TAC_afterjump_index,:));

         diff_allscenarios_ISO = mean(Elevation_ISO(ISO_beforejump_index,:))...
                                 -mean(Elevation_ISO(ISO_afterjump_index,:));
         data_TAC_afterjump  = [mean(TAC_DATA(TAC_afterjump_index,2)) ...
                                  std(TAC_DATA(TAC_afterjump_index,2))];
         data_TAC_beforejump  = [mean(TAC_DATA(TAC_beforejump_index,2)) ...
                                  std(TAC_DATA(TAC_beforejump_index,2))];
         diff_data_TAC =         mean(TAC_DATA(TAC_beforejump_index,2))...
                                 -mean(TAC_DATA(TAC_afterjump_index,2));

         data_ISO_afterjump  = [mean(TEMP_DATA(ISO_afterjump_index,2)) ...
                                  std(TEMP_DATA(ISO_afterjump_index,2))];
         data_ISO_beforejump  = [mean(TEMP_DATA(ISO_beforejump_index,2)) ...
                                  std(TEMP_DATA(ISO_beforejump_index,2))];

         diff_data_ISO =         mean(TEMP_DATA(ISO_beforejump_index,2))...
                                 -mean(TEMP_DATA(ISO_afterjump_index,2));

         diff_summary_TAC = [mean(diff_allscenarios_TAC);...
                             mean(diff_allscenarios_TAC)+std(diff_allscenarios_TAC);...
                             mean(diff_allscenarios_TAC)-std(diff_allscenarios_TAC)];

         diff_summary_ISO = [mean(diff_allscenarios_ISO);...
                             mean(diff_allscenarios_ISO)+std(diff_allscenarios_ISO);...
                             mean(diff_allscenarios_ISO)-std(diff_allscenarios_ISO)];

    case 'Median and percentiles'
          Elevation_TAC_afterjump_range  = quantile(Elevation_TAC(TAC_afterjump_index,:),[0.5 quantile_min quantile_max], 'all');
          Elevation_TAC_beforejump_range = quantile(Elevation_TAC(TAC_beforejump_index,:),[0.5 quantile_min quantile_max], 'all');


          Elevation_ISO_afterjump_range  = quantile(Elevation_ISO(ISO_afterjump_index,:),[0.5 quantile_min quantile_max], 'all');
          Elevation_ISO_beforejump_range = quantile(Elevation_ISO(ISO_beforejump_index,:),[0.5 quantile_min quantile_max], 'all');

          diff_allscenarios_TAC = median(Elevation_TAC(TAC_beforejump_index,:))...
                        -median(Elevation_TAC(TAC_afterjump_index,:));

          diff_allscenarios_ISO = median(Elevation_ISO(ISO_beforejump_index,:))...
                        -median(Elevation_ISO(ISO_afterjump_index,:));


           diff_data_TAC =         median(TAC_DATA(TAC_beforejump_index,3))...
                                 -median(TAC_DATA(TAC_afterjump_index,3));
           diff_data_ISO =         median(TEMP_DATA(ISO_beforejump_index,2))...
                                 -median(TEMP_DATA(ISO_afterjump_index,2));

          diff_summary_TAC = quantile(diff_allscenarios_TAC, [0.5 quantile_min quantile_max], 'all');
          diff_summary_ISO = quantile(diff_allscenarios_ISO, [0.5 quantile_min quantile_max], 'all');

end

Elevation_Stats_Export = [Elevation_TAC_afterjump_range Elevation_TAC_beforejump_range diff_summary_TAC Elevation_ISO_afterjump_range Elevation_ISO_beforejump_range diff_summary_ISO;]
Elevation_Stats_Export2  = [data_TAC_beforejump  data_TAC_afterjump diff_data_TAC NaN;...
                            mean(Elevation_TAC(TAC_beforejump_index,:), 'all') std(Elevation_TAC(TAC_beforejump_index,:),0, 'all') mean(Elevation_TAC(TAC_afterjump_index,:), 'all') std(Elevation_TAC(TAC_afterjump_index,:),0, 'all')  mean(diff_allscenarios_TAC) std(diff_allscenarios_TAC)]

%[h p ci stats ] = ttest2(TAC_DATA(TAC_beforejump_index,3),TAC_DATA(TAC_afterjump_index,3),0.05,'both','unequal')
%[h p ci stats ] = ttest2(reshape(Elevation_TAC(TAC_beforejump_index,:),1,[]),reshape(Elevation_TAC(TAC_afterjump_index,:),1,[]),0.05,'both','unequal')

Export.Stats = Elevation_Stats_Export;
Export.ElevationTAC = [Elevation_TAC_export  Elevation_TAC_export(:,2:4)-Elevation_TAC_afterjump_range(1)];
Export.ElevationISO = [Elevation_ISO_export  Elevation_ISO_export(:,2:4)-Elevation_ISO_afterjump_range(1)];
Export.Parameters = Parameter_export ;
%Export.RunInfo = [error_inputs;sensitivity]
figure(101)
subplot(1,2,1)
hold on
plot(Export.ElevationTAC(:,1),Export.ElevationTAC(:,12),'r')
plot(Export.ElevationTAC(:,1),Export.ElevationTAC(:,13),'r')
plot(Export.ElevationISO(:,1),Export.ElevationISO(:,9),'b')
plot(Export.ElevationISO(:,1),Export.ElevationISO(:,10),'b')
xlim([5000 10000])
ylim([-500 1500])
xlabel('Time (years)')
ylabel('Change in Elevation (m)')
legend('TAC max','TAC min','ISO max','ISO min')

subplot(1,2,2)
hold on
histogram(diff_allscenarios_TAC,'Orientation', 'horizontal')
histogram(diff_allscenarios_ISO,'Orientation', 'horizontal')
ylim([-500 1500])
xlabel('n (number of scenarios)')
ylabel('Change in Elevation')
legend('TAC','ISO')

TSfilename = strcat(datestr(datetime('now'),30),'_',sensitivity,'_',Vc_propogation,'_Skytrain', '.mat');

save(TSfilename, 'Export')

%{

Elevation_ISO_afterjump_means = mean(Elevation_ISO(find(Elevation_ISO_export(:,1)>=afterjump(1) & Elevation_ISO_export(:,1)<=afterjump(2)),:));
Elevation_ISO_beforejump_means = mean(Elevation_ISO(find(Elevation_ISO_export(:,1)>=beforejump(1) & Elevation_ISO_export(:,1)<=beforejump(2)),:));

Elevaton_ISO_consistent_indices = find(Elevation_ISO_afterjump_means<=Elevation_TAC_afterjump_range(1) ...
    & Elevation_ISO_afterjump_means>=Elevation_TAC_afterjump_range(2)...
    & Elevation_ISO_beforejump_means<=Elevation_TAC_beforejump_range(1)...
    & Elevation_ISO_beforejump_means>=Elevation_TAC_beforejump_range(2));


figure(50)
hold on
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,4), 'b')
plot(Elevation_TAC_export(:,1),Elevation_TAC_export(:,3), 'b')
%plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,4), 'r')
%plot(Elevation_ISO_export(:,1),Elevation_ISO_export(:,3), 'r')
plot(Elevation_ISO_export(:,1),Elevation_ISO(:,Elevaton_ISO_consistent_indices ), 'k')
xlim([0000 10000])
xlabel('Time (years)')
ylabel('Elevation (m)')
legend('TAC max','TAC min','ISO max','ISO min')

figure(51)
subplot(1,2,1)
hold on
histogram(iso_slope)
histogram(iso_slope(Elevaton_ISO_consistent_indices))
subplot(1,2,2)
hold on
histogram(iso_slope.*lapse_rate)
histogram(iso_slope(Elevaton_ISO_consistent_indices).*lapse_rate(Elevaton_ISO_consistent_indices))

mean(iso_slope(Elevaton_ISO_consistent_indices).*lapse_rate(Elevaton_ISO_consistent_indices))
std(iso_slope(Elevaton_ISO_consistent_indices).*lapse_rate(Elevaton_ISO_consistent_indices))
mean(iso_slope(Elevaton_ISO_consistent_indices))
std(iso_slope(Elevaton_ISO_consistent_indices))
mean(lapse_rate(Elevaton_ISO_consistent_indices))
std(lapse_rate(Elevaton_ISO_consistent_indices))

%}
figure(1000)
for n = 1:iterations
subplot(3,1,1)
hold on
plot(n,diff_allscenarios_TAC(n),'.')
ylabel('Difference in Elevation (m)')
xlabel('iterations (n)')
subplot(3,1,2)
hold on
plot(n,mean(diff_allscenarios_TAC(1:n)),'.')
ylabel('cumulative mean (meters)')
xlabel('iterations (n)')
subplot(3,1,3)
hold on
plot(n,std(diff_allscenarios_TAC(1:n)),'.')
xlabel('iterations (n)')
ylabel('cumulative standard deviation (meters, 1-sigma)')
end

