"""
  Given a project or a list of projects, this code produce a csv file with
  the number of for loops, list comprehension. It also tells if the project has test
  and if it has requirements file or a setup.py file

  Code by Ossim Belias 21-08-2023
"""

from helper import *
import argparse


class Stats(ast.NodeVisitor):
    def __init__(self, project, out_file):
        assert project is not None, "Project must not be None"
        assert out_file is not None, "File to write result is not there"
        self.project = project
        self.out_file = out_file
        self.count_for = 0
        self.count_list = 0
        self.has_req = False  # requirement file
        self.has_test = False  # test folder
        self.has_setup = False  # setup.py file

    def check_requirement_file(self):
        txt_files = get_all_files(directory=self.project, extension=".txt")
        for file in txt_files:
            if file.endswith("requirements.txt"):
                self.has_req = True
                break

    def check_setup_file(self):
        python_files = get_all_files(directory=self.project, extension='.py')
        for file in python_files:
            if file.endswith("setup.py"):
                self.has_setup = True
                break

    def check_test_folder(self):
        directories = os.listdir(self.project)
        if "tests" in directories:
            self.has_test = True

    def save_stats(self):
        self.check_requirement_file()
        self.check_setup_file()
        self.check_test_folder()

        python_files = get_all_files(directory=self.project, extension=".py")
        current_file = ""
        stats_errors = open("error_stats.txt", "a")
        for file in python_files:
            try:
                current_file = file
                src = open_file(file_path=file)
                tree = ast.parse(src)
                self.visit(tree)
            except Exception as error:
                message = str(error) + " " + current_file
                print(message, file=stats_errors)

        row = [str(self.project),
               str(self.count_for),
               str(self.count_list),
               str(self.has_req),
               str(self.has_setup),
               str(self.has_test)]

        write_row_in_csv(filename=self.out_file, row=row)
        stats_errors.close()

    def visit_For(self, node: ast.For):
        self.count_for += 1

    def visit_ListComp(self, node: ast.ListComp):
        self.count_list += 1


def main():
    parser = argparse.ArgumentParser(
        description='Python AST parser')

    parser.add_argument("--p",
                        type=str,
                        nargs='?',
                        dest='project',
                        help="project to analyze")

    parser.add_argument("--l",
                        type=str,
                        nargs='?',
                        dest='projects',
                        help="list of projects")

    parser.add_argument("--out",
                        type=str,
                        nargs='?',
                        dest='out_file',
                        help="path of the result file.")

    args = (parser.parse_args())

    if args.project is None and args.projects is None and args.out_file is None:
        raise Exception("argument missing")
    elif args.project is not None and args.out_file is not None:
        write_row_in_csv(filename=args.out_file, row=['project', 'for', 'list', 'req', 'setup', 'test'])
        stats = Stats(project=args.project, out_file=args.out_file)
        stats.save_stats()
    elif args.projects is not None and args.out_file is not None:
        write_row_in_csv(filename=args.out_file, row=['project', 'for', 'list', 'req', 'setup', 'test'])
        fp = open(args.projects)
        projects = fp.readlines()
        for project in projects:
            print(project)
            stats = Stats(project=project.replace("\n",""), out_file=args.out_file)
            stats.save_stats()

    print("done")


if __name__ == '__main__':
    main()
