import ast
import copy
import testCases

from model import VariableType, Symbol, Scope, ScopeType
from symbolTable import SymbolTable


class Source(ast.NodeVisitor):
    """
        For a symbol at a specific line
        we retrace the value if possible
    """

    def __init__(self, tree):
        self.tree = tree
        self.symbolTable = SymbolTable(tree=tree)
        self.symbolTable.build()
        self.scope = None

    def reset(self):
        self.scope = None

    # TODO maybe rename this function to get_value
    def getSource(self, symbol: Symbol, lineno: int, var_type: VariableType) -> object:
        # reset first
        self.reset()
        value = None

        # find the scope for the given symbol and the lineno
        scopes = reversed(self.symbolTable.get(symbol))

        # need to handle case depending on the variable type
        if var_type == VariableType.ARGUMENT:
            for scope in scopes:
                # getting the most recent utilization
                self.scope, var = scope.get_most_recent_occurrence(symbol=symbol, lineno=lineno)
                if var is not None and self.scope is not None:
                    symbol, lineno = var
                    value = self.scope.get_value(symbol=symbol, lineno=lineno)
                    break

        else:
            # searching the good scope and the value so that we can retrace
            for scope in scopes:
                value = scope.get_value(symbol=symbol, lineno=lineno)
                if value is not None:
                    self.scope = scope
                    break

        # once we found the good scope and the node we can to retrace
        if value is not None:
            return self.visit(value)
        else:
            #print("{symbol} at line {lineno} could not be found check the number of the line and the symbol id"
            #      .format(symbol=symbol, lineno=lineno))
            #exit(0)
            #return None

            raise Exception("Symbol could not be found check the number of the line and the symbol id")

    def visit_Constant(self, node: ast.Constant):
        return node.value

    def visit_Name(self, node: ast.Name):
        # helper function to help process the variable
        def process_var(variable: tuple[Symbol, int], scope: Scope):
            s, l = variable
            v = scope.get_value(symbol=s, lineno=l)

            # TODO need to handle some case here
            if isinstance(v, ast.List):
                return s.getName()

            r = self.visit(v)
            # maybe add a marker to specify if the node was already visited or not
            return r

        symbol = Symbol.symbol(n=node.id)
        res = None

        if self.scope.type == ScopeType.FOR and self.scope.iter_var == symbol:
            return symbol.getName()

        self.scope, var = self.scope.get_most_recent_occurrence(symbol=symbol, lineno=node.lineno)

        if var is not None and self.scope is not None:
            # the scope did not change
            res = process_var(variable=var, scope=self.scope)
        else:
            # parent = self.scope.get_parent_scope()
            while self.scope is not None:
                # look for the most recent used of the variable
                self.scope, var = self.scope.get_most_recent_occurrence(symbol=symbol, lineno=node.lineno)
                if var is not None and self.scope is not None:
                    res = process_var(variable=var, scope=self.scope)
                    # no need to continue the search
                    break
                else:
                    if self.scope is not None:
                        self.scope = self.scope.get_parent_scope()

        return res

    def visit_BinOp(self, node: ast.BinOp):
        left = node.left
        right = node.right

        # need to save the current scope because we are going to do the search
        # 2 times for this node
        current_scope = self.scope
        left_res = self.visit(left)

        # better to work with copy instead of modifying directly the tree
        new_node = copy.deepcopy(node)

        if isinstance(left, ast.Name):
            new_node.left.id = left_res

        self.scope = current_scope
        right_res = self.visit(right)

        if isinstance(right, ast.Name):
            new_node.right.id = right_res

        self.scope = current_scope

        # return the good binop node
        return new_node

        # should return a tuple here because we process two node
        # return left_res, right_res


def main():
    src = ''.join(testCases.test_a)
    tree = ast.parse(src)
    source = Source(tree=tree)

    # symbol = Symbol.symbol(n="item")
    # lineno = 8

    print(source.symbolTable)
    print("number of scopes for this program", source.symbolTable.getNumScop())

    # res = source.getSource(symbol=symbol, lineno=lineno, var_type=VariableType.ARGUMENT)
    # print(res)


if __name__ == '__main__':
    main()
