""" helper to process ast node
    code by: Ossim Belias
"""
import ast

from node import *
from model import Symbol, VariableType
from source import Source


def get_source(variable=None, source=None, var_type=None) -> object:
    """ get the of given variable """
    assert variable is not None, "variable should not be note"
    assert var_type is not None, "variable type should not be none"
    assert type(source) == Source, "source should be of Source type"

    # import helper

    # print("variable")
    # helper.print_node(variable)

    if isinstance(variable, ast.Name):
        symbol = Symbol.symbol(n=variable.id)
        elt_source = source.getSource(symbol=symbol,
                                      lineno=variable.lineno,
                                      var_type=var_type)

        if isinstance(elt_source, str):
            variable.id = elt_source
        else:
            variable = elt_source

    elif isinstance(variable, ast.BinOp):
        left = variable.left
        right = variable.right

        if isinstance(left, ast.Name):
            symbol = Symbol.symbol(n=left.id)
            left_source = source.getSource(symbol=symbol,
                                           lineno=variable.lineno,
                                           var_type=var_type)
            variable.left.id = left_source

        if isinstance(right, ast.Name):
            symbol = Symbol.symbol(n=right.id)
            right_source = source.getSource(symbol=symbol,
                                            lineno=variable.lineno,
                                            var_type=var_type)
            variable.right.id = right_source

    return variable


def get_properties_expr(node):
    """
    extract properties of the giver ast.Expr node
    :param node: ast.Expr node
    :return: a node_expr containing the func and its arguments
    """
    func = node.value.func
    args = node.value.args
    node_expr = NodeExpr(func, args)
    return node_expr


def process_expr(node_expr):
    """
    Check if the node_expr is an appending to a list
    :param node_expr: the node we want to check
    :return: True if the node is an appending False otherwise
    """
    if isinstance(node_expr.value, ast.Call) \
            and isinstance(node_expr.value.func, ast.Attribute) \
            and node_expr.value.func.attr == 'append':
        return get_properties_expr(node_expr)

    return None


def process_body_If(body):
    """
    Extract node containing in the ast.If node body
    :param body: the body of the ast.If node
    :return: all the nodes that the given body contain
    """
    properties = []
    for node in body:
        if isinstance(node, ast.Expr):
            node_expr = process_expr(node)
            if node_expr is not None:
                # print("here", node_expr.func_expr, node_expr.args_expr)
                properties.append(node_expr)

    return properties


def process_If(node_if):
    """
    Process the node_if
    :param node_if: the if that we want to process
    :return: the good if node depending on the given if node
    """
    node_expr = process_body_If(node_if.body)

    if len(node_expr) == 0:
        return None

    if len(node_if.orelse) == 0:
        return NodeIf(node_if.test, node_expr)
    else:
        for node in node_if.orelse:
            if isinstance(node, ast.Expr):
                orelse_expr = process_expr(node)
                # return here because can only have one expression no need of a list
                return NodeIFExp(node_if.test, node_expr, orelse_expr)


def process_for(node):
    pass


def process_body(body):
    """
    Process body of a ast.For node
    :param body: the body of the ast.For node
    :return: node that the given body contains
    """
    nodes = []

    for node in body:
        if isinstance(node, ast.Expr):
            node_expr = process_expr(node)
            if node_expr is not None:
                nodes.append(node_expr)
        elif isinstance(node, ast.If):
            node_if = process_If(node)
            if node_if is not None:
                nodes.append(node_if)
        elif isinstance(node, ast.For):
            body = process_body(node.body)
            node_for = NodeFor(iter=node.iter, target=node.target, body=body)
            nodes.append(node_for)
        # TODO: want to process another type of node
        elif isinstance(node, ast.Try):
            print("Another type of node found inside the body. Can't transform this for node yet")
            return []

    return nodes


def istype(node, t):
    """
    check if the type of the node is the same as the given type
    :param node: the node that we want to check
    :param t: the type we want to compare
    :return: True is the type are same otherwise False
    """
    return type(node) == t


def generate_astIfExpr(node: NodeIFExp, source: Source):
    body = get_source(variable=node.node_expr[0].args_expr[0],
                      source=source,
                      var_type=VariableType.ARGUMENT)

    orelse = get_source(variable=node.orelse.args_expr[0],
                        source=source,
                        var_type=VariableType.ARGUMENT)

    return ast.IfExp(node.test, body, orelse)


def generate_astListComp(node: NodeListComp):
    return ast.ListComp(node.elt, node.generators)


def generate_astAssign(node: NodeAssign):
    ast_assign = ast.Assign(node.targets,
                            generate_astListComp(node.value),
                            type_comment=None)

    return ast_assign
