import ast
import enum
import sys


class Symbol(object):
    """
        class symbol:
        it makes that for every occurence of any string we get each the same
        symbol
    """
    symbol_dict = dict()

    def __init__(self, name):
        self.__name = name

    def __str__(self):
        return self.__name

    def __del__(self):
        self.symbol_dict.clear()
        self.__name = None

    def __hash__(self):
        return hash(self.__name)

    def __eq__(self, other):
        if isinstance(other, Symbol):
            return self.__name == other.__name

        return False

    def getName(self):
        return self.__name

    @staticmethod
    def symbol(n: str):
        # just to help us look faster in the dictionary
        u = sys.intern(n)
        s = Symbol(u)
        if s not in Symbol.symbol_dict.keys():
            Symbol.symbol_dict[u] = s

        return s


class ScopeType(enum.Enum):
    """
        different type of scope that we can have
        across the program
    """
    MAIN = 1
    FOR = 2
    IF = 3
    ELSE = 4
    AFTER_IF = 5
    FUNCTION_DEF = 6


class VariableType(enum.Enum):
    """
        different type of variable that we can have
        across if a scope
    """
    LOCAL = 0
    ARGUMENT = 1


class Scope(object):
    """
    a scope is represented by:
        - a set of instructions (symbol, line, value)
        - a set of variable declared in the scope
        - its type
        - list of inner scope
        - link to the parent scope
        - iter variable if it is an iterable scope (i.e for-loop, while-loop)
    """

    def __init__(self, id_scope=0, scope_type=ScopeType.MAIN):
        self.instructions = dict()
        self.local_vars = list()
        self.id = id_scope
        self.type = scope_type

        # iter var if it is scope type of for need to be set
        self.iter_var = None

        # parents scopes
        self.parent = None

        # successors scopes #TODO maybe not needed finally
        self.innerScopes = []

    def __del__(self):
        self.instructions.clear()
        self.local_vars.clear()
        self.innerScopes.clear()
        self.parent = None

    def __str__(self):
        s = "\n" + "local vars of the {type} scope {id}: ".format(id=self.id, type=self.type.name)
        for var in self.local_vars:
            symbol, lineno = var
            s += "({symbol}, {lineno})".format(symbol=symbol, lineno=lineno) + " "

        s += "\n"
        s += "----- Begin instructions list for scope {id} ----".format(id=self.id) + "\n"

        for key, value in self.instructions.items():
            s += "{key}: {value}".format(key=key, value=value) + "\n"

        s += "------ End instructions list {id} -----".format(id=self.id)

        return s

    def get_parent_scope(self):
        return self.parent

    def get_inner_scopes(self):
        for scope in reversed(self.innerScopes):
            yield scope

    def insert_instruction(self, symbol: Symbol, lineno: int, node):
        # normally we can't have duplicate here
        key = (symbol.getName(), lineno)
        if key not in self.instructions.keys():
            self.instructions[key] = node

    def insert_var(self, symbol: Symbol, lineno: int):
        var = (symbol, lineno)
        self.local_vars.append(var)

    def get_value(self, symbol: Symbol, lineno: int):
        key = (symbol.getName(), lineno)
        if key in self.instructions.keys():
            return self.instructions[key]
        return None

    def get_line_declaration(self, symbol: Symbol):
        for var in self.local_vars:
            s, lineno = var
            if s.getName() == symbol.getName():
                return lineno

    def get_most_recent_occurrence(self, symbol: Symbol, lineno: int):
        # not efficient at all
        # return the most recent occurrence of a symbol(var)
        # and the scope where it was found
        inv_instructions = reversed(list(self.instructions.items()))
        for key, node in inv_instructions:
            name, line = key
            if line == lineno:
                continue

            if name == "IF" or name == "ELSE" or name == "FOR":
                scope, res = node.get_most_recent_occurrence(symbol=symbol, lineno=lineno)
                if res is not None:
                    return scope, res
            else:
                inverse_local_vars = reversed(self.local_vars)
                for var in inverse_local_vars:
                    s, lineno = var
                    if s == symbol:
                        return self, var

        return self, None

    def get_all_occurrences(self, symbol: Symbol, lineno: int):
        all_occurences = []
        inv_instructions = reversed(list(self.instructions.items()))
        for key, value in inv_instructions:
            name, line = key
            if line == lineno:
                continue

            if name == "IF" or name == "ELSE":
                all_occurences += value.get_all_occurrences(symbol=symbol, lineno=lineno)
            else:
                inverse_local_vars = reversed(self.local_vars)
                for var in inverse_local_vars:
                    s, line = var
                    if s == symbol and line < lineno and var not in all_occurences:
                        all_occurences.append(var)

        return all_occurences

    def is_local_var(self, symbol: Symbol):
        for var in self.local_vars:
            s, lineno = var
            if s == symbol:
                return True
        return

    def get_instructions(self):
        for key, value in self.instructions.items():
            yield key, value


def calculate_complexity(node):
    """:return the cyclomatic complexity of a node"""
    count = 1
    if hasattr(node, "body"):
        for child in node.body:
            if isinstance(child, ast.If) or \
                    isinstance(child, ast.For) or \
                    isinstance(child, ast.While):
                count += 1

    return count


def fan_in(scope: Scope):
    """ return the list of fan in inside a scope"""
    fan_in_lst = []

    for key, value in scope.get_instructions():
        symbol, lineno = key
        if scope.is_local_var(symbol=symbol):
            fan_in_lst.append(symbol)

    return fan_in_lst


def fan_out(scope: Scope):
    """return the list of fan out inside a scope"""
    fan_out_lst = []
    for key, value in scope.get_instructions():
        symbol, lineno = key
        if not scope.is_local_var(symbol=symbol):
            fan_out_lst.append(symbol)

    return fan_out_lst


def func():
    a = 0
    b = 1
    lst_a = []

    for i in range(10):
        a = a + i
        lst_a.append(b)

    print(lst_a)