"""
This script generates GIS data on facilities:

Outputs:
* ResourceFile_Facility_locations.csv
* facility_distances.csv

The following variables are added to the dataset generated by consumables_avaialbility_estimation.py:
1. facility  GIS coordinates
2. Distance and drive time to corresponding District Health office
3. Distance and drive time to corresponding Regional Medical Store (warehouse)

Inputs:
Dropbox location - ~05 - Resources/Module-healthsystem/consumables raw files/gis_data/LMISFacilityLocations_raw.xlsx

NB. The comment of this file are commented-out because the script requires dependencies that are not included in the
TLO framework at the time of writing.
"""


"""
import datetime
from pathlib import Path

import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
import requests
import googlemaps as gmaps
import requests
from matplotlib.lines import Line2D


# Path to TLO directory
outputfilepath = Path("./outputs")
resourcefilepath = Path("./resources")
path_for_new_resourcefiles = resourcefilepath / "healthsystem/consumables"

# Set local Dropbox source
path_to_dropbox = Path(  # <-- point to the TLO dropbox locally
    'C:/Users/sm2511/Dropbox/Thanzi la Onse'
    # '/Users/tbh03/Dropbox (SPH Imperial College)/Thanzi la Onse Theme 1 SHARE'
)

path_to_files_in_the_tlo_dropbox = path_to_dropbox / "05 - Resources/Module-healthsystem/consumables raw files/"

# define a timestamp for script outputs
timestamp = datetime.datetime.now().strftime("_%Y_%m_%d_%H_%M")

# print the start time of the script
print('Script Start', datetime.datetime.now().strftime('%H:%M'))

# Use googlemaps package to obtain GIS coordinates using facility names
GCODE_URL = 'https://maps.googleapis.com/maps/api/geocode/json?'
GCODE_KEY = ''  # PLaceholder to enter googlemaps API
# gmaps = gmaps.Client(key=GCODE_KEY)

# 1. Clean Master Health Facility Registry (MHFR) data
######################################################################
# Clean locations for facilities for which GIS data was not available on incorrect in the MHFR
# --- 1.1 Load and set up data --- #
fac_gis = pd.read_excel(open(path_to_files_in_the_tlo_dropbox / 'gis_data/LMISFacilityLocations_raw.xlsx',
                             'rb'), sheet_name='final_gis_data')
fac_gis = fac_gis.rename(
    columns={'LMIS Facility List': 'fac_name', 'OWNERSHIP': 'fac_owner', 'TYPE': 'fac_type', 'STATUS': 'fac_status',
             'ZONE': 'zone', 'DISTRICT': 'district', 'DATE OPENED': 'open_date', 'LATITUDE': 'lat',
             'LONGITUDE': 'long'})

# Create a new column providing source of GIS data
fac_gis['gis_source'] = ""

# Store unique district names
districts = fac_gis['district'].unique()

# Preserve rows with missing or incorrect location data in order to derive GIS data using googlemaps API
cond1 = fac_gis['lat'] > -8.5
cond2 = fac_gis['lat'] < -17.5
cond3 = fac_gis['long'] > 36.5
cond4 = fac_gis['long'] < 32.5
conda = cond1 | cond2 | cond3 | cond4  # outside Malawi's boundaries
fac_gis_noloc = fac_gis[fac_gis.lat.isna() | conda]
fac_gis_noloc = fac_gis_noloc.reset_index()
fac_gis_noloc = fac_gis_noloc.drop(columns='index')

# Edit data source
cond_originalmhfr = fac_gis.lat.notna() & ~conda
fac_gis.loc[cond_originalmhfr, 'gis_source'] = 'Master Health Facility Registry'
cond_manual = fac_gis['manual_entry'].notna()
fac_gis.loc[cond_manual, 'gis_source'] = 'Manual google search'

fac_gis_clean = fac_gis[~conda & fac_gis.lat.notna()]  # save clean portion of raw data to be appended later


# --- 1.2 Geocode facilities with missing data --- #
# Define a function to geocode locations based on names
def reverse_gcode(location):
    location = str(location).replace(' ', '+')
    nav_req = 'address={}&key={}'.format(location, GCODE_KEY)
    request = GCODE_URL + nav_req
    result = requests.get(request)
    data = result.json()
    status = data['status']

    geo_location = {}
    if str(status) == "OK":
        sizeofjson = len(data['results'][0]['address_components'])
        for i in range(sizeofjson):
            sizeoftype = len(data['results'][0]['address_components'][i]['types'])
            if sizeoftype == 3:
                geo_location[data['results'][0]['address_components'][i]['types'][2]] = \
                    data['results'][0]['address_components'][i]['long_name']

            else:
                if data['results'][0]['address_components'][i]['types'][0] == 'administrative_area_level_1':
                    geo_location['state'] = data['results'][0]['address_components'][i]['long_name']

                elif data['results'][0]['address_components'][i]['types'][0] == 'administrative_area_level_2':
                    geo_location['city'] = data['results'][0]['address_components'][i]['long_name']
                    geo_location['town'] = geo_location['city']

                else:
                    geo_location[data['results'][0]['address_components'][i]['types'][0]] = \
                        data['results'][0]['address_components'][i]['long_name']

        formatted_address = data['results'][0]['formatted_address']
        geo_location['lat'] = data['results'][0]['geometry']['location']['lat']
        geo_location['lang'] = data['results'][0]['geometry']['location']['lng']
        geo_location['formatted_address'] = formatted_address

        return geo_location


# Extract latitude, longitude and city based on facility name
for i in range(len(fac_gis_noloc)):
    try:
        try:
            try:
                print("Processing facility", fac_gis_noloc['fac_name'][i])
                geo_info = reverse_gcode(fac_gis_noloc['fac_name'][i] + 'Malawi')
                fac_gis_noloc['lat'][i] = geo_info['lat']
                fac_gis_noloc['long'][i] = geo_info['lang']
                fac_gis_noloc['gis_source'][i] = 'Google maps geolocation'
                fac_gis_noloc['district'][i] = geo_info['city']
            except ValueError:
                pass
        except TypeError:
            pass
    except KeyError:
        pass

# Drop incorrect GIS coordinates from the above generated dataset
conda = fac_gis_noloc.district.isin(districts)  # districts not from Malawi
cond1 = fac_gis_noloc['lat'] > -8.5
cond2 = fac_gis_noloc['lat'] < -17.5
cond3 = fac_gis_noloc['long'] > 36.5
cond4 = fac_gis_noloc['long'] < 32.5
condb = cond1 | cond2 | cond3 | cond4  # outside Malawi's boundaries
fac_gis_noloc.loc[~conda | condb, 'lat'] = np.nan
fac_gis_noloc.loc[~conda | condb, 'long'] = np.nan
fac_gis_noloc.loc[~conda | condb, 'district'] = np.nan

cond = fac_gis_noloc.gis_source.isna()
fac_gis_noloc.loc[cond, 'lat'] = np.nan
fac_gis_noloc.loc[cond, 'long'] = np.nan

# Append newly generated GIS information to the raw data
fac_gis = fac_gis_noloc.append(fac_gis_clean)

# Drop incorrect GIS coordinates based on later comparison with district data from LMIS
list_of_incorrect_locations = ['Bilal Clinic', 'Biliwiri Health Centre', 'Chilonga Health care Health Centre',
                               'Diamphwi Health Centre', 'Matope Health Centre (CHAM)', 'Nambazo Health Centre',
                               'Nkhwayi Health Centre', 'Nsambe Health Centre (CHAM)', 'Padley Pio Health Centre',
                               'Phanga Health Centre', 'Somba Clinic', "St. Martin's Molere Health Centre CHAM",
                               'Ngapani Clinic', 'Mulungu Alinafe Clinic', 'Mdeza Health Centre',
                               'Matandani Health Centre (CHAM)',
                               'Sunrise Clinic', 'Sucoma Clinic']
mapped_to_malawi = fac_gis.lat == -13.254308
cond = fac_gis.fac_name.isin(list_of_incorrect_locations) | mapped_to_malawi
fac_gis.loc[cond, 'lat'] = np.nan
fac_gis.loc[cond, 'long'] = np.nan
fac_gis.loc[cond, 'gis_source'] = np.nan
fac_gis.loc[cond, 'district'] = np.nan

# 2. Clean data using information from LMIS #
#####################################################################################################
# --- 2.1 Load and set up LMIS data --- #
stkout_df = pd.read_csv(path_for_new_resourcefiles / "ResourceFile_Consumables_availability_and_usage.csv",
                        low_memory=False)

# Drop rows which can't be used in regression analysis
regsubset_cond1 = stkout_df['data_source'] == 'original_lmis_data'
regsubset_cond2 = stkout_df[
                      'fac_type_tlo'] == 'Facility_level_0'  # since only one facility from Mchinji reported in OpenLMIS
stkout_df_reg = stkout_df[regsubset_cond1 & ~regsubset_cond2]

# Clean some district names to match with master health facility registry
rename_districts = {
    'Nkhota Kota': 'Nkhotakota',
    'Nkhata bay': 'Nkhata Bay'
}
stkout_df['district'] = stkout_df['district'].replace(rename_districts)

# Keep only relevant columns
lmis_district = stkout_df[['fac_name', 'fac_type_tlo', 'district']]
lmis_district = lmis_district.drop_duplicates()

# --- 2.2 Clean district column and assign relevant DHO to each facility --- #
# Manual fixes before assigning DHO
# Master Health facility registry did not differentiate between Mzimba North and Mzimba South --> get this data
# and any other district discrepancies from LMIS
fac_gis = fac_gis.rename(columns={'district': 'district_mhfr'})
fac_gis = pd.merge(fac_gis, lmis_district, how='left', on='fac_name')

list_mhfr_district_is_correct = ['Chididi Health Centre', 'Chikowa Health Centre',
                                 'Chileka Health Centre']
cond_mhfr_district_is_correct = fac_gis.fac_name.isin(list_mhfr_district_is_correct)
cond_lmis_district_missing = fac_gis.district.isna()
fac_gis.loc[cond_mhfr_district_is_correct | cond_lmis_district_missing, 'district'] = fac_gis.district_mhfr
fac_gis = fac_gis.drop(columns=['zone', 'district_mhfr', 'open_date', 'manual_entry'])

# --- 1.3 Extract final file with GIS locations into .csv --- #
fac_gis = fac_gis[fac_gis['lat'].notna()]  # Keep rows with GIS locations
fac_gis.to_csv(path_for_new_resourcefiles / "ResourceFile_Facility_locations.csv")

# Locate the corresponding DHO for each facility
cond1 = fac_gis['fac_name'].str.contains('DHO')
cond2 = fac_gis['fac_name'].str.contains('istrict')
# Create columns indicating the coordinates of the corresponding DHO for each facility
dho_df = fac_gis[cond1 | cond2].reset_index()
# Rename columns
dho_df = dho_df.rename(columns={'lat': 'lat_dh', 'long': 'long_dh'})

# Merge main GIS dataframe with corresponding DHO
fac_gis = pd.merge(fac_gis, dho_df[['district', 'lat_dh', 'long_dh']], how='left', on='district')

# --- 2.3 Assign relevant CMST Regional Medical Store to each facility --- #
# Create columns indicating the coordinates of the corresponding CMST warehouse (regional medical store) for each
# facility
fac_gis['lat_rms'] = np.nan
fac_gis['long_rms'] = np.nan
fac_gis['rms'] = np.nan

# RMS Center (-13.980394, 33.783521)
cond_center1 = fac_gis['district'].isin(['Kasungu', 'Ntchisi', 'Dowa', 'Mchinji', 'Lilongwe', 'Ntcheu',
                                         'Dedza', 'Nkhotakota', 'Salima'])
cond_center2 = fac_gis['fac_name'].str.contains('Kamuzu Central Hospital')
fac_gis.loc[cond_center1 | cond_center2, 'lat_rms'] = -13.980394
fac_gis.loc[cond_center1 | cond_center2, 'long_rms'] = 33.783521
fac_gis.loc[cond_center1 | cond_center2, 'rms'] = 'RMS Center'

# RMS North (-11.425590, 33.997467)
cond_north1 = fac_gis['district'].isin(['Nkhata Bay', 'Rumphi', 'Chitipa', 'Likoma', 'Karonga',
                                        'Mzimba North', 'Mzimba South'])
cond_north2 = fac_gis['fac_name'].str.contains('Mzuzu Central Hospital')
fac_gis.loc[cond_north1 | cond_north2, 'lat_rms'] = -11.425590
fac_gis.loc[cond_north1 | cond_north2, 'long_rms'] = 33.997467
fac_gis.loc[cond_north1 | cond_north2, 'rms'] = 'RMS North'

# RMS South (-15.804544, 35.021192)
cond_south1 = fac_gis['district'].isin(['Blantyre', 'Balaka', 'Machinga', 'Zomba', 'Mangochi', 'Thyolo', 'Nsanje',
                                        'Chikwawa', 'Mwanza', 'Neno', 'Mulanje', 'Phalombe', 'Chiradzulu'])
cond_south2 = fac_gis['fac_name'].str.contains('Queen Elizabeth Central')
cond_south3 = fac_gis['fac_name'].str.contains('Zomba Central')
cond_south4 = fac_gis['fac_name'].str.contains('Zomba Mental')
fac_gis.loc[cond_south1 | cond_south2 | cond_south3 | cond_south4, 'lat_rms'] = -15.804544
fac_gis.loc[cond_south1 | cond_south2 | cond_south3 | cond_south4, 'long_rms'] = 35.021192
fac_gis.loc[cond_south1 | cond_south2 | cond_south3 | cond_south4, 'rms'] = 'RMS South'
fac_gis['district'].unique()

# 3. Generate data on distance and travel time between facilities and DHO/RMS #
#####################################################################################################
# --- 3.1 Distance and travel time of each facility from the corresponding DHO --- #
fac_gis['dist_todh'] = np.nan
fac_gis['drivetime_todh'] = np.nan
for i in range(len(fac_gis)):
    try:
        # print("Processing facility", i)
        latfac = fac_gis['lat'][i]
        longfac = fac_gis['long'][i]
        latdho = fac_gis['lat_dh'][i]
        longdho = fac_gis['long_dh'][i]
        origin = (latdho, longdho)
        dest = (latfac, longfac)

        fac_gis['dist_todh'][i] = \
            gmaps.distance_matrix(origin, dest, mode='driving')['rows'][0]['elements'][0]['distance']['value']
        fac_gis['drivetime_todh'][i] = \
            gmaps.distance_matrix(origin, dest, mode='driving')['rows'][0]['elements'][0]['duration']['value']
    except:
        pass

# --- 3.2 Distance and travel time of each facility from the corresponding RMS --- #
fac_gis['dist_torms'] = np.nan
fac_gis['drivetime_torms'] = np.nan
for i in range(len(fac_gis)):
    try:
        # print("Processing facility", i)
        latfac = fac_gis['lat'][i]
        longfac = fac_gis['long'][i]
        latdho = fac_gis['lat_rms'][i]
        longdho = fac_gis['long_rms'][i]
        origin = (latdho, longdho)
        dest = (latfac, longfac)

        fac_gis['dist_torms'][i] = \
            gmaps.distance_matrix(origin, dest, mode='driving')['rows'][0]['elements'][0]['distance']['value']
        fac_gis['drivetime_torms'][i] = \
            gmaps.distance_matrix(origin, dest, mode='driving')['rows'][0]['elements'][0]['duration']['value']
    except:
        pass

# Update distance values from DH to 0 for levels 2 and above
cond1 = fac_gis['fac_type_tlo'] == 'Facility_level_2'
cond2 = fac_gis['fac_type_tlo'] == 'Facility_level_3'
cond3 = fac_gis['fac_type_tlo'] == 'Facility_level_4'
fac_gis.loc[cond1 | cond2 | cond3, 'dist_todh'] = 0
fac_gis.loc[cond1 | cond2 | cond3, 'drivetime_todh'] = 0

# 4. Save data to be merge into Consumable availabilty dataset for regression analysis #
#####################################################################################################
# Keep only necessary columns and save as .csv
fac_gis = fac_gis[['district', 'rms', 'lat', 'long', 'lat_dh', 'long_dh', 'lat_rms', 'long_rms',
                   'dist_torms', 'drivetime_torms', 'dist_todh', 'drivetime_todh', 'fac_name', 'gis_source']]

# - 1.2.5 Export distances file to dropbox - #
fac_gis.to_csv(path_to_files_in_the_tlo_dropbox / 'gis_data/facility_distances.csv')

# 5. Descriptive graphs #
#####################################################################################################
groups = fac_gis.groupby('district')

# Scatterplot of distance and drive time to DHO
fig, ax = plt.subplots()
ax.margins(0.05)  # Optional, just adds 5% padding to the autoscaling
for name, group in groups:
    ax.plot(group.dist_todh / 1000, group.drivetime_todh, marker='o', linestyle='', ms=5, label=name)
# Shrink current axis by 20% to fit legend
box = ax.get_position()
ax.set_position([box.x0, box.y0, box.width * 0.8, box.height])
# Put a legend to the right of the current axis
ax.legend(loc='center left', bbox_to_anchor=(1, 0.5))
plt.xlabel("Distance (kilometers)", fontsize=12)
plt.ylabel("Drive time (minutes)", fontsize=12)
plt.savefig('C:/Users/sm2511/OneDrive - University of York/Desktop/faclocation_wrtdh_new.png')

# Scatterplot of distance and drive time to RMS
groups = fac_gis.groupby('rms')
fig, ax = plt.subplots()
ax.margins(0.05)  # Optional, just adds 5% padding to the autoscaling
for name, group in groups:
    ax.plot(group.dist_torms / 1000, group.drivetime_torms, marker='o', linestyle='', ms=5, label=name)
# Shrink current axis by 20% to fit legend
box = ax.get_position()
ax.set_position([box.x0, box.y0, box.width * 0.8, box.height])
# Put a legend to the right of the current axis
ax.legend(loc='center left', bbox_to_anchor=(1, 0.5))
plt.xlabel("Distance (kilometers)", fontsize=12)
plt.ylabel("Drive time (minutes)", fontsize=12)
plt.savefig('C:/Users/sm2511/OneDrive - University of York/Desktop/faclocation_wrtrms.png')
"""
