/* Copyright 2023 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include <memory>
#include <utility>

#include "mhlo/IR/hlo_ops.h"
#include "mhlo/transforms/passes.h"
#include "mhlo/transforms/rewriters.h"
#include "mlir/Dialect/Func/IR/FuncOps.h"
#include "mlir/IR/BuiltinTypes.h"
#include "mlir/IR/MLIRContext.h"
#include "mlir/IR/PatternMatch.h"
#include "mlir/IR/Value.h"
#include "mlir/Pass/Pass.h"
#include "mlir/Support/LogicalResult.h"
#include "mlir/Transforms/GreedyPatternRewriteDriver.h"

namespace mlir {
namespace mhlo {

#define GEN_PASS_DEF_LEGALIZECROSSREPLICASUMTOALLREDUCEPASS

#include "mhlo/transforms/mhlo_passes.h.inc"

namespace {

struct CrossReplicaSumToAllReducePattern
    : public OpRewritePattern<CrossReplicaSumOp> {
  using OpRewritePattern<CrossReplicaSumOp>::OpRewritePattern;
  LogicalResult matchAndRewrite(CrossReplicaSumOp crossReplicaSumOp,
                                PatternRewriter &rewriter) const override {
    auto allReduceOp = rewriter.replaceOpWithNewOp<AllReduceOp>(
        crossReplicaSumOp, crossReplicaSumOp.getType(),
        crossReplicaSumOp.getOperand(), crossReplicaSumOp.getReplicaGroups(),
        /*channel_handle=*/ChannelHandleAttr(),
        /*use_global_device_ids=*/false);

    auto *block = rewriter.createBlock(&allReduceOp.getComputation());
    auto elementType = RankedTensorType::get({}, allReduceOp.getResults()
                                                     .front()
                                                     .getType()
                                                     .dyn_cast<TensorType>()
                                                     .getElementType());
    auto location = allReduceOp.getComputation().getLoc();
    block->addArguments({elementType, elementType}, {location, location});

    auto addOp = rewriter.create<AddOp>(location, block->getArgument(0),
                                        block->getArgument(1));

    rewriter.create<ReturnOp>(location, addOp.getResult());

    return success();
  }
};

struct LegalizeCrossReplicaSumToAllReducePass
    : public impl::LegalizeCrossReplicaSumToAllReducePassBase<
          LegalizeCrossReplicaSumToAllReducePass> {
  void runOnOperation() override {
    RewritePatternSet patterns(&getContext());
    populateCrossReplicaSumToAllReducePatterns(&getContext(), &patterns);
    if (failed(applyPatternsAndFoldGreedily(getOperation(),
                                            std::move(patterns)))) {
      return signalPassFailure();
    }
  }
};

}  // namespace

void populateCrossReplicaSumToAllReducePatterns(mlir::MLIRContext *context,
                                                RewritePatternSet *patterns) {
  patterns->add<CrossReplicaSumToAllReducePattern>(context);
}

std::unique_ptr<OperationPass<func::FuncOp>>
createLegalizeCrossReplicaSumToAllReducePass() {
  return std::make_unique<LegalizeCrossReplicaSumToAllReducePass>();
}

}  // namespace mhlo
}  // namespace mlir
