%
%   This Matlab script develops parameters for a growth/death model of
%   Scenedesmus, taking into account temperature, pH and nutrient addition
%   by Dust.  To run this script you will need the function nllGR, which
%   has the negative log posterior likelihood to bo optimized during the
%   fitting procedure.  You will also need the function aGR3.m, which
%   calculates the rate curve for given parameters, inputs.
%  
%   The growth model for the algae depends on a model for nutrient release
%   by the Dust, which is parameterized in the script Nut_model.m
%
%   This is a cleaned-up and commented version of the original code,
%   written by James Powell, USU, in May 2023

%   Load Scenedesmus growth rate data from Juanma Gonzalez-Olalla
% 

%   Temp  pH  nutrient  rate

data_all=[
19	6.3	0	0.039983707
19	6.3	0	0.037263844
19	6.3	0	0.026233978
19	6.3	10	0.090262212
19	6.3	10	0.084091831
19	6.3	10	0.096357375
19	6.3	25	0.053365423
19	6.3	25	0.058619415
19	6.3	25	0.048055636
19	6.3	75	0.140401299
19	6.3	75	0.144819592
19	6.3	75	0.135943615
19	6.8	0	0.021586107
19	6.8	0	0.013063165
19	6.8	0	0.029966199
19	6.8	10	0.106843702
19	6.8	10	0.058237549
19	6.8	10	0.062148915
19	6.8	25	0.067959186
19	6.8	25	0.069881035
19	6.8	25	0.066029921
19	6.8	75	0.141260981
19	6.8	75	0.151140563
19	6.8	75	0.146225173
19	7.3	0	0.089953876
19	7.3	0	0.052318523
19	7.3	0	0.061996255
19	7.3	10	0.124953617
19	7.3	10	0.149683955
19	7.3	10	0.098936163
19	7.3	25	0.145646244
19	7.3	25	0.173248545
19	7.3	25	0.199406462
19	7.3	75	0.249592684
19	7.3	75	0.251236522
19	7.3	75	0.247943423
23	6.3	0	0.04957798
23	6.3	0	0.057798831
23	6.3	0	0.065633176
23	6.3	10	0.108384733
23	6.3	10	0.124563509
23	6.3	10	0.091664895
23	6.3	25	0.162038505
23	6.3	25	0.190739283
23	6.3	25	0.193828893
23	6.3	75	0.281210938
23	6.3	75	0.264111498
23	6.3	75	0.246406505
23	6.8	0	0.036798353
23	6.8	0	0.03464135
23	6.8	0	0.03893234
23	6.8	10	0.052966757
23	6.8	10	0.037258645
23	6.8	10	0.05681787
23	6.8	25	0.175752143
23	6.8	25	0.144556975
23	6.8	25	0.11811908
23	6.8	75	0.143258441
23	6.8	75	0.120438543
23	6.8	75	0.133372469
23	7.3	0	0.069485636
23	7.3	0	0.083527135
23	7.3	0	0.045169636
23	7.3	10	0.123430171
23	7.3	10	0.101656027
23	7.3	10	0.069485636
23	7.3	25	0.11915054
23	7.3	25	0.140191533
23	7.3	25	0.123430171
23	7.3	75	0.164324925
23	7.3	75	0.202126327
23	7.3	75	0.209355374
27	6.3	0	0.048471064
27	6.3	0	0.056557509
27	6.3	0	0.040251686
27	6.3	10	0.005952463
27	6.3	10	0.040251686
27	6.3	10	0.023396126
27	6.3	25	0.076219316
27	6.3	25	0.10977233
27	6.3	25	0.040251686
27	6.3	75	0.304427421
27	6.3	75	0.268999413
27	6.3	75	0.268999413
27	6.8	0	-0.004984253
27	6.8	0	-0.021884293
27	6.8	0	-0.036326332
27	6.8	10	0.009477249
27	6.8	10	0.002411746
27	6.8	10	0.016301639
27	6.8	25	-0.01042939
27	6.8	25	-0.014160894
27	6.8	25	-0.017977859
27	6.8	75	0.073223032
27	6.8	75	0.086517772
27	6.8	75	0.077694005
27	7.3	0	0.030331788
27	7.3	0	0.018300392
27	7.3	0	0.005498634
27	7.3	10	0.028767364
27	7.3	10	0.077306413
27	7.3	10	0.054213062
27	7.3	25	0.032120177
27	7.3	25	0.02486862
27	7.3	25	0.039117982
27	7.3	75	0.118835948
27	7.3	75	0.093478378
27	7.3	75	0.14296934    ];
%

%%
%   This cell estimates parameters for the Scenedesmus growth model, which
%   depends on a separate nutrient model.  Negative log posterior likelihood, which
%   is minimized, is alculated in the function nllGR
%
%   For the nutrient model (parameterized in ):
    %   T    = temperature, degrees C
    %   P    = pH
    %   Du   = nutrient concentration (dust)
    %   parameters controlling available nutrient response to pH and T
    %   these were fit using log normal variance
    %        KP = additive correction for low dust (1.405)
    %        a = max nutrient content for dust (multiplier) (.9425)
    %        mu, sig2 = mean and variance for log-normal nutrient model
    %        beta(1) = half sat constant for logisttic (-9.7098)
    %        beta(2) = multiplier for pH in exponent (+1.5483)
    %        beta(3) = Arrhenius constant 1/(T+273) (54.122)
%
% Parameters to be estimated for the rate curve are in parms:
%
%   parms(1:3) are b_j controlling thermal Briere rate curve
%       parms(1) = b1 = base rate
%       parms(2) = b2 = estimated upper threshold in T
%       parms(3) = b3 = adjustment of upper threshold with available R
%   parms(4:6) are c_j controlling type II nutrient uptake 
%       parms(4) = K = base half-sat value for type II uptake (Thomas)
%       parms(5) = c2 = exp change of half-sat with pH (relative to 7.3)
%       parms(6) = c3 = exp change of half-sat with T (relative to 23)
%   parms(7:10) are d_j controlling death rate from Thomas + pH response
%       parms(7) = d0 = base death rate
%       parms(8) = d1 = proportionality of exp. death rate increase 
%       parms(9) = d2 = exp change with T (relative to 23)
%       parms(10) = d3 = exponent of death response to pH (relative to 7.3)
%
%   parms(11) is the variance parameter for the error model
%
%   NOTE: the roles of c2 and c3, as well as d2 and d3, are switched from
%   what appears in the final published paper!
%
%         1    2    3    4     5   6    7     8    9    10  11
%       [ b1   b2   b3    K    c2   c3   d0   d1    d2   d3  sig
% 
data=data_all;
p0=[ .0014  27  .07   1.8   -1.6   -.09  0 .03   0.2  1.3  .04];  % initial guess
parms=p0; 

[parms, fval, iflag] = fminsearch(@(x) nllGR(data,x, 1), parms) % find min NLL using
[parms, fval, iflag] = fminsearch(@(x) nllGR(data,x, 1), parms) % run until converges
N=length(data(:,1)); k=length(parms); 
parms_SSE=parms;
AIC_SSE=2*fval+2*k % -356.93 
%     parms_SSE =[  0.0014081        27.36      0.06755       1.8691      -1.6071    -0.086225  ...
%        -3.0777e-10     0.033604  0.20428       1.2785     0.041725];      

%%
% plot line graphs of  rates and prediction vs. observation
%
pout=[parms_SSE ];
Te=linspace(10,30,1001); Dust=[0 10 25 75];
T1opt=0*Dust; T2opt=T1opt; T3opt=T1opt;
r1opt=0*Dust; r2opt=T1opt; r3opt=T1opt;

figure(2)
for j=1:4
    subplot(2,2,j)
    Du=Dust(j);
    r1=aGR3(Te,6.3,Du,pout); r2=aGR3(Te,6.8,Du,pout); r3=aGR3(Te,7.3,Du,pout);
    r1opt(j)=max(r1); T1opt(j)=Te(find(r1==r1opt(j))); % find optimum temps
    r2opt(j)=max(r2); T2opt(j)=Te(find(r2==r2opt(j)));
    r3opt(j)=max(r3); T3opt(j)=Te(find(r3==r3opt(j)));
    plot(Te,max(0,r1),'k',Te,max(0,r2),'b',Te,max(0,r3),'g')
    axis([10 30 0 .24]), title(['Dust = ' num2str(Du)])
    %legend('pH = 6.3','pH = 6.8','pH = 7.3','Location','NorthWest')
end
subplot(2,2,3), xlabel('T (^oC)'), ylabel('Growth Rate (d^-^1)')
subplot(2,2,4), xlabel('T (^oC)')
subplot(2,2,1), ylabel('Growth Rate (d^-^1)'), legend('pH = 6.3','pH = 6.8','pH = 7.3','Location','NorthEast')
subplot(2,2,2)

obs=data(:,4);
pred=aGR3(data(:,1) ,data(:,2), data(:,3), pout);
figure(6), plot(obs,pred,'b*',[min(obs) max(obs)], [min(obs)  max(obs)], 'r')
axis square, xlabel('Observed'), ylabel('Predicted'), title('Growth Rate Performance')

%% 
% density plots of rates 

figure(9)
ndu=401; pH=7.3;  % change pH for different density plots

% set up a grid of dust and temperatures for density plot
du=linspace(0,75,ndu);
ts=linspace(2,30,801);
[TS,DU]=meshgrid(ts,du);
GR63=aGR3(TS,pH,DU,pout);   % responses
topt=0*du;

for idu=1:ndu
    r1=aGR3(ts,pH,du(idu),pout); 
    topt(idu)=ts(find(r1==max(r1))); % find optimum temps at this dust level
end

% do a running average to smooth out discretization effects in the line
% plot of optimal temps
%
nsmooth=15;
for idu=1:ndu-nsmooth
        topt(idu)=mean(topt(idu:idu+nsmooth-1)); % find optimum temps at this dust level
end

pcolor(TS,DU,GR63), shading interp, colormap hot
c=colorbar;
c.Label.String='(dy^-^1)';
hold on, plot(topt, du, 'b', 'Linewidth',2'), hold off
hold on, contour(TS,DU,GR63,[0 0],'g', 'Linewidth',2'), hold off
xlabel('T (^oC)'), ylabel('Dust (mg/L)')
title(['Growth Rate, pH = ' num2str(pH)])
caxis([-.06 0.21])



