% 
%   this Matlab script develops and parameterizes a model for release of
%   phosphorus from Dust in Juanma Gonzalez-Ollala's experiments on the
%   growth rate of Scenedesmus under conditions of temperature, pH and Dust
%   addition.
%
%   This is a cleaned-up and commented version of the original code,
%   written by James Powell, USU, in May 2023

%   Load Juanma's observations of Phosphorus (P):

% Temp  pH  Dust   P
Pdata=[ 
    19	6.3	0	1.57
19	6.3	10	10.7475
19	6.3	25	18
19	6.3	75	44.6
19	6.8	0	1.323333333
19	6.8	10	4.4
19	6.8	25	11.2
19	6.8	75	15.75
19	7.3	0	1.45
19	7.3	10	2.016666667
19	7.3	25	3.18
19	7.3	75	14.2
23	6.3	0	1.76
23	6.3	10	3.736666667
23	6.3	25	8.23
23	6.3	75	34.625
23	6.8	0	1.123333333
23	6.8	10	6.2
23	6.8	25	8.0425
23	6.8	75	28.6
23	7.3	0	1.465
23	7.3	10	2.55
23	7.3	25	5.195
23	7.3	75	15.9
27	6.3	0	1.636666667
27	6.3	10	4.065
27	6.3	25	10.13666667
27	6.3	75	30.875
27	6.8	0	2.053333333
27	6.8	10	2.023333333
27	6.8	25	4.713333333
27	6.8	75	16.63333333
27	7.3	0	2.125
27	7.3	10	4.5225
27	7.3	25	7.53
27	7.3	75	16.7 ];
%%
%   determine coefficients for a logistic model of Phosphorus release by
%   Dust, with the exponent determined by an Arrhenius thermal response and
%   a linear dependce on pH.
%
%   Beyond the logistic response there is a multiplier (a) and a basic level 
%   of phosphorus  in the medium (Kp) which need to be estimated.  The
%   strategy is to do a grid search over a, Kp with an associated logistic
%   regression at each a, Kp.  The resulting log-normal error surface is
%   searched for an optimal a, Kp (and parameters recovered)
%

% bring in data as column vectors:
% temperarture   pH          Dust           phosphorus
T=Pdata(:,1); pH=Pdata(:,2); Du=Pdata(:,3); P=Pdata(:,4);
N=length(T);

% set up a grid of a, Kp values:
kp=linspace(0.1,1.6,601);
ap=linspace(.1,1.6,601);
[Kp,Ap]=meshgrid(kp,ap);
err=0*Kp; % initialize the error surface

for i=1:length(kp)
    for j=1:length(ap)
        
        KP=kp(i);
        a=ap(j);
        %  Use logit transform on the observations, assuming 
        %   P = KP + a*Du
        y=log(max(P-KP,.01)./max(.1, KP+a*Du-P));

        % set up design matrix for explanatory variables
        % linear in pH, Arrhenius in temperature
        X=[ ones(N,1) (pH) 1./(T+273) ];
        % find least-squares projection for the betas
        A=X'*X; b=X'*y;
        beta=inv(A)*b;
        % calculate the prediction on logit scale using the betas
        ex=beta(1)+beta(2)*(pH)+beta(3)./(T+273); 
        % prediction after logistic transformation
        Ppred=KP+(a*Du).*exp(ex)./(1+exp(ex));  
        
        % calculate log-normal error parameters
        mu=mean(log(P./Ppred));
        sig2=sum( (log(P./Ppred)-mu).^2 )/N; % variance for mode log normal
        % negative log log-normal error for this choice of a, Kp:
        err(j,i)=sum( 1+ log(2*pi*sig2*(P./Ppred).^2) );
    end
end
fval=min(min(err));  % find the minimum NLL over the grid of a, Kp
[jind,iind]=find(err==fval );  % min error is 405.59
% the best a and KP parameters:
KP=kp(iind) 
a=ap(jind)  

%%
%  plot the error surface and performance curve for the nutrient model

% plot the error surface and the minimum just to check
figure(12), pcolor(Kp,Ap,err), shading interp, colormap hot, title('Nutrient NLL') %plot(kp,err), title('SSE')
hold on, contour(Kp,Ap,err,30,'g'), plot([KP KP],[a a],'go'), hold off
xlabel('KP'),ylabel('a')

% calculate prediction for a, KP found above:
y=log(max(P-KP,.01)./max(.1, KP+a*Du-P));
    X=[ ones(N,1) (pH) 1./(T+273) ];
    A=X'*X; b=X'*y;
    beta=inv(A)*b 
    % beta = [9.7098, -1.5483, 54.122] is best for mode log-normal
    ex=beta(1)+beta(2)*(pH)+beta(3)./(T+273); % exponent in logistic prediction
    Ppred=KP+(a*Du).*exp(ex)./(1+exp(ex));  % actual prediction
    % for log-normal need to correct so mean is one
    mu=mean(log(P./Ppred)) % =-0.092667
    sig2=sum( (log(P./Ppred)-mu).^2 )/N % variance for mode log normal = 0.11497

    % plot prediction vs. observation:
figure(13) 
plot(P,exp(mu+.5*sig2)*Ppred,'b*',[0 45], [0 45],'r'), axis square
corrcoef([P exp(mu+.5*sig2)*Ppred]) % r = .95 for log normal


xlabel('Observed'), ylabel('Predicted'), title('Available Phosphorus')



