#--------------------------------------------------------------------------------------
#---------------Krishan Khurana's Jupiter magnetic field model for Python -------------
#--------------------------------------------------------------------------------------
# NAME:
#   JRM33+KK2005
#
# PURPOSE:
#   Given right-handed system III, either spherical or cartesian coordinates,
#   this program outputs the strength of jupiter's magnetic field in spherical 
#   or cartesian coordinates at the specified coordinates. 
#
# DESCRIPTION:
#   Calculates the magnetic field of a tilted and warped shielded magnetosphere
#   using general deformation technique.
#
# INPUTS:
#   - time  [format 'YYYY-MM-DDThh:mm:ss']
#   - SIIIRH coordinates :
#      R,theta, phi [Rj,deg,deg] or x,y,z [Rj,Rj,Rj]  
#   - xyz_in : 
#      False -> input coordinates are R,theta,phi
#      True  -> input coordinates are x,y,z 
#   - Bxyz_Out : 
#      True  -> output magnetic field in cartesian coordinates (Bx,By,Bz)
#      False -> output magnetic field in spherical coordinates (Br,Bt,Bp)
#
# OUTPUTS:
#   - Magnetic field components : 
#     Bx,By,Bz or Br,Bt,Bp [nT]
#   - mpCheck :        
#      1 -> inside the magnetopause
#      0 -> outside the magnetopause
#  
# MODIFICATION HISTORY:
#   2003, written by Krishan Khurana in Fortran 77/90
#   2008, ctimer function added by Mariel Desroche
#         (converts year, month, day, hour and second to J2000 seconds)
#   2009, converted to IDL by Adam Shinn
#   2009, leapSecond function added by Adam Shinn and Rob Wilson
#         (replaces eetime function to easily append for future leap seconds)
#   2023, converted to Python by Jonas Rabia (jonas.rabia@irap.omp.eu)
#         VIP4 internal magnetic field model (initially used in kk_2005) replaced
#         by the JRM33 internal magnetic field model
#   2023, leapSecond function updated with the new leap seconds 
#
# WEBSITE:
#   The IDL version of this code is available on the website for the 
#   Magnetospheres of Outer Planets group at the Univeristy of Colorado 
#   at Boulder, associated with the Laboratory for Atmospheric and Space Physics:
#   http://lasp.colorado.edu/MOP/resources/
#
#   JRM33 model was initially implemented by Marissa Vogt (mvogt@bu.edu) 
#   and Rob Wilson (rob.wilson@lasp.colorado.edu). The original code is available at: 
#   https://github.com/rjwilson-LASP/PSH, citation DOI: 10.5281/zenodo.6814109 
#--------------------------------------------------------------------------------------


from datetime import date
import datetime as dt 
import numpy as np 


###################### TIME-RELATED FUNCTIONS #################################

def doy(t):
    # ----------------------------------------------------------- # 
    # Translated from function doy
    # Not used 
    # ----------------------------------------------------------- # 
    y=int(t[0:4])
    m=int(t[5:7])
    d=int(t[8:10])
    doy = date(y, m, d).timetuple().tm_yday

    return doy 

def ctimer(t):
    # ----------------------------------------------------------- # 
    # Adapted from function ctimer
    # Input format: "YYYY-MM-DDThh:mm:ss" 
    # Output : ctime (sec since 1966-01-01)
    # ----------------------------------------------------------- # 

    iyr = int(t[0:4])
    imon= int(t[5:7])
    iday= int(t[8:10])
    ihr = int(t[11:13])
    imin= int(t[14:16])
    isec= int(t[17:19])

    # Number of seconds since Jan 1, 1966 
    d0 = dt.datetime(1966, 1, 1, 0, 0, 0)
    d1 = dt.datetime(iyr,imon,iday,ihr,imin,isec)

    ctime = (d1 - d0).total_seconds()
    
    return ctime 

def leapSecond(t):
    # ----------------------------------------------------------- # 
    # Adapted from function leapSecond
    #
    # Input format: "YYYY-MM-DDThh:mm:ss" 
    # Output : seconds in J2000 
    #
    # Note : new leap seconds have been added (04/2023)
    # ----------------------------------------------------------- # 
    
    iyr = int(t[0:4])
    imon= int(t[5:7])

    time = iyr*100+imon
    # Formats year and month as YYYYMM
    # Leap seconds 
    
    if (time < 197207):
        Tcor = 10 # leap second jun 30 1972
    elif (time < 197301): 
        Tcor = 11 # leap second dec 31 1972
    elif (time < 197401) : 
        Tcor = 12 # leap second dec 31 1973
    elif (time < 197501) : 
        Tcor = 13 # leap second dec 31 1974
    elif (time < 197601) : 
        Tcor = 14 # leap second dec 31 1975
    elif (time < 197701) :
        Tcor = 15 # leap second dec 31 1976
    elif (time < 197801) : 
        Tcor = 16 # leap second dec 31 1977
    elif (time < 197901) : 
        Tcor = 17 # leap second dec 31 1978
    elif (time < 198001) : 
        Tcor = 18 # leap second dec 31 1979
    elif (time < 198107) : 
        Tcor = 19 # leap second jun 30 1981
    elif (time < 198207) : 
        Tcor = 20 # leap second jun 30 1982
    elif (time < 198307) :
        Tcor = 21 # leap second jun 30 1983
    elif (time < 198507) : 
        Tcor = 22 # leap second jun 30 1985
    elif (time < 198801) : 
        Tcor = 23 # leap second dec 31 1987
    elif (time < 199001) : 
        Tcor = 24 # leap second dec 31 1989
    elif (time < 199101) : 
        Tcor = 25 # leap second dec 31 1990
    elif (time < 199207) : 
        Tcor = 26 # leap second jun 30 1992
    elif (time < 199307) : 
        Tcor = 27 # leap second jun 30 1993
    elif (time < 199407) :
        Tcor = 28 # leap second jun 30 1994
    elif (time < 199601) :
        Tcor = 29 # leap second dec 31 1995
    elif (time < 199707) :
        Tcor = 30 # leap second jun 30 1997
    elif (time < 199901) :
        Tcor = 31 # leap second dec 31 1998
    elif (time < 200601) : 
        Tcor = 32 # leap second dec 31 2005 
    elif (time < 200901) :
        Tcor = 33 # leap second dec 31 2008
    elif (time < 201207) :
        Tcor = 34 # leap second jun 30 2012  
    elif (time < 201507) :
        Tcor = 35 # leap second jun 30 2015 
    elif (time < 201701) :
        Tcor = 36 # leap second dec 31 2016  
    else : 
        Tcor = 37
        
    thirty_four_years = 0.1072958367816e10    
    leaptime = ctimer(t) + Tcor - thirty_four_years
    
    return leaptime 


###################### COORDINATES CONVERSION #################################

def JSUN(t_in): 
    # ----------------------------------------------------------- # 
    # Translated from procedure JSUN 
    #
    # INPUT: time of the data point 
    # OUTPUT: - latitude  of the Sun in SIIIRH [rad] (stheta)
    #         - longitude of the Sun in SIIIRH [rad] (sphi)
    #         - orbital phase angle of Jupiter [rad] (phase)
    # ----------------------------------------------------------- # 
    # Initialize variables 
    pi = np.pi 
    twopi = 2*pi 
    radian = pi/180  
    yrjup = 11.85652502*86400*365.25
    omega = 870.536/86400
    omegay = 2*pi/yrjup
    year = 86400*365.25
    etime1 = -8.25767955817479e8
    three = 3.123*radian
    tan3 = 0.054560676
    
    """
    # Not used : 
    sin3 = 0.054479647
    cos3 = 0.99851488
    """
    
    aa = [0.14347029, 3.1145815, -0.12025561, 0.093909436, -0.39321884e-5, 0.10194945e-3, -0.12799464]
    bb = [-4.5467523, 3.1848875, -0.16329986, -0.09776818, 0.17556527e-3, -0.01978317, 44.55915]
    
    # First calculate the latitude and longitude in non-rotating Jupiter coordinates
    # Calculate the best fit theta and fphi  

    t = leapSecond(t_in) - etime1
    
    cos_omeg = np.cos(omegay*t)
    sin_omeg = np.sin(omegay*t) 
    x5 = t/year
    x = [cos_omeg, sin_omeg, np.cos(2*omegay*t), 2*cos_omeg*sin_omeg, x5*x5, x5, 1]
    
    # fphi is phi in Jupiter fixed (non-rotating) coordinate
    fphi = np.dot(bb,x)
    stheta = np.dot(aa,x)
    
    # Now rotate the longitude to Jupiter System III
    # First Add the rotation of Jupiter around the Sun
    # fphi is the phi of the Sun as unspinning Jupiter goes around the sun 
    fphi = np.fmod((fphi + t/yrjup*360),360)

    # Next add the rotation of Jupiter around its axis.
    sphi = np.fmod((fphi - t*omega),360)
    if (sphi < 0) :
        sphi = sphi + 360
  
    sphi = sphi*radian
    stheta = stheta*radian
    
    # Added lines to take into account tan(stheta)/tan3 > 1 or < -1 
    # acostanstheta is redefined in the next lines in any case 
    if np.tan(stheta)/tan3 < 1 and np.tan(stheta)/tan3 > -1 :
        acostanstheta = np.arccos(np.tan(stheta)/tan3)
    else:
        acostanstheta = 0 
    
        
    # Now compute the orbital phase (called phi2 or phase here)
    # There are two solutions to the problem. Only one that is close to phi2b is correct
    if (stheta >= three) :
        stheta = three
        acostanstheta = 0
    if (-stheta >= three) :
        stheta = -three 
        acostanstheta = pi
    
    
    phi21 = (sphi + pi) + acostanstheta
    phi22 = (sphi + pi) - acostanstheta
    phi21 = np.fmod(phi21,twopi)
    phi22 = np.fmod(phi22,twopi)
    dphi = fphi + 48.23012
    phi2b = sphi - dphi*radian
    phi2b = np.fmod(phi2b,twopi)
    phase = phi21
    dif2 = np.abs(phi22 - phi2b)
    
    if (dif2 > 350*radian) :
        dif2 = twopi - dif2
    dif1 = np.abs(phi21 - phi2b)
    if (dif1 > 350*radian) :
        dif1 = twopi - dif1
    if (dif2 < dif1) :
        phase = phi22
    
    phase = np.fmod(phase,twopi)

    return stheta, sphi, phase
    
def JROT(t_in,vecin,From,To): 
    # ----------------------------------------------------------- # 
    # Translated from procedure JROT 
    #
    # INPUT:  - Time : [YYYY-MM-DDThh:mm:ss]
    #         - Vecin: incoming coordinate [list or array 1x3]
    #         - From : incoming coordinate system [str] 
    #         - To   : outgoing coordinate system [str] 
    # 
    # OUTPUT: - Vecout: outgoing coordinate
    #
    # Supported coordinate systems : 
    #         - S3C System III(RH) Cartesian 
    #         - JSO Jupiter-Sun-Orbital 
    #         - JSM Jupiter-Sun-Magnetic
    #         - DIP Dipole (cartesian) 
    #         - JSS Jupiter-Sun-Spin
    # ----------------------------------------------------------- # 
    
    # Initialize variables 
    Identity = np.identity(3)
    sin3 = 0.054479647
    cos3 = 0.99851488
    
    """
    # Not used :
    pi = np.pi
    twopi = 2*pi
    three = 3.123*radian
    radian = pi/180
    degree = 180/pi
    tan3 = 0.054560676
    TH_DIP = 0.16755161
    PH_DIP = -3.5255651
    """

    
    dipole = np.zeros([3,3])
    dipole[:,0] = [-0.9141996,   0.36936062, -0.16676875]
    dipole[:,1] = [-0.3746066,  -0.92718385,           0]
    dipole[:,2] = [-0.154625290, 0.0624726744,  0.98599604]
    
    dummy = np.zeros([3,3])
    second = np.zeros([3,3])

    # Calling JSUN function 
    stheta, sphi, phase = JSUN(t_in)
    
    sin_stheta = np.sin(stheta)
    cos_stheta = np.cos(stheta)
    sin_sphi = np.sin(sphi)
    cos_sphi = np.cos(sphi)
    
    # Calculation on incoming coordinates 
    if From == 's3c':
        first = Identity
        
    if From == 'jso':
        # Calculate the rotation matrix to go from JSO to S3R
        # Define X component in system III of XJSO unit vector etc.
        dummy[:,0]=[cos_stheta*cos_sphi, cos_stheta*sin_sphi, sin_stheta]
        # Calculate the Z axis of the JSO coordinate system from the fact that the Z axis is tilted
        # by 3.12 degrees from the Z axis of SIII and is normal to the X axis of the JSO system.
        # Define X component in system III of ZJSO unit vector etc.
        dummy[:,2] = [sin3*np.cos(phase), sin3*np.sin(phase), cos3]
        #Define X component in system III of YJSO unit vector etc.
        dummy[0,1] = dummy[1,2]*dummy[2,0] - dummy[2,2]*dummy[1,0]
        dummy[1,1] = dummy[2,2]*dummy[0,0] - dummy[0,2]*dummy[2,0]
        dummy[2,1] = dummy[0,2]*dummy[1,0] - dummy[1,2]*dummy[0,0]
        
        first = np.transpose(dummy)
    
    if From == 'jsm':
        # Now define the JSM transpose vector
        # Define X component in system III of XJSM unit vector etc.
        dummy[0,0] = cos_stheta*cos_sphi
        dummy[1,0] = cos_stheta*sin_sphi
        dummy[2,0] = sin_stheta
        # Now define the Y vector so that it is perpendicular to the dipole vector and X
        dummy[0,1] = dummy[2,0]*dipole[1,2] - dummy[1,0]*dipole[2,2]
        dummy[1,1] = dummy[0,0]*dipole[2,2] - dummy[2,0]*dipole[0,2]
        dummy[2,1] = dummy[1,0]*dipole[0,2] - dummy[0,0]*dipole[1,2]
        denom = np.sqrt(dummy[0,1]*dummy[0,1] + dummy[1,1]*dummy[1,1] + dummy[2,1]*dummy[2,1])
        dummy[0,1] = dummy[0,1]/denom
        dummy[1,1] = dummy[1,1]/denom
        dummy[2,1] = dummy[2,1]/denom
        # Now define the z vector
        dummy[0,2] = dummy[1,0]*dummy[2,1] - dummy[2,0]*dummy[1,1]
        dummy[1,2] = dummy[2,0]*dummy[0,1] - dummy[0,0]*dummy[2,1]
        dummy[2,2] = dummy[0,0]*dummy[1,1] - dummy[1,0]*dummy[0,1]
        first = np.transpose(dummy)
        
    if From == 'dip' : 
        first = np.transpose(dipole)
        
    if From == 'jss' :
        # Calculate the rotation matrix to go from JSS to S3R
        # Define X component in system III of ZJSS unit vector etc.
    
        dummy[0,2] = 0
        dummy[1,2] = 0
        dummy[2,2] = 1
        # Define X component in system III of YJSS unit vector etc.
        # so that it is perpendicular to the system 3 Z spin axis and the 
        # Jupiter-Sun line
        dummy[0,1] = -cos_stheta*sin_sphi
        dummy[1,1] = cos_stheta*cos_sphi
        dummy[2,1] = 0
        denom = np.sqrt(dummy[0,1]*dummy[0,1] + dummy[1,1]*dummy[1,1])
        dummy[0,1] = dummy[0,1]/denom
        dummy[1,1] = dummy[1,1]/denom
        dummy[2,1] = dummy[2,1]/denom
        # Define X component in system III of XJSS unit vector etc.
        # so that it is perpendicular to ZJSS and YJSS
        dummy[0,0] = dummy[1,1]
        dummy[1,0] = -dummy[0,1]
        dummy[2,0] = 0       
        first =  np.transpose(dummy)
        
        
        
    # Calculation on outgoing coordinates 
    if To == 's3c':
        second = Identity
        
    if To == 'jso':
        # Get the matrix that Rotates from System III cartesian into JSO.	
        # Define X component in system III of XJSO unit vector etc.
        second[0,0] = cos_stheta*cos_sphi
        second[1,0] = cos_stheta*sin_sphi
        second[2,0] = sin_stheta
        # Calculate the Z axis of the JSO coordinate system from the fact that the Z axis is tilted
        # by 3.12 degrees from the Z axis of SIII and is normal to the X axis of the JSO system.
        #  Define X component in system III of ZJSO unit vector etc.
        second[0,2] = sin3*np.cos(phase)
        second[1,2] = sin3*np.sin(phase)
        second[2,2] = cos3
        # Define X component in system III of YJSO unit vector etc.
        second[0,1] = second[1,2]*second[2,0] - second[2,2]*second[1,0]
        second[1,1] = second[2,2]*second[0,0] - second[0,2]*second[2,0]
        second[2,1] = second[0,2]*second[1,0] - second[1,2]*second[0,0]   
        
    if To =='jsm':
        # Get the matrix that Rotates from System III cartesian into JSM.	
        # Define X component in system III of XJSM unit vector etc.
        second[0,0] = cos_stheta*cos_sphi
        second[1,0] = cos_stheta*sin_sphi
        second[2,0] = sin_stheta
        # Now define the Y vector so that it is perpendicular to  the dipole vector and X 
        second[0,1] = second[2,0]*dipole[1,2] - second[1,0]*dipole[2,2]
        second[1,1] = second[0,0]*dipole[2,2] - second[2,0]*dipole[0,2]
        second[2,1] = second[1,0]*dipole[0,2] - second[0,0]*dipole[1,2]
        denom = np.sqrt(second[0,1]*second[0,1] + second[1,1]*second[1,1] + second[2,1]*second[2,1])
        second[0,1] = second[0,1]/denom
        second[1,1] = second[1,1]/denom
        second[2,1] = second[2,1]/denom
        # Now define the z vector
        second[0,2] = second[1,0]*second[2,1] - second[2,0]*second[1,1]
        second[1,2] = second[2,0]*second[0,1] - second[0,0]*second[2,1]
        second[2,2] = second[0,0]*second[1,1] - second[1,0]*second[0,1]   
        
    if To == 'dip':
        second = np.transpose(dipole)
        
    if To == 'jss':
        # Define X component in system III of ZJSS unit vector etc.
        second[0,2] = 0
        second[1,2] = 0
        second[2,2] = 1
        # Define X component in system III of YJSS unit vector etc.
        # so that it is perpendicular to the system 3 Z spin axis and the 
        # Jupiter-Sun line
        second[0,1] = -cos_stheta*sin_sphi
        second[1,1] = cos_stheta*cos_sphi
        second[2,1] = 0
        denom = np.sqrt(second[0,1]*second[0,1] + second[1,1]*second[1,1] + second[2,1]*second[2,1])
        second[0,1] = second[0,1]/denom
        second[1,1] = second[1,1]/denom
        second[2,1] = second[2,1]/denom
        # Define X component in system III of XJSS unit vector etc.
        # so that it is perpendicular to ZJSS and YJSS
        second[0,0] = second[1,1]*second[2,2] - second[2,1]*second[1,2]
        second[1,0] = second[2,1]*second[0,2] - second[0,1]*second[2,2]
        second[2,0] = second[0,1]*second[1,2] - second[1,1]*second[0,2]
    
    # Now multimply vecin with first and second matrices to get the vecout
    vector = vecin @ first 
    vecout = vector @ second   
 
        
    return vecout   

def cyl2car_pos(rho,phi,z):
    # ----------------------------------------------------------- # 
    # Translated from procedure cyl2car_pos 
    #
    # INPUT:  - rho, phi [rad], z (Cylindrical coordinates)
    # OUTPUT: - x, y, z (Cartesian coordinates)
    # ----------------------------------------------------------- # 
    Xcar = rho*np.cos(phi)
    Ycar = rho*np.sin(phi)
    Zcar = z  
    return Xcar, Ycar, Zcar 

def car2cyl_pos(x,y,z):
    # ----------------------------------------------------------- # 
    # Translated from procedure car2cyl_pos
    #
    # INPUT:  - x, y, z (Cartesian coordinates)
    # OUTPUT: - rho, phi [rad], z (Cylindrical coordinates)
    # ----------------------------------------------------------- # 
    phi = np.arctan2(y,x)
    rho = np.sqrt(x*x + y*y)
    Z = z
    return rho, phi, Z 

def car2cyl_vect(phi, Bx, By, Bz):
    # ----------------------------------------------------------- # 
    # Translated from procedure car2cyl_vect
    #
    # INPUT:  - phi [Rad], Bx, By, Bz (B in cartesian coordinates)
    # OUTPUT: - Brho, Bphi, Bz (B in cylindrical coordinates)
    # ----------------------------------------------------------- # 
    cos_phi = np.cos(phi)
    sin_phi = np.sin(phi)
    Brho = Bx*cos_phi + By*sin_phi
    Bphi = -Bx*sin_phi + By*cos_phi
    BZ = Bz
    return Brho, Bphi, BZ  

def cyl2car_vect(phi, Brho, Bphi, Bz):
    # ----------------------------------------------------------- # 
    # Translated from procedure cy12car_vect
    #
    # INPUT:  - phi [Rad], Brho, Bphi, Bz (B in cylindral coordinates)
    # OUTPUT: - Bx, By, Bz (B in cartesian coordinates)
    # ----------------------------------------------------------- # 
    cos_phi = np.cos(phi)
    sin_phi = np.sin(phi)
    Bx = Brho*cos_phi - Bphi*sin_phi
    By = Brho*sin_phi + Bphi*cos_phi
    BZ = Bz
    return Bx, By, BZ

def CAR2SPH_pos(x,y,z):
    # ----------------------------------------------------------- # 
    # Translated from procedure CAR2SPH_pos
    #
    # INPUT:    - x,y,z [Rj,Rj,Rj] 
    # OUTPUT:   - r,theta,phi [Rj,rad,rad]        
    # ----------------------------------------------------------- # 
    r = np.sqrt(x*x + y*y + z*z)
    phi = np.arctan2(y,x)
    th = np.arccos(z/r) # ERROR IN KK_2009
    
    return r, th, phi 

def SPH2CAR_pos(r,th,phi):
    # ----------------------------------------------------------- # 
    # Translated from procedure SPH2CAR_pos
    # 
    # INPUT:    - r,theta,phi [Rj,rad,rad] 
    # OUTPUT:   - x,y,z [Rj,Rj,Rj] 
    # ----------------------------------------------------------- # 
    sin_th = np.sin(th)
    x = r*sin_th*np.cos(phi)
    y = r*sin_th*np.sin(phi)
    z = r*np.cos(th)
    
    return x, y, z 

def SPH2CAR_MAG(th,phi,Br,Bth,Bphi):
    # ----------------------------------------------------------- # 
    # Translated from procedure SPH2CAR_MAG
    #
    # INPUT:    - theta,phi, Br,Bt,Bp [Rj,rad,rad,nT,nT,nT] 
    # OUTPUT:   - Bx,By,Bz [nT,nT,nT] 
    # ----------------------------------------------------------- # 
    sin_th = np.sin(th)
    cos_th = np.cos(th)
    sin_phi = np.sin(phi)
    cos_phi = np.cos(phi)
    Bx = Br*sin_th*cos_phi + Bth*cos_th*cos_phi - Bphi*sin_phi
    By = Br*sin_th*sin_phi + Bth*cos_th*sin_phi + Bphi*cos_phi
    Bz = Br*cos_th - Bth*sin_th
    return Bx, By, Bz  

def CAR2SPH_MAG(th,phi,Bx,By,Bz):
    # ----------------------------------------------------------- # 
    # Translated from procedure CAR2SPH_MAG
    #
    # INPUT:    - theta,phi, Bx,By,Bz [Rj,rad,rad,nT,nT,nT] 
    # OUTPUT:   - Br,Bt,Bp [nT,nT,nT] 
    # ----------------------------------------------------------- # 
    sin_th = np.sin(th)
    cos_th = np.cos(th)
    sin_phi = np.sin(phi)
    cos_phi = np.cos(phi)
    Br = Bx*sin_th*cos_phi + By*sin_th*sin_phi + Bz*cos_th
    Bth = Bx*cos_th*cos_phi + By*cos_th*sin_phi - Bz*sin_th
    Bphi = -Bx*sin_phi + By*cos_phi

    return Br, Bth, Bphi   

def PJSO2S3(phi, Bxjso, Byjso, Bzjso):
    # ----------------------------------------------------------- # 
    # Translated from procedure PJSO2S3
    #
    # INPUT:    - phi, Bx,By,Bz (B cartesian, JSO reference frame)
    # OUTPUT:   - Bx,By,Bz (B cartesian, S3RH reference frame)
    # ----------------------------------------------------------- # 
    sin_phi = np.sin(phi)
    cos_phi = np.cos(phi)
    Bxs3 = Bxjso*cos_phi - Byjso*sin_phi
    Bys3 = Bxjso*sin_phi + Byjso*cos_phi
    Bzs3 = Bzjso

    return Bxs3, Bys3, Bzs3

def S32PJSO(phi, Bxs3, Bys3, Bzs3):
    # ----------------------------------------------------------- # 
    # Translated from procedure S32PJSO
    #
    # INPUT:    - phi, Bx,By,Bz (B cartesian, S3RH reference frame)
    # OUTPUT:   - Bx,By,Bz (B cartesian, JSO reference frame)
    # ----------------------------------------------------------- # 
    sin_phi = np.sin(phi)
    cos_phi = np.cos(phi)
    Bxjso = Bxs3*cos_phi + Bys3*sin_phi
    Byjso = -Bxs3*sin_phi + Bys3*cos_phi
    Bzjso = Bzs3

    return Bxjso, Byjso, Bzjso 


###################### BESSEL FUNCTIONS #######################################

def bessj1(x_in):
    # ----------------------------------------------------------- # 
    # Translated from function bessj1
    # ----------------------------------------------------------- # 
    x = np.array([x_in])
    a = np.abs([min(x),max(x)])
    if a[0]>a[1]:
        b = np.array(np[a[1],a[0]])
    else : 
        b = a 
    # everything here is between -8 and 8  
    if b[1]<8:
        y = x*x 
        bessj1 = x*(72362614232 + y*(-7895059235 + y*(242396853.1 + y*(-2972611.439 + y*(15704.48260 + y*(-30.160366606))))))/(144725228442 + y*(2300535178 + y*(18583304.74 + y*(99447.43394 + y*(376.9991397 + y*1.0)))))
        
        return bessj1[0] 

    # everything is either < or = to -8 or > or = to +8
    if (b[0] >= 8) : 
        ax = abs(x)
        z = 8/ax
        y = z*z
        xx = ax - 2.356194491
        if x > 0 :
            one = 1
        else :
            one = -1
        
        bessj1 = np.sqrt(0.636619772/ax)*(np.cos(xx)*(1.0 + y*(0.183105e-2 + y*(-0.3516396496e-4 + y*( 0.2457520174e-5 + y*(-0.240337019e-6) ))))-z*np.sin(xx)*(0.04687499995 + y*(-0.88228987e-6 + y*(0.105787412e-6 + y*(-0.88228987e-6 + y*0.105787412e-6)))))*one
        return bessj1[0] 

def bessj1_multi(x_in):
    # ----------------------------------------------------------- # 
    # New function : 
    #   allows to use bessj1(x) with arrays as input
    # ----------------------------------------------------------- # 
    ans = []
    for i in range(len(x_in)):
        ans.append(bessj1(x_in[i]))
    return np.array(ans)

def bessj0(x_in):
    # ----------------------------------------------------------- # 
    # Translated from function bessj0
    # ----------------------------------------------------------- # 
    x = np.array([x_in])
    a = np.abs([min(x), max(x)])
    if (a[0] > a[1]) :
        b = np.array([a[1],a[0]])     
    else:
        b = a
    # everything here is between -8 and 8
    if (b[1] < 8):
        y = x*x
        bessj0 = (57568490574 + y*(-13362590354 + y*(651619640.7 + y*(-11214424.18 + y*( 77392.33017 - 184.9052456*y)))))/(57568490411 + y*(1029532985 + y*(9494680.718 + y*(59272.64853 + y*(267.8532712 + y*1.0)))))
    
        return bessj0[0]    
        
     # everything is either < or = to -8 or > or = to +8
    if (b[0] > 8) :
        ax = abs(x)
        z = 8/ax
        y = z*z
        xx = ax - 0.785398164
        bessj0 = np.sqrt(0.636619772/ax)*(np.cos(xx)*(1.0 + y*(-0.1098628627e-2 + y*(.2734510407e-4 + y*(-0.2073370639e-5 + y*0.2093887211e-6))))- z*np.sin(xx)*(-0.1562499995e-1 + y*(0.1430488765e-3 + y*(-0.6911147651e-5 + y*(0.7621095161e-6 - 0.934945152e-7*y)))))
        return bessj0[0]   
 
    return 

def bessj0_multi(x_in):
    # ----------------------------------------------------------- # 
    # New function : 
    #   allows to use bessj0(x) with arrays as input
    # ----------------------------------------------------------- # 
    ans = []
    for i in range(len(x_in)):
        ans.append(bessj0(x_in[i]))
    return np.array(ans)

def DBSJ2(x):
    # ----------------------------------------------------------- # 
    # Translated from function DBSJ2
    # ----------------------------------------------------------- # 
    DBSJ2 = (2/x)*bessj1(x) - bessj0(x)
    return DBSJ2  

def DBSJ2_multi(x_in):
    # ----------------------------------------------------------- # 
    # New function : 
    #   allows to use DBSJ2(x) with arrays as input
    # ----------------------------------------------------------- # 
    ans = []
    for i in range(len(x_in)):
        ans.append(DBSJ2(x_in[i]))
    return np.array(ans) 



def DBSJ3(x):
    # ----------------------------------------------------------- # 
    # Translated from function DBSJ3
    # ----------------------------------------------------------- # 
    DBSJ3 = (4/x)*DBSJ2(x) - bessj1(x)
    return DBSJ3   

def DBSJ3_multi(x_in):
    # ----------------------------------------------------------- # 
    # New function : 
    #   allows to use DBSJ3(x) with arrays as input
    # ----------------------------------------------------------- # 
    ans = []
    for i in range(len(x_in)):
        ans.append(DBSJ3(x_in[i]))
    return np.array(ans)


################ MAGNETIC FIELDS AND CURRENTS MEASUREMENTS ####################


def B_mp_perp(rho, phi, x, Nmodes):
    # ----------------------------------------------------------- # 
    # Translated from procedure B_mp_perp
    # ----------------------------------------------------------- # 
    A = np.array([14.8636286417325, -20.6733061430101, -68.9799158183312, -1.42668213471797e-006, 
    -2.10151726837109e-004, 7.79886406933501e-006, 1.68914984127525e-008, -1.07213724493682e-007, 
    5.22885344671751e-007, -4.46205598371400e-008, 2.83331673403534e-007, -1.38338264037040e-006, 
    7.94736850421990, 23.4906454129448, -17.3725909963207, -3.95338621079697e-004, 
    -4.24194742905271e-003, 1.92089873646750e-003, -6.94191156816239e-002, -0.170169494909972, 
    -0.171632981837208, 4.57144670517197e-009,  6.53500870745233e-007, 6.56440399411187e-007, 
    -7.67358819268301e-002, -0.133394608915796, -0.127893292811889, 3.60644094744725e-006 ,  
    2.04677410835679e-005, 3.26595444981956e-005])
    
    B = np.array([32.9198722839323, 64.1556701660173, 128.854949951183, 30.4416656494141, 
    45.2417564392090, 107.303962707520, 28.7235050200276, 63.9741210937499, 
    128.002441406250, 33.8896331786845, 60.2880744933923, 119.902839660605, 
    32.1036300655818, 62.8152580261215, 127.073867797851])
    
    Modes1 = Nmodes
    Modes2 = 2*Modes1
    Modes3 = 3*Modes1
    Modes4 = 4*Modes1
    
    """
    # Not used : 
    Modes5 = 5*Modes1
    """
    
    rhom = rho - 0.1
    rhop = rho + 0.1
    
    phi2 = 2*phi
    phi3 = 3*phi
    
    xm = x - 0.1
    xp = x + 0.1
    
    sinf = np.sin(phi)
    cosf = np.cos(phi)
    sin2f = 2*sinf*cosf 
    cos2f = 2*cosf*cosf - 1
    sin3f = np.sin(phi3)
    cos3f = np.cos(phi3)
    
    expb = np.exp(x/B)
    exp_xm_minus_xp = np.exp(xp/B) - np.exp(xm/B)
    
    sin_001 = np.sin(0.001)
    sinP_minus_sinM = 2*np.cos(phi)*sin_001
    cosP_minus_cosM = -2*np.sin(phi)*sin_001
    sin2P_minus_sin2M = 2*np.cos(phi2)*sin_001
    cos2P_minus_cos2M = -2*np.sin(phi2)*sin_001
    sin3P_minus_sin3M = 2*np.cos(phi3)*sin_001
    cos3P_minus_cos3M = -2*np.sin(phi3)*sin_001
    
    rhobb = rho/B
    xmb = (x - B)
    fiveHundredoverRho = (500/rho)
    
    Bperpr = 0
    Bperpf = 0
    Bperpx = 0
    KA = 0
        
    # --------------------------------------------------------------------------------------
    # Now compute the terms associated with phi
    Nmodes_indgen = []
    for i in range(Nmodes):
        Nmodes_indgen.append(i)
    
    Nmodes_indgen = np.array(Nmodes_indgen)
    Nmodes_x2 = 2*Nmodes
    ind = Nmodes_indgen
    
    KA1     = KA + Nmodes_indgen
    KA_Temp = ind + Nmodes


    bessj1PminusM = bessj1_multi(rhop/B[ind]) - bessj1_multi(rhom/B[ind])
    bessj1RHO     = bessj1_multi(rhobb[ind])
    angle_diff    = sinf*A[KA1] + cosf*A[KA_Temp]
    
    Bperpr += expb[ind] * bessj1PminusM  *angle_diff
    Bperpf += expb[ind]*bessj1RHO * (sinP_minus_sinM*A[KA1] + cosP_minus_cosM*A[KA_Temp])
    Bperpx += bessj1RHO*exp_xm_minus_xp[ind]*angle_diff
    KA += 2*Nmodes
    
    # Now compute the terms associated with 2*phi
    ind  = Nmodes_indgen + Nmodes
    KA1     = KA + Nmodes_indgen
    KA_Temp = KA1 + Nmodes
    dbsj2PminusM = DBSJ2_multi(rhop/B[ind]) - DBSJ2_multi(rhom/B[ind])
    dbsj2Rhobb   = DBSJ2_multi(rhobb[ind])
    angle_diff = sin2f*A[KA1] + cos2f*A[KA_Temp]
    
    Bperpr += expb[ind] * dbsj2PminusM * angle_diff
    Bperpf += expb[ind] * dbsj2Rhobb *(sin2P_minus_sin2M*A[KA1] + cos2P_minus_cos2M*A[KA_Temp])
    Bperpx += dbsj2Rhobb*exp_xm_minus_xp[ind]*angle_diff
    KA += Nmodes_x2
    
    
    # Now compute the terms associated with 3*phi  
    ind     = Nmodes_indgen + Modes2
    KA1     = KA + Nmodes_indgen
    KA_Temp = KA1 + Nmodes
    dbsj3PminusM = DBSJ3_multi(rhop/B[ind]) - DBSJ3_multi(rhom/B[ind])
    dbsj3Rhobb   = DBSJ3_multi(rhobb[ind])
    angle_diff = sin3f*A[KA1] + cos3f*A[KA_Temp]

    Bperpr += expb[ind] * dbsj3PminusM * angle_diff
    Bperpf += expb[ind] * dbsj3Rhobb *(sin3P_minus_sin3M * A[KA1] + cos3P_minus_cos3M * A[KA_Temp])
    Bperpx += dbsj3Rhobb * exp_xm_minus_xp[ind] * angle_diff
    KA += Nmodes_x2
    
    
    # Now include the terms associated with the derivative term
    ind     = Nmodes_indgen + Modes3
    KA1     = KA + Nmodes_indgen
    KA_Temp = KA1 + Nmodes
    besj0 = bessj0_multi(rhobb[ind])
    besj1 = bessj1_multi(rhobb[ind])
    BJ0minusBJ1 = rhop*bessj0_multi(rhop/B[ind]) - rhom*bessj0_multi(rhom/B[ind]) - xmb[ind]*(bessj1_multi(rhom/B[ind]) - bessj1_multi(rhop/B[ind]))
    expxp_minus_expxm = np.exp(xp/B[ind])*(rho*besj0 + (xp - B[ind])*besj1) - np.exp(xm/B[ind])*(rho*besj0 + (xm - B[ind])*besj1)
    BJ0plusBJ1        = rho*besj0 + xmb[ind]*besj1
    angle_diff = sinf*A[KA1] + cosf*A[KA_Temp]

    Bperpr += expb[ind] * BJ0minusBJ1 *angle_diff
    Bperpf += expb[ind] *BJ0plusBJ1 *(sinP_minus_sinM*A[KA1] + cosP_minus_cosM*A[KA_Temp])
    Bperpx += expxp_minus_expxm* angle_diff
    KA += Nmodes_x2

    ind = Nmodes_indgen + Modes4
    KA1  = KA + Nmodes_indgen
    KA_Temp = KA1 + Nmodes
    besj2 = DBSJ2_multi(rhobb[ind])
    besj3 = DBSJ3_multi(rhobb[ind])
    dbsj2minusdbsj3  = rhop*DBSJ2_multi(rhop/B[ind]) - rhom*DBSJ2_multi(rhom/B[ind]) - (x - 3*B[ind])*(DBSJ3_multi(rhom/B[ind]) - DBSJ3_multi(rhop/B[ind]))
    expxp_minus_expxm = np.exp(xp/B[ind])*(rho*besj2 + (xp - 3*B[ind])*besj3) - np.exp(xm/B[ind])*(rho*besj2 + (xm - 3*B[ind])*besj3)
    BJ2minusBJ3 = rho*besj2 + (x - 3*B[ind])*besj3
    angle_diff = sin3f*A[KA1] + cos3f*A[KA_Temp]
    
    Bperpr += expb[ind] * dbsj2minusdbsj3 * angle_diff
    Bperpf += expb[ind] * BJ2minusBJ3 *(sin3P_minus_sin3M*A[KA1] + cos3P_minus_cos3M*A[KA_Temp])
    Bperpx += expxp_minus_expxm*angle_diff
    
    # --------------------------------------------------------------------------------------

    Bperpr = sum(Bperpr) * 5
    Bperpf = sum(Bperpf) * fiveHundredoverRho
    Bperpx = sum(Bperpx) * 5


    return Bperpr, Bperpf, Bperpx

def dipole(x,y,z,B0x,B0y,B0z): 
    # ----------------------------------------------------------- # 
    # Translated from procedure dipole 
    # 
    # Calculates the field of Jupiter's dipole for shielding in the magnetopause
    # 
    # INPUT:  - B0x, B0y, B0z  (the dipole moment)
    #         - x,y,z (position vector)
    # OUTPUT: - Bx,By,Bz (output field vector)
    # ----------------------------------------------------------- #
    r = np.sqrt(x*x + y*y + z*z)
    a = np.array([[3*x*x - r*r, 3*x*y, 3*x*z], 
                  [3*x*y, 3*y*y - r*r, 3*y*z], 
                  [3*x*z, 3*y*z, 3*z*z - r*r]])
    r5 = r*r*r*r*r
    a = a/r5
    Bx = a[0,0]*B0x + a[0,1]*B0y + a[0,2]*B0z
    By = a[1,0]*B0x + a[1,1]*B0y + a[1,2]*B0z
    Bz = a[2,0]*B0x + a[2,1]*B0y + a[2,2]*B0z
    
    return Bx, By, Bz 

def dipole_shielded(PARMOD,x,y,z):
    # ----------------------------------------------------------- # 
    # Translated from procedure dipole_shielded
    # WRITTEN BY K. K. KHURANA   11/2002
    # MODIFIED BY H. K. SCHWARZL 11/2003
    # PARMOD is an input array (real*8) that contains the model parameters
    # Dimension of PARMOD is: DIMENSION PARMOD(10)
    # currently just PARMOD[0] is used for the dipole tilt angle
    #
    # INPUT:  - x,y,z position
    # OUTPUT: - Bx,By,Bz  magnetic field component
    # ----------------------------------------------------------- # 
    psir = PARMOD[0]
    #B0 = 426411.1411689427332021 #VIP4 dipole magnitude
    B0 = np.sqrt(410993.4*410993.4 + 71305.9*71305.9 + 20958.4*20958.4) #JRM33 dipole magnitude
    B0x = B0*np.sin(psir)
    B0y = 0
    B0z = B0*np.cos(psir)
    
    # We will  first calculate the field of the dipole
    Bxd, Byd, Bzd = dipole(x,y,z, B0x, B0y, B0z)

    # Now calculate the parallel dipole shielding field
    if (z == 0) and (y == 0):
        cos_phi = 1
        sin_phi = 0
    else :
        phi = np.arctan2(z,y)
        cos_phi = np.cos(phi)
        sin_phi = np.sin(phi)
    
    rho = y*cos_phi + z*sin_phi
    
    # Number of dipole modes
    Nmodes = 3
    
    # Call B_mp_par(rho,phi,x,Brho1,Bphi1,Bx1,Nmodes) !no par dipole anymore
    #Brho1, Bphi1, Bx1 = 0,0,0
    Brho2, Bphi2, Bx2 = B_mp_perp(rho, phi, x, Nmodes)
    
    By2 = Brho2*cos_phi - Bphi2*sin_phi
    Bz2 = Brho2*sin_phi + Bphi2*cos_phi

    Bx = Bxd + Bx2
    By = Byd + By2
    Bz = Bzd + Bz2
    
    return Bx, By, Bz 

def dipole_shiel_cyl_S3(rmap, pmap, zmap, PARMOD, sphiOut):
    # ----------------------------------------------------------- # 
    # Translated from procedure dipole_shield_cyl_S3
    # ----------------------------------------------------------- # 
    Xcar, Ycar, Zcar = cyl2car_pos(rmap, pmap, zmap)
    
    xpJSO, ypJSO, zpJSO = S32PJSO(sphiOut, Xcar, Ycar, Zcar)
    
    BxpJSO, BypJSO, BzpJSO = dipole_shielded(PARMOD, xpJSO, ypJSO, zpJSO)
    
    BxS3, ByS3, BzS3 = PJSO2S3(sphiOut, BxpJSO, BypJSO, BzpJSO)
    
    Brds, Bpds, Bzds = car2cyl_vect(pmap, BxS3, ByS3, BzS3)
    
    return Brds, Bpds, Bzds

def U_B(x1,y1,z1,a,c,p_1,p_2,r_1,r_2):
    # ----------------------------------------------------------- # 
    # Translated from function U_B
    # ----------------------------------------------------------- # 
    Up = [0, 0, 0]
    Um = [0, 0, 0]
    B = [0, 0, 0]
    
    xp = [x1 + 0.001, x1, x1]
    yp = [y1, y1 + 0.001, y1]
    zp = [z1, z1, z1 + 0.001]
    xm = [x1 - 0.001, x1, x1]
    ym = [y1, y1 - 0.001, y1]
    zm = [z1, z1, z1 - 0.001]
    
    
    term_p = np.empty((8,8))
    term_r = np.empty((8,8))
    for i in range(7):
        term_p[0:8,i] = 1/(p_1[i]*p_1[i]) + 1/(p_2[0:8]*p_2[0:8])
        term_r[0:8,i] = 1/(r_1[i]*r_1[i]) + 1/(r_2[0:8]*r_2[0:8])
   
    term_p = np.sqrt(term_p)
    term_r = np.sqrt(term_r)
    
    for i in range(8):
        cos_yp = np.cos(yp/p_1[i])
        cos_ym = np.cos(ym/p_1[i])
        sin_yp = np.sin(yp/r_1[i])
        sin_ym = np.sin(ym/r_1[i])    
        for k in range(8):
            Up = Up + a[k,i]*np.exp(term_p[k,i]*xp)*cos_yp*np.sin(zp/p_2[k]) + c[k,i]*np.exp(term_r[k,i]*xp)*sin_yp*np.sin(zp/r_2[k])    
            Um = Um + a[k,i]*np.exp(term_p[k,i]*xm)*cos_ym*np.sin(zm/p_2[k]) + c[k,i]*np.exp(term_r[k,i]*xm)*sin_ym*np.sin(zm/r_2[k])
        
    B = Up - Um    
        
    return B 

def B_tail_shield(x,y,z,M,ModeIn):
    # ----------------------------------------------------------- # 
    # Translated from procedure B_tail_shield 
    # ----------------------------------------------------------- # 
    aMode = np.zeros((8,8,6))
    cMode = np.zeros((8,8,6))
    
    p = np.zeros((16,8))
    r = np.zeros((16,8))
    
    # Coefficient a 
    aMode[:,0,0] = [0.13217647646022754326e-20, 0.50643011679165539362e-16, -0.62311994734606299672e-13, 0.73000393989897114366e-10, 
    -0.10249726242696675093e-07, 0.32914667786334130816e-06, -0.24800841788149776689e-05, 0.37244082145067776146e-05]
    aMode[:,1,0] = [0.22584041437827493403e-17, 0.20229608874904410065e-12, -0.13496084238172885161e-09, 0.19512264195191708182e-06, 
    -0.28536292479920302156e-04, 0.92636894313957363067e-03, -0.69993880953881300044e-02, 0.10518459214365802889e-01]
    aMode[:,2,0] = [0.36523294131124144357e-14, -0.64420435736454777497e-09, -0.57096883444695674114e-07, -0.27055379124230958254e-04, 
    0.44272407289722757184e-02, -0.14889477801070045259e+00, 0.11356290643917379412e+01, -0.17106683807405858033e+01]
    aMode[:,3,0] = [-0.18416420070040647516e-11, 0.27046079272174843310e-06, 0.44203411790099522704e-04, 0.12944858989375875779e-03, 
    -0.51975317855123419619e-02, 0.20369979936316231494e+00, -0.16254799676715984801e+01, 0.24787158145629919481e+01]
    aMode[:,4,0] = [0.13636392185761578854e-09, -0.20156479561298437097e-04, -0.39633156929923650579e-02, -0.13365044136678470021e-01, 
    -0.10442458756051709034e-01, 0.11380270847337552453e+00, -0.88061814759885734815e+00, 0.13114182967621741404e+01]
    aMode[:,5,0] = [-0.22305468438407642928e-08, 0.33146982079580284974e-03, 0.69006040956576439882e-01, 0.27321338618451562751e+00, 
    0.15256440935574380191e+00, 0.82160642450780390078e-01, -0.38978676634608163453e+00, 0.47434195602775641731e+00]
    aMode[:,6,0] = [0.84447121731127730015e-08, -0.12569305425012970989e-02, -0.26569849211645015785e+00, -0.11069710175423441711e+01, 
    -0.64840056779980832502e+00, -0.97562574536155111104e-02, 0.15631498731831953818e+00, -0.16203940584912954747e+00]
    aMode[:,7,0] = [-0.63486912141303140089e-08, 0.94534756045114320954e-03, 0.20061320014222592256e+00, 0.84751706801178663397e+00, 
    0.50107973765980506897e+00, -0.14177955455321484379e-03, 0.56476656376811078530e-01, -0.52516865628955500255e-02]
    
    aMode[:,0,1] = [0.13619937133704040910e-23, 0.43621985467324879692e-18, -0.15098830327683961272e-14, 0.63123957003196471404e-11, 
    -0.11978885144275235319e-08, 0.41934992058305047279e-07, -0.31951294540338555094e-06, 0.48135741458667444803e-06]
    aMode[:,1,1] = [0.24119037788698203250e-20, 0.40881122046673903369e-14, -0.21545235983615116381e-11, 0.11640656169328844615e-07, 
    -0.21959806946493167778e-05, 0.76662481447507042631e-04, -0.58366072779967188566e-03, 0.87913032319456423380e-03]
    aMode[:,2,1] = [0.31144339871300497080e-16, -0.22620785858629064435e-10, -0.74029445112059493183e-08, -0.93859788335645291112e-07, 
    0.82317630609657417295e-04, -0.36569184498810431982e-02, 0.29432171359264206245e-01, -0.44946922231727173269e-01]
    aMode[:,3,1] = [-0.58080626963236774429e-14, 0.39682408099390080735e-08, 0.16840660739515575627e-05, 0.38335076286793983157e-05, 
    0.20836381731764990199e-03, -0.33920492396785180133e-02, 0.15820849562165701485e-01, -0.19644945238949279797e-01]
    aMode[:,4,1] = [0.41076898294619867968e-12, -0.27989015436962660920e-06, -0.13840368413368968703e-03, -0.70869170102476930495e-03, 
    0.26033168308504186505e-03, -0.24526926326940952094e-02, -0.35013567157541127805e-01, 0.69275925576020052076e-01]
    aMode[:,5,1] = [-0.70799772156343383500e-11, 0.48274807820722331896e-05, 0.24974637159216332982e-02, 0.15464887864241698700e-01, 
    0.11439264646130791192e-01, 0.38097770315818785036e-01, -0.13590064465091589163e+00, 0.17180997533186371128e+00]
    aMode[:,6,1] = [0.26783558161963370025e-10, -0.18266964017400193043e-04, -0.95586630597219102156e-02, -0.62918902435418315732e-01, 
    -0.64039968185764291064e-01, -0.22222194307812768165e-03, 0.50813426126672718297e-01, -0.46991112128761134414e-01]
    aMode[:,7,1] = [-0.20108572992030198101e-10, 0.13715428126943252085e-04, 0.71979427421967230316e-02, 0.48199320155821636646e-01, 
    0.49407199622570425745e-01, 0.92225425694026483824e-02, 0.53857653772015723347e-02, 0.91367943357959795491e-03]
    
    aMode[:,0,2] = [-0.13119415239474609968e-23, -0.60256122420913360571e-18, 0.37920759812307474057e-14, -0.18206762505356291370e-10, 
    0.26774304351790334521e-08, -0.87684408227198336049e-07, 0.66457031757235851543e-06, -0.99948895864718618753e-06]
    aMode[:,1,2] = [-0.25960636282069127211e-19, -0.42000012814022671392e-16, 0.74796457397241065123e-11, -0.25318674869395683124e-07, 
    0.34976874196721219334e-05, -0.11242351418125537954e-03, 0.84791358984972013956e-03, -0.12736554117114038398e-02]
    aMode[:,2,2] = [0.76343432773888704190e-16, 0.64378927275727946266e-12, 0.16598929643645348619e-08, -0.92837331289370776943e-05, 
    0.14285390095735186477e-02, -0.47503134695174127344e-01, 0.36153292729534660665e+00, -0.54431063875700962384e+00]
    aMode[:,3,2] = [-0.25391252497671850107e-13, -0.31368629291300997863e-09, 0.14987071996970728449e-05, 0.17824032262389087222e-04, 
    0.11355427076600370650e-02, -0.37610982091768150326e-01, 0.28513697308043282063e+00, -0.42869687938445020236e+00]
    aMode[:,4,2] = [0.17211923850683952252e-11, 0.31101556003157209140e-07, -0.12650545583107459801e-03, -0.30769692479640595728e-02, 
    -0.45994232482610790668e-02, 0.15393025113972294448e-01, 0.86726269138905074385e-02, -0.41633915235090732664e-01]
    aMode[:,5,2] = [-0.27638896781422706006e-10, -0.55187436783116812222e-06, 0.21199958794900246594e-02, 0.61391435739846613728e-01, 
    0.52329651593674135767e-01, -0.83734155027368615265e-01, 0.12671335310678004670e+00, -0.18213430508768142956e+00]
    aMode[:,6,2] = [0.10413809392985686752e-09, 0.21324280383467479893e-05, -0.80597222736636293660e-02, -0.24311900160452664110e+00, 
    -0.13586934681330335994e+00, 0.10699875405203342904e-02, -0.67114222283495701404e-01, 0.56104987912055399590e-02]
    aMode[:,7,2] = [-0.78195074731598692707e-10, -0.16113432047677196834e-05, 0.60646416575606716392e-02, 0.18453527959746301334e+00, 
    0.97748021711249979404e-01, -0.86931770826402736673e-01, -0.25859340706079221305e-01, 0.30535054676239381521e-01]
    
    aMode[:,0,3] = [-0.11168229337158983582e-20, -0.96888835482058599524e-16, 0.28484097342054432999e-12, -0.88991894939124396302e-09, 
    0.13436311859750933450e-06, -0.43949492953948432472e-05, 0.33267991354851349505e-04, -0.50016760854385964307e-04]
    aMode[:,1,3] = [-0.20765635100251089717e-17, -0.36197386357992140659e-12, 0.35732517136988684036e-09, -0.11320141902699047964e-05, 
    0.16737866108943116216e-03, -0.54399134640077706492e-02, 0.41108560676656589194e-01, -0.61778272173930721677e-01]
    aMode[:,2,3] = [-0.49227064641270947831e-14, 0.13773911494082902162e-08, 0.28000536764000152345e-06, -0.60982707102378110874e-05, 
    -0.45545917365779216012e-03, 0.30474617144202409413e-01, -0.26240435518843665541e+00, 0.40666851745242809101e+00]
    aMode[:,3,3] = [0.24046710130055206633e-11, -0.56928231643345874601e-06, -0.14658425595003152785e-03, -0.54669311790752939117e-03, 
    0.63289641147589756897e-01, -0.23416266019622873351e+01, 0.18376229679993183907e+02, -0.27890706711124808592e+02]
    aMode[:,4,3] = [-0.17790734879856566763e-09, 0.41757078162568088686e-04, 0.12661549470502249103e-01, 0.30791263485754920559e-01, 
    0.52395704373591174274e-01, -0.15856752943903087427e+01, 0.11848914793373501730e+02, -0.17521753728113509396e+02]
    aMode[:,5,3] = [0.29086256117761690731e-08, -0.68275932316891383422e-03, -0.21784801486997875663e+00, -0.66623989758662736093e+00, 
    -0.28050159504711622560e+00, -0.67774272696429793683e+00, 0.10271422822879689995e+01, -0.63392178592402839143e+00]
    aMode[:,6,3] = [-0.11009990780568283952e-07, 0.25849182195501789749e-02, 0.83614032826616710991e+00, 0.27685938326621157834e+01, 
    0.16426046025686344975e+01, 0.78572426172170208857e+00, -0.18258375964853595263e+01, 0.12152196449017165225e+01]
    aMode[:,7,3] = [0.82768727273709110647e-08, -0.19433482187601436308e-02, -0.63081127374393464180e+00, -0.21366815638955616307e+01, 
    -0.13446774521063316054e+01, -0.58592309940003861612e+00, 0.31827661237739279798e+00, -0.20240366730373042791e+00]

    aMode[:,0,4] = [-0.13947208148057608312e-20, -0.20210120268149411870e-15, 0.45539475964762736737e-12, -0.11170239869341296312e-08, 
    0.16610385153669340319e-06, -0.54155353001624000341e-05, 0.40961114502502580236e-04, -0.61571027127369957199e-04]
    aMode[:,1,4] = [-0.15010983747727246750e-17, -0.56103448860629763217e-12, 0.43831990483680147718e-09, -0.10609291877688533656e-05, 
    0.15387190085182278487e-03, -0.49799395553291745386e-02, 0.37594128280158018995e-01, -0.56482482158755402679e-01]
    aMode[:,2,4] = [-0.84949700310157272298e-14, 0.19131227654780431635e-08, 0.23797873673175664599e-06, 0.67357172013098089990e-04, 
    -0.11412327840147937774e-01, 0.38787752218711162299e+00, -0.29661484451688613361e+01, 0.44710237927891309794e+01]
    aMode[:,3,4] = [0.36903300990875096410e-11, -0.78266544732189755606e-06, -0.14356129913638142170e-03, -0.52616304038202654780e-03, 
    0.50792590874790795041e-01, -0.18789041423837609556e+01, 0.14745664699188227864e+02, -0.22381820221529649117e+02]
    aMode[:,4,4] = [-0.26744771000491356360e-09, 0.57361959249991540943e-04, 0.12466277937525278574e-01, 0.35752121631968263315e-01, 
    0.49443768069258187125e-01, -0.11592738471082895124e+01, 0.85575941704881230975e+01, -0.12621255590568500881e+02]
    aMode[:,5,4] = [0.43525625631567042006e-08, -0.93805114409436534117e-03, -0.21484367014734804257e+00, -0.75606598389073855770e+00, 
    -0.38916312244605659742e+00, -0.45744351551847461934e+00, 0.91714655267490634571e+00, -0.70503052173487583687e+00]
    aMode[:,6,4] = [-0.16457579884296458239e-07, 0.35517012004150618764e-02, 0.82496531512910689087e+00, 0.31057726963483678339e+01, 
    0.18813656203693941648e+01, 0.51092359576032500001e+00, -0.12428990388842335867e+01, 0.89617638629098603786e+00]
    aMode[:,7,4] = [0.12368783158269132105e-07, -0.26702315450896003667e-02, -0.62244835948407777337e+00, -0.23879836056356684714e+01, 
    -0.14962346641346604414e+01, -0.32180960449888429408e+00, 0.14080714553385571541e+00, -0.12793599129792307955e+00]

    aMode[:,0,5] = [-0.71007031654422592126e-20, -0.82331351329343824829e-15, 0.14860700245019204501e-11, -0.18336776863420102046e-08, 
    0.26381273327308489839e-06, -0.85305767261458509409e-05, 0.64390806859274407614e-04, -0.96740352003902945199e-04]
    aMode[:,1,5] = [-0.47759968145964668551e-17, -0.11110703618965760419e-11, 0.93331659872542349631e-09, -0.11361217227825839426e-05, 
    0.16101720194965896126e-03, -0.51853068878838977084e-02, 0.39098877702660601585e-01, -0.58726418277052969685e-01]
    aMode[:,2,5] = [-0.20561632805425666958e-13, 0.30615328304455808883e-08, 0.24271749535760407390e-06, 0.15937085189299786236e-03, 
    -0.25318551119527441528e-01, 0.84552595243637060917e+00, -0.64376874089793245659e+01, 0.96932895116823303283e+01]
    aMode[:,3,5] = [0.87503551825887093684e-11, -0.13163374744119336057e-05, -0.21315095274792228430e-03, -0.75679626156015320503e-03, 
    0.43875888066817978483e-01, -0.16187067723465442981e+01, 0.12704254977048992092e+02, -0.19288846975099222191e+02]
    aMode[:,4,5] = [-0.62980025164513007140e-09, 0.99033147185834717873e-04, 0.19286483194829802556e-01, 0.67190391627066974322e-01, 
    0.59018569821895905391e-01, -0.57675212643809139478e+00, 0.41759144712554761014e+01, -0.60616007357605896643e+01]
    aMode[:,5,5] = [0.10235134109675949609e-07, -0.16329320547735449054e-02, -0.33676325501000334838e+00, -0.13741116541224409619e+01, 
    -0.82897929864310349046e+00, -0.29064544069947594095e+00, 0.99716723490406433683e+00, -0.11955304085306053352e+01]
    aMode[:,6,5] = [-0.38688354152446052580e-07, 0.61964020776766650655e-02, 0.12977027325262353585e+01, 0.55698639583852660450e+01, 
    0.35055306218486181890e+01, 0.24501453852569592406e+00, -0.51229945875859312920e+00, 0.42642747170716877036e+00]
    aMode[:,7,5] = [0.29074290348871754119e-07, -0.46611910769336493132e-02, -0.98002188003461245813e+00, -0.42652335402388423801e+01, 
    -0.27213526220765031915e+01, -0.21604814310942872523e+00, -0.40325674485435270000e+00, 0.12987860657487151350e+00]
  
    # coefficient c
    cMode[:,0,0] = [0.62608370709224905326e-18, -0.69296927042311144973e-15, 0.53793142672880289723e-22, 0.55953376048336647130e-13, 
    -0.13644845214970875435e-11, 0.51560868494277940499e-10, -0.38841223167697958018e-09, 0.58169693048278663383e-09]
    cMode[:,1,0] = [0.79948744245788159190e-02, 0.28880230453715999061e-01, 0.18816017618889148366e-03, 0.30432220614412108794e-01, 
    0.20335251312180755434e-01, 0.12253559136086547010e+01, -0.11162104253917091156e+02, 0.17710281832472645646e+02]
    cMode[:,2,0] = [-0.30562966462902028119e+03, -0.11990795349370895195e+04, -0.65802994518949660118e+01, -0.12273805927634358070e+04, 
    -0.65349097657709229736e+03, -0.12824769943213845024e+03, -0.88590404530202224719e+02, -0.92014824378423334394e+02]
    cMode[:,3,0] = [-0.95152195796331646704e+02, -0.36956353969242154988e+03, -0.20695307247843501841e+01, -0.38362497103275590148e+03, 
    -0.22776414558915512031e+03, -0.46624811555373160132e+02, 0.45685596890036768158e+02, -0.25467538299896208542e+03]
    cMode[:,4,0] = [0.32918009134216310584e+03, 0.12851583684685703445e+04, 0.71222223072393315845e+01, 0.13249069943127731452e+04, 
    0.74436590714498462872e+03, 0.14749387222546454623e+03, 0.42968129090712094964e+02, 0.33666974003731491293e+03]
    cMode[:,5,0] = [0.20156315877857942098e+03, 0.80514059433962259504e+03, 0.42634222620010087112e+01, 0.79880139299917756190e+03, 
    0.34272486825284445011e+03, 0.52606372363987690121e+02, 0.74826789775518696146e+01, -0.49206154796489274261e+01]
    cMode[:,6,0] = [-0.53421252140832686805e+03, -0.21573846013695305856e+04, -0.11180286753450849879e+02, -0.20919394990520148169e+04, 
    -0.77083430972020012816e+03, -0.74269446238765679524e+02, 0.86081879160377337001e+01, -0.33189293006747679903e+01]
    cMode[:,7,0] = [0.64042667076955872573e+03, 0.25941583461638844099e+04, 0.13364255832056319839e+02, 0.24982768986941823463e+04, 
    0.87945971890665362025e+03, 0.70070351276938351858e+02, -0.86139878258589916981e+01, 0.30011846979293261838e+01]

    cMode[:,0,1] = [-0.63013334128716040893e-13, 0.60865772431920603935e-05, -0.40016262502772708131e-09, -0.23069864565108759713e-04, 
    0.23255146051530801720e-03,-0.51951144258714805346e-02, 0.36136455088219365805e-01, -0.53017682607232261560e-01]
    cMode[:,1,1] = [0.55436122531818750047e-03, -0.66992043381081387565e-02, 0.83803970952918511727e-02, 0.80754210845424374554e-02, 
    -0.61427751801047527635e-02, 0.54122156026262873140e+00, -0.43010880044939874267e+01, 0.65777678112627633311e+01]
    cMode[:,2,1] = [-0.23084376754368056694e+02, 0.20899246010739616075e+03, -0.36836390749521057408e+03, -0.52914996781002994197e+03, 
    -0.20007855689466294002e+03, -0.36056061195965001253e+02, -0.16132602175333610183e+02, 0.21643026623710106548e+01]
    cMode[:,3,1] = [-0.30489331973819799870e+01, 0.33304902492752548326e+02, -0.47908018506183607243e+02, -0.66968634395671120529e+02, 
    -0.34606055921065168590e+02, -0.74942275618948936966e+01, 0.20818848022147209420e+02, -0.68532587218682348151e+02]
    cMode[:,4,1] = [0.44092752992499013586e+01, -0.47413797913825259655e+02, 0.69401964003076273002e+02, 0.97614272148050673649e+02, 
    0.48704912842393142113e+02, 0.99941378554753921292e+01, -0.14570861805562320689e+02, 0.65068431858518227528e+02]
    cMode[:,5,1] = [0.27754841758044097588e+02, -0.24286715291266780525e+03, 0.44379959765035561503e+03, 0.63697225078825798760e+03, 
    0.23079424266757904149e+03, 0.39146925311929221535e+02, 0.12991883575647105075e+02, -0.34552170740359109402e+01]
    cMode[:,6,1] = [-0.20272909693943255149e+02, 0.14757207634388509021e+03, -0.32708628601917539846e+03, -0.46225115463536488036e+03, 
    -0.13753186496272293837e+03, -0.14358017591372269627e+02, 0.41733787941502891172e+00, -0.14829734668187728452e+00]
    cMode[:,7,1] = [0.22054195737479753702e+02, -0.15104824369566767217e+03, 0.35667767962821925742e+03, 0.50036616092734789162e+03, 
    0.14078440554104824755e+03, 0.12025910667067751802e+02, -0.70328949883151059552e+00, 0.24434145951697048282e+00]

    cMode[:,0,2] = [-0.28862183650305684778e-01, 0.16202748718849406373e-02, 0.62452442623701900359e-08, -0.10292552636752272388e-04, 
    0.14899979598450072693e-10, 0.54701873831866327790e+00, -0.33801096158365382393e+01, 0.46663975792709919687e+01]
    cMode[:,1,2] = [0.82086900572783427776e+00, 0.66781995371601139410e+00, 0.17689517609606757453e+01, 0.18458732991639696052e+01, 
    -0.13408206456871818446e-02, -0.55045820392109447993e+01, 0.34057914707574474810e+02, -0.44172744618314503384e+02]
    cMode[:,2,2] = [-0.33440561402073085695e+04, -0.63006668870531044035e+04, -0.16148889754716360123e+05, -0.16734685574071548330e+05, 
    0.98894432761575998824e+01, -0.10391741179165414621e+04, -0.17857836173455567951e+04, 0.29803489709704852117e+04]
    cMode[:,3,2] = [0.31915064954348162373e+04, 0.59270193788343030760e+04, 0.15182397380443129364e+05, 0.15734311019448650625e+05, 
    -0.93434105903445914265e+01, 0.99260524095183413351e+03, 0.20341329769010583206e+04, -0.32962143597458424260e+04]
    cMode[:,4,2] = [-0.92169284592923155230e+02, -0.14814666530740792538e+03, -0.37839662711403616590e+03, -0.39255805350361141492e+03, 
    0.24567231705288499199e+00, -0.11195431933835682247e+02, -0.27154689478860825069e+03, 0.37730991124653248114e+03]
    cMode[:,5,2] = [0.54075863200485683179e+03, 0.13068947201920562140e+04, 0.34015030753731947399e+04, 0.35222027482034166112e+04, 
    -0.19372427035775936943e+01, 0.10115813738135293053e+03, -0.23832476149210184424e+02, -0.16890808415607296844e+02]
    cMode[:,6,2] = [-0.77613909679256050111e+03, -0.22744934351169181496e+04, -0.60171788754328519033e+04, -0.62294563978631467549e+04, 
    0.32775011290202917813e+01, -0.72131812461142832404e+02, 0.27116603294455088324e+02, -0.71224576605140779150e+01]
    cMode[:,7,2] = [0.69369255218888747904e+03, 0.22078310233442990373e+04, 0.58858136079721381506e+04, 0.60932655114670426499e+04, 
    -0.31519955877359868701e+01, 0.44446156981790601037e+02, -0.14233922505288183479e+02, 0.34387488557126792976e+01]

    cMode[:,0,3] = [-0.36868002035455376130e-21, 0.10541748946585167701e-17, 0.27755566547052099579e-13, -0.34780372211895671519e-15, 
    -0.99328472275559516191e-12, 0.37378357885053432596e-10, -0.27315035342737905565e-09, 0.40965931239852784173e-09]
    cMode[:,1,3] = [-0.33737087677076624814e-04, -0.11396625664731094840e-02, -0.10789923154822660400e-01, -0.80796849963401182748e-02, 
    0.32166227895723737972e-01, -0.25571424523548520468e+01, 0.21267119641439142796e+02, -0.33078925310689939465e+02]
    cMode[:,2,3] = [-0.17479871537929662750e+01, -0.72444600080193755076e+02, -0.55033093275560522883e+03, -0.50210145669063086515e+03, 
    -0.25844521786360195036e+03, -0.55113408951058726614e+02, 0.20187135793924890769e+03, -0.81289982453232934034e+03]
    cMode[:,3,3] = [0.78603614447709357904e+00, 0.32329428441630612134e+02, 0.24762820448825531016e+03, 0.22413360241783610860e+03, 
    0.11996249854833778059e+03, 0.30448969122459486058e+02, -0.23988687682537666034e+03, 0.67671098556149500424e+03]
    cMode[:,4,3] = [0.12159934205840530196e+01, 0.51016427583112839982e+02, 0.38232595903016939331e+03, 0.35346430190826185757e+03, 
    0.17043627550273194870e+03, 0.36343814970370997841e+02, 0.25302217119464871508e+02, 0.16017628158741096910e+03]
    cMode[:,5,3] = [-0.13415948717761450037e+01, -0.59032589809690589888e+02, -0.41777999557453684431e+03, -0.40902457469604014406e+03, 
    -0.14475183395103458749e+03, -0.29478536348031818548e+02, -0.13387094046024579085e+01, 0.23831368497822609242e+01]
    cMode[:,6,3] = [0.55621147775238517496e+01, 0.24890779339281028370e+03, 0.17228008107309571883e+04, 0.17256605312262474072e+04, 
    0.52806415548478540245e+03, 0.64772764588200661961e+02, -0.12923255027738644784e+02, 0.58752559976531326668e+01]
    cMode[:,7,3] = [-0.73285857760141963623e+01, -0.32941251542733325230e+03, -0.22662989686584209536e+04, -0.22842752627361360140e+04, 
    -0.66988125859857134969e+03, -0.67321294156948754405e+02, 0.12094082762406057618e+02, -0.50344380663030463551e+01]

    cMode[:,0,4] = [0.57951308146739251014e+01, -0.14999087019121093433e+03, -0.19559326802070165385e+01, -0.66720804039446477418e+03, 
    -0.75756207610197607849e+02, -0.79695623597154439110e+02, 0.12741669724908954997e+04, -0.25779398744795645193e+04]
    cMode[:,1,4] = [0.25559373539070939784e+03, -0.26954324048080380293e+04, 0.42741834334120607508e+02, -0.13609107834915077361e+05, 
    -0.18219129315199742435e+04, -0.49540503909243787106e+03, -0.86525145704320394202e+03, -0.19349682251463748983e+04]
    cMode[:,2,4] = [-0.87923543175638503299e+05, 0.30861003822923382955e+06, -0.25217316373834348652e+05, 0.21333098058144601694e+07, 
    0.33404584481156045505e+06, 0.42354436002087458845e+05, 0.58637814837743640339e+04, -0.30291614862820930298e+04]
    cMode[:,3,4] = [0.21501887419077996277e+06, -0.98416860975998154970e+06, 0.50756266605703403982e+05, -0.58885368624678218196e+07, 
    -0.88230632458050202160e+06, -0.24671074506661172520e+06, -0.14496629047056444505e+06, 0.68193611837692298394e+05]
    cMode[:,4,4] = [0.18016800297232990146e+02, 0.11854731605905291047e+00, 0.41242056558479642802e-02, 0.10788775915866732901e+00, 
    0.76539029541206167195e+00, 0.67292188204585867694e+02, -0.31151519702859546967e+03, 0.44202304203357769551e+03]
    cMode[:,5,4] = [0.20068140169309098830e+06, -0.93099363016509286694e+06, 0.47215756436253171202e+05, -0.55473705494319487385e+07, 
    -0.82963829864856943885e+06, -0.23595446376207669381e+06, -0.14387731979865256981e+06, 0.66726697732453104094e+05]
    cMode[:,6,4] = [-0.10659833147754620430e+06, 0.36525155976544767533e+06, -0.32972599682804299980e+05, 0.26280915665374022793e+07, 
    0.41399542203693986408e+06, 0.40200322950430580348e+05, 0.33714384362519780324e+04, -0.16170695162643912823e+04]
    cMode[:,7,4] = [0.45385852289903034773e+05, -0.15367285853252436567e+06, 0.15938904084451054998e+05, -0.11759967065602889846e+07, 
    -0.18617563074881875451e+06, -0.11629254312624910383e+05, -0.37356632983270783299e+03, 0.13368622059973152005e+03]

    cMode[:,0,5] = [0.37071935082976326114e-15, 0.46201169190899653571e-08, -0.62075147292222574435e-11, -0.36146708876274598054e-06, 
    0.87046861427880060091e-05, -0.10076975633470059978e-03, 0.89298240723925399464e-03, -0.14060364310276510124e-02]
    cMode[:,1,5] = [-0.56188781974792609830e-01, -0.40650854381275376425e+01, -0.14988738284605299000e+01, -0.38632728626301734209e+01, 
    -0.22261011246466999580e+01, -0.14212548821848476343e+02, 0.19383430976405556123e+03, -0.34806064213022160913e+03]
    cMode[:,2,5] = [0.27593841470781566016e+03, 0.22140385471263797079e+05, 0.78362796461845718454e+04, 0.19668830633326862766e+05, 
    0.10181292569179314355e+05, 0.39096643822536609746e+04, 0.26876686999693699675e+04, 0.33310618692009836827e+04]
    cMode[:,3,5] = [0.44061551736067183782e+02, 0.34746817101944778016e+04, 0.12384774155446498511e+04, 0.31312056838986070950e+04, 
    0.17260274467090328087e+04, 0.70922681375271823256e+03, -0.30968222241454737009e+03, 0.29976750308301927105e+04]
    cMode[:,4,5] = [-0.29826526924997253331e+03, -0.23828019858653441964e+05, -0.84485266842828430355e+04, -0.21247202725657969857e+05, 
    -0.11181459854792739072e+05, -0.43708622712852429614e+04, -0.24560179552632419586e+04, -0.60645360750106398484e+04]
    cMode[:,5,5] = [-0.74893303991611572811e+02, -0.62909773314430736945e+04, -0.21848887114479706994e+04, -0.53368828632132494504e+04, 
    -0.22162146435679854761e+04, -0.52579256909996399116e+03, -0.11309723650631768876e+03, 0.63863048387678587047e+02]
    cMode[:,6,5] = [0.19717911116782794067e+03, 0.16899744672548959734e+05, 0.58199397638247170050e+04, 0.13983841291452328015e+05, 
    0.50676093636595025415e+04, 0.75432123378980122652e+03, 0.10996364977646411276e+02, -0.48990730626876937137e+01]
    cMode[:,7,5] = [-0.22796489432205970793e+03, -0.19665709629971770411e+05, -0.67538957248075544015e+04, -0.16130393271007925193e+05, 
    -0.55557948391593710013e+04, -0.67829337377268315023e+03, 0.63691256079676055179e+01, -0.36820923261738309761e+01]

    # coefficient p 
    p[:,0] = [0.32901733177624610249e-01, 0.54475681390900216882e-01, 0.11033542666255162778e+00, 
    0.21824017019698205288e+00, 0.43540342906595910221e+00, 0.87057139406808232706e+00, 
    0.17411041057694376377e+01, 0.34822027229232799250e+01, 0.29391078654639329670e-01, 
    0.53092570701901200536e-01, 0.10068022495463000431e+00, 0.21707901052684266396e+00, 
    0.43526892089758080217e+00, 0.87054714666225887498e+00, 0.17411005963420354447e+01, 
    0.34822023079824373503e+01]
    p[:,1] = [0.32535198971349430507e-01, 0.57792553225847100861e-01, 0.11708104989648597804e+00, 
    0.21135795367821805790e+00, 0.42489217791799518408e+00, 0.86257818103884389415e+00, 
    0.17181130605554559842e+01, 0.34339371403040641617e+01, 0.25660275820190880935e-01, 
    0.45569108067616728163e-01, 0.86363227093303382986e-01, 0.20636215029847653212e+00, 
    0.42568882948609800820e+00, 0.85912749746654117899e+00, 0.17163445496343952001e+01, 
    0.34350464850833977159e+01]
    p[:,2] = [0.32114531391568896800e-01, 0.54668167014354631660e-01, 0.10999415084820487464e+00, 
    0.21830735865316683863e+00, 0.43522952245694588313e+00, 0.87059254539023882557e+00, 
    0.17410412227916809868e+01, 0.34821651743153156921e+01, 0.25685870467860096866e-01, 
    0.45698157579046663201e-01, 0.90356977415432240263e-01, 0.21884901172766766386e+00, 
    0.43511983572804924236e+00, 0.87029548334738358050e+00, 0.17410538620391950992e+01, 
    0.34822549594214162738e+01]
    p[:,3] = [0.32356337372804593321e-01, 0.54152800183054496940e-01, 0.10961184933750356407e+00, 
    0.21794853515003347332e+00, 0.43533813536787366871e+00, 0.87056003928681793269e+00, 
    0.17411018194244403112e+01, 0.34822024269907854154e+01, 0.27588798509226557520e-01, 
    0.48812593148800704767e-01, 0.93213066985517798457e-01, 0.21740973446223286202e+00, 
    0.43526939652742839825e+00, 0.87054487704825334049e+00, 0.17411007942523434977e+01, 
    0.34822023168779896451e+01]
    p[:,4] = [0.32735344319694603676e-01, 0.54347222488222799441e-01, 0.10989084818706484902e+00, 
    0.21805120636819852464e+00, 0.43536325140487690532e+00, 0.87056578462259590622e+00, 
    0.17411028067887455605e+01, 0.34822028139735676788e+01, 0.27779830466381087994e-01, 
    0.50806515286805851161e-01, 0.95276608648916276678e-01, 0.21745769582596392588e+00, 
    0.43527860088967376128e+00, 0.87054808117661632849e+00, 0.17411000172672461694e+01, 
    0.34822021788641484008e+01]
    p[:,5] = [0.33568077258163553366e-01, 0.53905902840230535133e-01, 0.10958409128391679487e+00, 
    0.21675427898182215713e+00, 0.43240630217346920360e+00, 0.86456015794363612059e+00, 
    0.17290777196751532684e+01, 0.34581494323600296958e+01, 0.28810823097181398111e-01, 
    0.53751387895676616679e-01, 0.10088751776373805491e+00, 0.21567365023402529367e+00, 
    0.43226560363373209838e+00, 0.86453474685580449232e+00, 0.17290747052830268692e+01, 
    0.34581487532588752742e+01]


    # coefficient r
    r[:,0] = [0.23400582376918621640e-01, 0.29906910603763133593e+00, 0.59419217080359816307e+00, 
    0.51683309796288279258e+00, 0.55280210585007472090e+00, 0.89024092184836405294e+00, 
    0.17236578859735887547e+01, 0.34342717240262339295e+01, 0.89287729231317083389e-01, 
    0.15439963383324775136e+00, 0.50639243810157408276e-01, 0.24456407532792816539e+00, 
    0.39627924482769603997e+00, 0.86233612861692066076e+00, 0.17188649618875711411e+01, 
    0.34339261899212027984e+01]
    r[:,1] = [0.70036641254021372304e-01, 0.33297901220247387854e+00, 0.80631162662909527938e+00, 
    0.54483599484759341891e+00, 0.56290677049809243470e+00, 0.87422019588133075274e+00, 
    0.17295204193052738261e+01, 0.34651766994561397083e+01, 0.57115996571135090320e-01, 
    0.23559291883706992010e+00, 0.11690775021980281955e+00, 0.26929992715503474620e+00, 
    0.42836762405666322095e+00, 0.87049839288489376798e+00, 0.17350924596759142559e+01, 
    0.34557476257228634253e+01]
    r[:,2] = [0.11127103098086079668e+00, 0.42445608810651771491e+00, 0.71058432225687280237e+00, 
    0.69760620618057709307e+00, 0.58851822769682966551e+00, 0.10410712531090018373e+01, 
    0.18016710591015945297e+01, 0.35162043268667866335e+01, 0.50015594405195891170e+00, 
    0.32951760674056020938e+00, 0.11251791947458271714e+00, 0.19597524022502758711e+00, 
    -0.52856178098938624287e-01, 0.95266905942729742662e+00, 0.18279298809137124237e+01, 
    0.35144044692591625000e+01]
    r[:,3] = [0.22860180522694295568e-01, 0.25001972739726898709e+00, 0.46402414947596950511e+00, 
    0.44398927627952060603e+00, 0.50272011857581864191e+00, 0.88445628872250843244e+00, 
    0.17389166922733014786e+01, 0.34834653146383227628e+01, 0.49049989399817732760e-01, 
    0.85147557568237122183e-01, 0.23424821227315173466e+00, 0.14250169643289223309e+00, 
    0.40586458823694400166e+00, 0.87361464455904407344e+00, 0.17279679104606691097e+01, 
    0.34828619038557646625e+01]
    r[:,4] = [0.48558255088791124620e+00, -0.60422350819111194653e+00,-0.13683535645881987896e+01, 
    -0.85855064913451286656e+00, 0.28100294887963377377e+00, 0.84995521582071482669e+00, 
    0.17002422172314077819e+01, 0.34601959118742100507e+01, -0.60269701842058500673e+00, 
    -0.31233918670609019940e+00, -0.71139286069860601102e-01, 0.13800610282920311533e+00, 
    0.38218789451073229557e+00, 0.81980250822178710734e+00, 0.17183601833584307705e+01, 
    0.34710558209881763325e+01]
    r[:,5] = [0.38821262813796547419e-01, 0.35983028187700409894e+00, 0.57018233005798357737e+00, 
    0.51153414239116576922e+00, 0.55367851931772325002e+00, 0.92841885168264770555e+00, 
    0.17036409347489702703e+01, 0.34382652552776309384e+01, 0.54101616716924096905e-01, 
    0.17168542829213128797e+00, 0.10028211367824637623e+00, 0.27153034480893150082e+00, 
    0.43311645903980116045e+00, 0.75314774033930556029e+00, 0.17038798689307688150e+01, 
    0.34370447884477832722e+01]
    
    
    #pvec, rvec = np.zeros(16), np.zeros(16)
    CS = np.array([1.068627, 1.572350, 1.015095, 1.714487, -5.232800, 6.528898])/0.002
    
    #Bx, By, Bz = 0,0,0
    Bxout, Byout, Bzout = 0,0,0
    
    xp = [x + 0.001, x, x]
    yp = [y, y + 0.001, y]
    zp = [z, z, z + 0.001]
    xm = [x - 0.001, x, x]
    ym = [y, y - 0.001, y]
    zm = [z, z, z - 0.001]
    
    
    for Mode in range(0,6):
        ain = aMode[:,:,Mode]
        cin = cMode[:,:,Mode]

        p_1 = p[0:8,Mode]
        p_2 = p[8:16,Mode]

        r_1 = r[0:8,Mode]
        r_2 = r[8:16,Mode]
        
        Up = np.zeros(3)
        Um = Up 
        
        term_p, term_r = np.zeros((8,8)), np.zeros((8,8))
        
        for i in range(8):
            term_p[0:8,i] = 1/(p_1[i]*p_1[i]) + 1/(p_2[0:8]*p_2[0:8])
            term_r[0:8,i] = 1/(r_1[i]*r_1[i]) + 1/(r_2[0:8]*r_2[0:8])

        term_p = np.exp(np.sqrt(term_p))
        term_r = np.exp(np.sqrt(term_r))
    
        for i in range(8):
            cos_yp = np.cos(yp/p_1[i])
            cos_ym = np.cos(ym/p_1[i])
            sin_yp = np.sin(yp/r_1[i])
            sin_ym = np.sin(ym/r_1[i])
            for k in range(8):
                Up = Up + ain[k,i]*(term_p[k,i]**xp)*cos_yp*np.sin(zp/p_2[k]) + cin[k,i]*(term_r[k,i]**xp)*sin_yp*np.sin(zp/r_2[k])
                Um = Um + ain[k,i]*(term_p[k,i]**xm)*cos_ym*np.sin(zm/p_2[k]) + cin[k,i]*(term_r[k,i]**xm)*sin_ym*np.sin(zm/r_2[k])
        
        
        B = Up - Um
        Bxout += B[0]*CS[Mode]
        Byout += B[1]*CS[Mode]
        Bzout += B[2]*CS[Mode]    
            
            
    return Bxout, Byout, Bzout 
    
def tail_mag_shield_cyl_S3(rmap,pmap,zmap,sphiOut,M,Mode):
    # ----------------------------------------------------------- # 
    # Translated from procedure tail_mag_shield_cyl_S3
    # ----------------------------------------------------------- #     
    
    Xcar, Ycar, Zcar = cyl2car_pos(rmap, pmap, zmap)
    xJSO, yJSO, zJSO = S32PJSO(sphiOut, Xcar, Ycar, Zcar)
    
    # Divide position by 100 to match the coefficients
    xJSO = xJSO/100
    yJSO = yJSO/100
    zJSO = zJSO/100
        
    # works in the original coordinate system S3
    BxJSO, ByJSO, BzJSO = B_tail_shield(xJSO, yJSO, zJSO, M,Mode)
    
    BxS3, ByS3, BzS3 = PJSO2S3(sphiOut, BxJSO, ByJSO, BzJSO)
    
    # transform back to cylindrical
    Brcss, Bpcss, Bzcss = car2cyl_vect(pmap,BxS3, ByS3, BzS3)
    
    return Brcss, Bpcss, Bzcss 



def csheet_struc(rho,phi,stheta,XJSO, YJSO):
    # ----------------------------------------------------------- # 
    # Translated from procedure csheet_struc
    #
    # This program calculates the distance of current sheet from the
    # Jovigraphic equator in system III coordinates
    # ----------------------------------------------------------- #      
    radian = 0.01745329252
    period = 9.927953
    Omegaj = 36.26125
    x0 = -45
    phip0 = 6.12611
    ten = 0.16913733749783
    C1 = 0.005973
    C2 = 5.114e-5
    C3 = 1.59e-5
    psi2 = -1.201201
    C4 = 0.313244
    C5 = -0.366166
    psi4 = 2.2604522
    
    if (np.abs(XJSO) < 1e-6) : 
        XJSO = 1e-6 
    
    RLT = np.arctan2(YJSO, XJSO)
    
    delay1 = C1*rho + 0.5*C2*(rho*rho)*np.cos(RLT - psi2) + 0.5*C3*rho*rho
    Alfven = -360/( period*(C4*np.cos(RLT - psi4) + C5) )
    delay2 = rho/Alfven*Omegaj*radian
    
    phip = phip0 - delay1 - delay2
    
    hyptan = x0*np.tanh(XJSO/x0)
    rho1 = np.sqrt(hyptan*hyptan + YJSO*YJSO)
    
    ZNS3 = rho1*ten*np.cos(phi - phip) + XJSO*(1 - np.tanh(np.abs(x0/XJSO) ))*np.tan(stheta)
    
    return ZNS3

def csheet_N(t,rho, phi, zs3, stheta):
    # ----------------------------------------------------------- # 
    # Translated from procedure csheet_struc
    #
    # This procedure uses csheet_structure to calculate the normal
    # direction to the current sheet in system III coordinates
    # ----------------------------------------------------------- #    
    delta = 0.1
    delta2 = 2*delta
    xs3 = rho*np.cos(phi)
    ys3 = rho*np.sin(phi)
    xp = xs3 + delta
    xm = xs3 - delta
    yp = ys3 + delta
    ym = ys3 - delta
    
    # First calculate the x derivatives
    # Define the positive parameters
    rhop = np.sqrt(xp*xp + ys3*ys3)
    phip = np.arctan2(ys3, xp)
    vecin = [xp, ys3, zs3]
    
    vecou = JROT(t, vecin, 's3c', 'jso')
    
    XJSOP = vecou[0]
    YJSOP = vecou[1] 
    ZNS3P = csheet_struc(rhop,phip,stheta,XJSOP, YJSOP)
    
    # Now define the negative parameters
    rhom = np.sqrt(xm*xm + ys3*ys3)
    phim = np.arctan2(ys3,xm)
    vecin = [xm, ys3, zs3]
    vecou = JROT(t,vecin, 's3c', 'jso')
    XJSOM = vecou[0]
    YJSOM = vecou[1]
    
    ZNS3M = csheet_struc(rhom,phim,stheta,XJSOM, YJSOM)
    dzdx = (ZNS3P - ZNS3M)/delta2
    
    # Next the y derivative
    # Define the positive parameters
    rhop = np.sqrt(xs3*xs3 + yp*yp)
    phip = np.arctan2(yp, xs3)
    vecin = [xs3, yp, zs3]
    vecou = JROT(t,vecin,'s3c','jso')
    XJSOP = vecou[0]
    YJSOP = vecou[1]
    
    ZNS3P = csheet_struc(rhop,phip,stheta,XJSOP, YJSOP)

    # Now define the negative parameters
    rhom = np.sqrt(xs3*xs3 + ym*ym)
    phim = np.arctan2(ym, xs3)
    vecin = [xs3, ym, zs3]
    vecou = JROT(t,vecin, 's3c', 'jso')
    XJSOM = vecou[0]
    YJSOM = vecou[1]
    ZNS3M = csheet_struc(rhom,phim,stheta,XJSOM, YJSOM)


    dzdy = (ZNS3P - ZNS3M)/delta2
    RNx = -dzdx
    RNy = -dzdy
    RNz = 1
    RN = np.sqrt(RNx*RNx + RNy*RNy + RNz*RNz)
    RNx = RNx/RN
    RNy = RNy/RN
    RNz = RNz/RN
    
  
    return RNx, RNy, RNz 
     
def get_mapped_sunangle(sthetaIN,sphiIN,xpx,xpy,xpz,ypx,ypy,ypz,zpx,zpy,zpz):
    # ----------------------------------------------------------- # 
    # Translated from procedure get_mapped_sunangle
    # ----------------------------------------------------------- #    
 
    Xin, Yin, Zin = SPH2CAR_pos(1, sthetaIN, sphiIN)
    Xout = Xin*xpx + Yin*xpy + Zin*xpz
    Yout = Xin*ypx + Yin*ypy + Zin*ypz
    Zout = Xin*zpx + Yin*zpy + Zin*zpz 
    
    Rnotused, sthetaOut, sphiOut = CAR2SPH_pos(Xout, Yout, Zout)
    return  sthetaOut, sphiOut

def getBimf(t,thetaS3,phiS3):
    # ----------------------------------------------------------- # 
    # Translated from getBimf
    #
    # Initially modified by Fran Bagenal Nov. 2008 
    #
    # Input background IMF field: 
    #        Br=Bx = +-0.2
    #        Bp=By = -+1.0
    #        Bt=Bz = 0.0 
    #        OR Zero...    
    # ----------------------------------------------------------- #        
    
    BxIMFgsm, ByIMFgsm, BzIMFgsm = 0, 0, 0
    # go from JSM to S3
    vecin = [BxIMFgsm, ByIMFgsm, BzIMFgsm]
    
    vecou = JROT(t, vecin, 'jsm', 's3c')
    BxIMFs3 = vecou[0]
    ByIMFs3 = vecou[1]
    BzIMFs3 = vecou[2]
    
    # go to spherical
    BrS3IMF, BtS3IMF, BpS3IMF = CAR2SPH_MAG(thetaS3, phiS3, BxIMFs3, ByIMFs3, BzIMFs3)

    return BrS3IMF, BtS3IMF, BpS3IMF
    
def checkIfInsideMagnetop(x, y, z):
    # ----------------------------------------------------------- # 
    # Translated from procedure checkIfInsideMagnetop
    #
    # INPUT:   - x,y,z position 
    # OUTPUT:  - 1 -> inside magnetopause 
    #          - 0 -> outside magnetopause
    # ----------------------------------------------------------- # 
    answer = 1
    
    # Symmetric Magnetopause Parameter
    A = 13779.1546267827
    B = -130.0469911647737
    C = -0.2216972473764874
    D = 0
    E = -0.8453162863101464
    F = 0
    
    phi = np.arctan2(z,y)
    rho_in = np.sqrt(z*z + y*y)
    cos_phi = np.cos(phi)
    sin_phi = np.sin(phi)
    
    # check if we are on the dayside
    if (x >= 0) :
        aa = E*cos_phi*cos_phi - sin_phi*sin_phi + F*cos_phi*sin_phi
        bb = D*cos_phi
        cc = A + B*x + C*x*x
        term = bb*bb - 4*aa*cc
        sqrt_term = np.sqrt(term)
        rho1 = (-bb + sqrt_term)/(2*aa)
        rho2 = (-bb - sqrt_term)/(2*aa)
        # The right solution has rho positive (i.e. the bigger of the two)
        rho = rho1
        if (rho2 > rho1) :
            rho = rho2
        if (term < 0) or (rho < 0) or (rho_in > rho) :
            answer = 0    
            
        # Now the nightside
        aa = E*cos_phi*cos_phi - sin_phi*sin_phi + F*cos_phi*sin_phi
        bb = D*cos_phi
        cc = A + B*x + C*x*x
        term = bb*bb - 4*aa*cc
        sqrt_term = np.sqrt(term)
        rho1 = (-bb + sqrt_term)/(2*aa)
        rho2 = (-bb - sqrt_term)/(2*aa)   
        # The right solution has rho positive (i.e. the bigger of the two)
        rho = rho1
        if (rho2 > rho1) :
            rho = rho2
        if (term < 0) or (rho < 0) or (rho_in > rho) :
            answer = 0   
            
            
    return answer 

def CheckIfInsideMappedMp(t,XS3, YS3, ZS3, ZNS3):
    # ----------------------------------------------------------- # 
    # Translated from procedure CheckIfInsideMappedMp
    # ----------------------------------------------------------- #
    vecin = [XS3, YS3, ZS3 + ZNS3]
    vecou = JROT(t,vecin,'s3c','jsm')
    
    XJSOold = vecou[0]
    YJSOold = vecou[1]
    ZJSOold = vecou[2]
    
    ianswer = checkIfInsideMagnetop(XJSOold, YJSOold, ZJSOold)
    return ianswer
    
def tail_mag_notilt_all_modes(rho, phi, zz, RLT, Mode):
    # ----------------------------------------------------------- # 
    # Translated from procedure tail_mag_notilt_allmodes
    # ; Modified from csheet_deform to calculate the field of only one point
    # The subroutine first calculates the Brho and Bz components at the stretched location
    # and then multiplies with the stretch matrices to calculate Bphi
    # alat,colat RLT are in radians, RLT is measured from Noon
    # ----------------------------------------------------------- #    
    
    f, beta = np.zeros((6,6)), np.zeros((6,6))
    X = np.zeros(6)
    isumno = 6
    
    #begin Hannes Modes
    C = [-28.402612, -18.160631, 14.543719, 12.873892, -5.590983, 3.461748] # short model
    
    # ring current
    f[:,0] = [188.880535100774, 170.216250249208, -232.209387453753, 17729.2564306361, -17426.2836367108, -176.740242190339]
    beta[:,0] = [59.2554626464844, 14.3016757965088, 3.41575503349304, 5.47725486755371, 5.55856323242188, 30.7563076019287]
    # L = 6
    f[:,1] = [34.5422110518, 6.0536537495, 414.2147150061, -155.3647545840, 316.3317439470, -73.2913775361]
    beta[:,1] = [59.2781028748, 16.4449157715, 4.1893663406, 3.0382030010, 9.1110754013, 30.9898948669]
    # L = 12
    f[:,2] = [-165.1020233300, 554.5217109532, 315.5180924885, -125.0952135958, 449.1102649649, 371.9172713735]
    beta[:,2] = [55.9542961121, 14.5898332596, 3.9975905418, 2.8588902950, 8.6061649323, 21.9151973724]
    # L = 24
    f[:,3] = [591.9706482108, 1376.0166168630, 298.8401265534, -156.9445820815, 521.9904943354, 2387.3415724416]
    beta[:,3] = [59.0788803101, 16.2336521149, 3.5862164497, 2.7877552509, 8.4433155060, 30.6951904297]
    # L = 48
    f[:,4] = [9180.3492664995, 1679.4592666407, 274.9968669893, -144.4824637207, 532.5683191252, 4540.5253277291]
    beta[:,4] = [59.2788314819, 16.4751930237, 3.5809681416, 2.7559804916, 8.5185470581, 31.0038948059]
    # L = 96
    f[:,5] = [28927.9037953983, 2214.0596352902, 242.5405186300, -108.6915959578, 609.8932996003, 7437.5789048160]
    beta[:,5] = [72.2351608276, 18.0498085022, 3.7080323696, 2.6697711945, 8.9703912735, 35.2125587463] 
    # end Hannes Modes
    
    P = 5.73e-3
    if (Mode > 6) or (Mode < 1) :
        istartLoop = 0
        istopLoop = 5
    else : 
       istartLoop = Mode - 1
       istopLoop = istartLoop 
    
    # We work in the dipole magnetic coordinate system
    # In this subroutine:
    # index I = 1,7 corresponds to each component of a mode
    # index L = 1,6 denotes the six L modes
    # index j sums over observations (Br and Btheta) and is twice the number of observations
    # index M is used to obtain least squares equations (same dimension as L)
    
    PI = np.pi
    drho = 0.05
    dz = 0.05
    phi = phi
    # D is half thickness of current sheet
    D0 = 11.0
    D1 = 9.0
    D = D0 + D1*np.cos(RLT - 1.5*PI)
    D = 2.0

    #RMSBr,RMSBt, RMSBp = 0,0,0
    # Start calculating the field
    Brhodm = 0
    Bzdm = 0
    Bpdm = 0

    RHOMAG = rho
    ZMAG = zz
    Z = ZMAG
    
    # First calculate Brhod
    for L in range(istartLoop,istopLoop+1):
        ZM = np.abs(Z - dz)    
        if (ZM < D) : 
            ZM = 0.5*(ZM*ZM/D + D)
        ZP = np.abs(Z + dz)    
        
        if (ZP < D) :
            ZP = 0.5*(ZP*ZP/D + D)
            
        xlpp = 0
        xlpm = 0    
            
        for i in range(0,isumno):
            S1p = np.sqrt( (beta[i,L] + ZP)*(beta[i,L] + ZP) + (RHOMAG + beta[i,L])*(RHOMAG + beta[i,L]) )
            S2p = np.sqrt( (beta[i,L] + ZP)*(beta[i,L] + ZP) + (RHOMAG - beta[i,L])*(RHOMAG - beta[i,L]) )
            S1m = np.sqrt( (beta[i,L] + ZM)*(beta[i,L] + ZM) + (RHOMAG + beta[i,L])*(RHOMAG + beta[i,L]) )
            S2m = np.sqrt( (beta[i,L] + ZM)*(beta[i,L] + ZM) + (RHOMAG - beta[i,L])*(RHOMAG - beta[i,L]) )
            tp = 2*beta[i,L]/(S1p + S2p)
            tm = 2*beta[i,L]/(S1m + S2m)
            AAp = tp*np.sqrt(1 - tp*tp)/(S1p*S2p)
            AAm = tm*np.sqrt(1 - tm*tm)/(S1m*S2m)
            xlpp = xlpp + f[i,L]*AAp*RHOMAG
            xlpm = xlpm + f[i,L]*AAm*RHOMAG
        
        dxpldz = (xlpp - xlpm)/(2*dz)
        X[L] = -dxpldz
        Brhodm = Brhodm + X[L]*C[L] 
    
    # Now calculate BZD
    # Now corrected for the current sheet derivative term. 11/2002.
    for L in range(istartLoop, istopLoop+1) :
        rhom = RHOMAG - drho
        rhop = RHOMAG + drho
            
         # First calculate it for rhom   
        #rhos3m = rho - drho
        ZNM = 0
        ZM = np.abs(ZMAG - ZNM)   
        if (ZM < D) :
            ZM = 0.5*(ZM*ZM/D + D)
        
        # Now calculate it for rhop
        #rhos3p = rho + drho
        ZNP = 0
        ZP = np.abs(ZMAG - ZNP)
        if (ZP < D) :
            ZP = 0.5*(ZP*ZP/D + D)
        xlpp = 0
        xlpm = 0
        
        for i in range(0,isumno):
            S1p = np.sqrt( (beta[i,L] + ZP)*(beta[i,L] + ZP) + (rhop + beta[i,L])*(rhop + beta[i,L]) )                
            S2p = np.sqrt( (beta[i,L] + ZP)*(beta[i,L] + ZP) + (rhop - beta[i,L])*(rhop - beta[i,L]) )
            S1m = np.sqrt( (beta[i,L] + ZM)*(beta[i,L] + ZM) + (rhom + beta[i,L])*(rhom + beta[i,L]) )
            S2m = np.sqrt( (beta[i,L] + ZM)*(beta[i,L] + ZM) + (rhom - beta[i,L])*(rhom - beta[i,L]) )     
        
            tp = 2*beta[i,L]/(S1p + S2p)
            tm = 2*beta[i,L]/(S1m + S2m)
            AAp = tp*np.sqrt(1 - tp*tp)/(S1p*S2p)
            AAm = tm*np.sqrt(1 - tm*tm)/(S1m*S2m)
            xlpp = xlpp + f[i,L]*AAp*rhop
            xlpm = xlpm + f[i,L]*AAm*rhom
        
        dxpldr = (rhop*xlpp - rhom*xlpm)/(2*drho)
        X[L] = dxpldr/RHOMAG
        Bzdm = Bzdm + X[L]*C[L]
    
    Bpdm = 0    
    # Now add the stretch field to Bpdm   
    Bpdm = Bpdm - P*RHOMAG*Brhodm   
      
    # Now rotate into system III coordinates.
    # Bphi_N=0  !changed Hannes
    Brcs = Brhodm
    Bpcs = Bpdm
    Bzcs = Bzdm     
        
    return Brcs, Bpcs, Bzcs 
    

################# JUPITER INTERNAL MAGNETIC FIELD CALCULATION #################


def JRM33_no_dipole(r_in,t_in,p_in):  
    # Calculates only the non-dipolar part of Jupiter's magnetic field 
    # g10, g11 and h11 coefficients are set to zero here and defined in line 947
    # Dipolar contribution is then calculate by the current sheet model above 
    
    
    r_rj_dbl       = np.array([r_in])
    colat_rads_dbl = np.array([t_in])
    elong_rads_dbl = np.array([p_in])
    
    N_input = len(r_rj_dbl)
    
    # ============
    # Begin hard-coding for JRM33_ORDER13
    # Values from Connerney et al. (2021),  https://doi.org/10.1029/2021JE007055
    # See supplemental online information, file 2021JE007055-sup-0002-Supporting Information SI-S02.mod
    # ============
    
    # order = 13; # degree = order for this code 
    # k     = order + 1
    # k     = 14  # order + 1 
    k_plus1 = 15  # k+1, used in for loops when I want to go up to k 

    rec = np.array([0                         , \
                    0                         ,             0.33333333333333331482962 ,             0                         ,             0.26666666666666666296592 , \
                    0.20000000000000001110223 ,             0                         ,             0.25714285714285711748062 ,             0.22857142857142856429142 , \
                    0.14285714285714284921269 ,             0                         ,             0.25396825396825395415590 ,             0.23809523809523808202115 , \
                    0.19047619047619046561692 ,             0.11111111111111110494321 ,             0                         ,             0.25252525252525254151337 , \
                    0.24242424242424243097105 ,             0.21212121212121212709967 ,             0.16161616161616162989922 ,             0.09090909090909091161414 , \
                    0                         ,             0.25174825174825177231952 ,             0.24475524475524476630817 ,             0.22377622377622377602968 , \
                    0.18881118881118880148406 ,             0.13986013986013987042689 ,             0.07692307692307692734701 ,             0                         , \
                    0.25128205128205127749652 ,             0.24615384615384616751044 ,             0.23076923076923078204103 ,             0.20512820512820512108831 , \
                    0.16923076923076924016343 ,             0.12307692307692308375522 ,             0.06666666666666666574148 ,             0                         , \
                    0.25098039215686274161499 ,             0.24705882352941177515504 ,             0.23529411764705882026405 ,             0.21568627450980393245317 , \
                    0.18823529411764705621124 ,             0.15294117647058824704942 ,             0.10980392156862744945656 ,             0.05882352941176470506601 , \
                    0                         ,             0.25077399380804954454049 ,             0.24767801857585139413409 ,             0.23839009287925697067045 , \
                    0.22291021671826624639401 ,             0.20123839009287924906033 ,             0.17337461300309597866942 ,             0.13931888544891640746570 , \
                    0.09907120743034056320475 ,             0.05263157894736841813099 ,             0                         ,             0.25062656641604008633806 , \
                    0.24812030075187968547468 ,             0.24060150375939848288454 ,             0.22807017543859647856763 ,             0.21052631578947367252397 , \
                    0.18796992481203006475354 ,             0.16040100250626565525636 ,             0.12781954887218044403241 ,             0.09022556390977443108170 , \
                    0.04761904761904761640423 ,             0                         ,             0.25051759834368531043580 ,             0.24844720496894409644817 , \
                    0.24223602484472050999642 ,             0.23188405797101449556941 ,             0.21739130434782608092270 ,             0.19875776397515526605630 , \
                    0.17598343685300207872579 ,             0.14906832298136646342002 ,             0.11801242236024844789455 ,             0.08281573498964803214939 , \
                    0.04347826086956521618454 ,             0                         ,             0.25043478260869567186830 ,             0.24869565217391303990624 , \
                    0.24347826086956522728677 ,             0.23478260869565217849875 ,             0.22260869565217392129775 ,             0.20695652173913042792819 , \
                    0.18782608695652172614565 ,             0.16521739130434781595014 ,             0.13913043478260869734164 ,             0.10956521739130434256460 , \
                    0.07652173913043477937457 ,             0.04000000000000000083267 ,             0                         ,             0.25037037037037035425158 , \
                    0.24888888888888888173412 ,             0.24444444444444443642617 ,             0.23703703703703704608330 ,             0.22666666666666665519436 , \
                    0.21333333333333334702608 ,             0.19703703703703703831174 ,             0.17777777777777778456247 ,             0.15555555555555555802272 , \
                    0.13037037037037035869247 ,             0.10222222222222222820509 ,             0.07111111111111111104943 ,             0.03703703703703703498107 , \
                    0                         ], dtype='float64')
    
    # This is the modified g array, not the original g coefficients, and will be further modified.
    g = np.array([  0                         , \
                    0                         ,                                     0 ,                                     0 ,         17695.05000000000291038304567 , \
               -98679.09142913710093125700951 ,         41785.89893767991452477872372 ,          6998.25000000000000000000000 ,       -114784.31409169112157542258501 , \
                29815.77499244317732518538833 ,         -1177.79031452971298676857259 ,       -150508.75000000000000000000000 ,        -44719.03330350511532742530107 , \
                -9549.96682340441475389525294 ,        -22690.84741462402962497435510 ,        -13251.35315516754053533077240 ,       -143842.38750000001164153218269 , \
                42921.27286392349196830764413 ,        127570.69442965975031256675720 ,        -81633.02337143303884658962488 ,         -5645.04937803636312310118228 , \
                -3499.17460275232133426470682 ,       -302725.50000000000000000000000 ,        186906.53619596210774034261703 ,        182206.05752274760743603110313 , \
              -125020.45832024884293787181377 ,         14963.80719322771983570419252 ,          3624.24518831162868082174100 ,          5385.77113291791010851738974 , \
                 1606.06874999999990905052982 ,        190333.43941625900333747267723 ,       -205607.27056667389115318655968 ,        -31401.55576882112654857337475 , \
               -87130.35157315820106305181980 ,         18897.56891766996704973280430 ,         -6026.22614284811970719601959 ,          2395.37925024410378682659939 , \
               545441.66015625000000000000000 ,         88735.96875000000000000000000 ,       -389895.83466339309234172105789 ,         -3934.85946617597619479056448 , \
              -126906.14003801737271714955568 ,        -19304.97414549050154164433479 ,        -29414.75344365271303104236722 ,         -3601.55780058668460696935654 , \
                -1895.53694641443667023850139 ,        846519.78125000000000000000000 ,       -446765.73728798632510006427765 ,         31302.07698931798586272634566 , \
                64195.52501902574294945225120 ,        202541.11524796698358841240406 ,       -106827.25692624624934978783131 ,         24466.70976716701261466369033 , \
               -11500.67915122823251294903457 ,          3393.53795505362222684198059 ,         -1409.70572312946706006187014 ,       -454041.47851562500000000000000 , \
               458156.33419361733831465244293 ,        597543.35517407325096428394318 ,        703995.08215286664199084043503 ,        441386.13744140026392415165901 , \
                56501.09635237817565212026238 ,         87280.41578810986538883298635 ,         14527.30073426677881798241287 ,         20426.21477027422588435001671 , \
                 2230.28467025431518777622841 ,           700.12476547087453582207672 ,        451779.59531249996507540345192 ,       1425696.72961616632528603076935 , \
              -842884.25393627304583787918091 ,       1164319.22554796561598777770996 ,        141241.38706735140294767916203 ,         30860.91705950191681040450931 , \
              -101016.46548193244962021708488 ,        -25878.59093201200448675081134 ,         15597.45408843598124803975224 ,          3086.98667892851653959951364 , \
                 -888.46463288199674934730865 ,           631.48165122007844729523640 ,       1518777.06982421875000000000000 ,       1514541.16614053281955420970917 , \
              -231939.38036545089562423527241 ,        313703.49667393940035253763199 ,       -283336.19316209433600306510925 ,       -273363.07484114210819825530052 , \
              -228212.28592688051867298781872 ,       -232679.70717926806537434458733 ,         28356.30298026848322479054332 ,        -37627.76339165167155442759395 , \
                -6286.74167907329319859854877 ,         -2101.13289573513384311809205 ,          -125.36317707760989037524268 ,        954107.77587890625000000000000 , \
              -789653.82299604720901697874069 ,       1795741.50312742590904235839844 ,      -2961219.63433324638754129409790 ,        -75875.64023424909100867807865 , \
               -11760.33511734100466128438711 ,        268579.05760751606430858373642 ,       -253967.82629875544807873666286 ,        148831.17948716515093110501766 , \
               -43689.24707565712742507457733 ,          1180.95746522386184551578481 ,         -2299.43438245620791349210776 ,          -114.40525444356161699488439 , \
                 -160.23044573031521053962933 ], dtype='float64')
    
    # This is the modified h array, not the original h coefficients, and will be further modified.
    h = np.array([  0                         , \
                    0                         ,             0                         ,                                     0 ,             0                         , \
               -73697.02981124815414659678936 ,         17512.33270262702353647910058 ,             0                         ,       -100706.48416748049203306436539 , \
                82336.52755369272199459373951 ,        -21659.78366249880491523072124 ,             0                         ,        179591.12419298454187810420990 , \
               107370.50591299502411857247353 ,         -1048.75334326046368005336262 ,          -979.92466507316260049265111 ,             0                         , \
               461187.64343489689053967595100 ,         -6349.52304212686522078001872 ,         28240.09962311459821648895741 ,         23447.19900435869931243360043 , \
                 7079.80040974297844513785094 ,             0                         ,        264963.20919183432124555110931 ,       -151222.18075837698415853083134 , \
                -2938.03606418524441323825158 ,         76113.94933750208292622119188 ,         -8578.72983101319005072582513 ,          3212.77617244015573305659927 , \
                    0                         ,       -271512.72097865847172215580940 ,       -330112.68882052105618640780449 ,         44458.57413206643104786053300 , \
                65469.09192571333551313728094 ,        -40863.22971117807901464402676 ,         -4683.32440422766285337274894 ,         -3755.98417943188178469426930 , \
                    0                         ,       -153997.59375000000000000000000 ,       -719733.06207425787579268217087 ,        415007.55691891378955915570259 , \
               -46136.02057253671227954328060 ,         35402.09248114033107412979007 ,        -22223.42771144288417417556047 ,          2271.43559762761515230522491 , \
                -1991.86175917113155264814850 ,             0                         ,      -1006436.42691994667984545230865 ,       -144319.85027737272321246564388 , \
               544906.81881777732633054256439 ,        -71895.97914541495265439152718 ,        121863.52687899750890210270882 ,         -3389.11687594690647529205307 , \
                14767.96891140620027726981789 ,          2472.60829147248250592383556 ,          1115.59577164790675851691049 ,             0                         , \
             -1384249.10274789924733340740204 ,       1173617.20366917247883975505829 ,       -420076.51331166969612240791321 ,       -209806.54908645161776803433895 , \
               -16098.82101720037644554395229 ,        -25961.40947766388489981181920 ,        -32461.94297629480570321902633 ,          3719.80820997743148836889304 , \
                -2230.01919177910349390003830 ,           938.76318815977697340713348 ,             0                         ,        256092.79800851742038503289223 , \
              1852005.60425209673121571540833 ,      -1385008.89819696894846856594086 ,       -661513.62886436795815825462341 ,        -75409.63306120591005310416222 , \
              -255472.92826251508085988461971 ,        -66926.19299495992891024798155 ,        -37531.30277518012007931247354 ,        -12971.16190189409462618641555 , \
                -1585.96105624679739776183851 ,          -289.87374107255254784831777 ,             0                         ,       3771282.85399669967591762542725 , \
              1771549.32963395491242408752441 ,      -1234233.68693465669639408588409 ,       -618967.12278338952455669641495 ,        358538.51971466519171372056007 , \
              -231607.55816182776470668613911 ,        102176.33383478372707031667233 ,        -49039.72397764078777981922030 ,        -11821.41667073273674759548157 , \
                 -207.51435399791816394099442 ,           306.79766798991960286002723 ,          -404.64826224281051736397785 ,             0                         , \
              7138027.63398880977183580398560 ,        -68090.85327267039974685758352 ,       -584339.81344802852254360914230 ,       2126289.34228428779169917106628 , \
               177727.19461354395025409758091 ,        499002.54970085067907348275185 ,        324358.61438092868775129318237 ,         28001.33771220673952484503388 , \
               -14138.76070055155105364974588 ,         33944.01060500412131659686565 ,          4741.39154200049142673378810 ,          2048.50698775369892246089876 , \
                   28.56088209160934354713390 ], dtype='float64')
    
    # ============
    # End parts that are hard-coded for JRM33_ORDER13
    # ============
    

    a         = np.zeros((N_input,k_plus1),dtype='float64')
    DINDGEN_k = a.copy()
    for i in range(k_plus1):
        DINDGEN_k[:,i] = i
    
    da = np.float64(1)/r_rj_dbl
    
    
    a[:,0] = da
    for i in range(1,k_plus1):
        a[:,i] = a[:,i-1]*da
    
    b = a  * DINDGEN_k
    
    cos_phi   = np.cos(elong_rads_dbl,dtype='float64')
    sin_phi   = np.sin(elong_rads_dbl,dtype='float64')
    cos_theta = np.cos(colat_rads_dbl,dtype='float64')
    sin_theta = np.sin(colat_rads_dbl,dtype='float64')
    #not_bk = (sin_theta >= 0.00001 )  # = 1d-5 - also see bk both times below

    bk = (sin_theta <  0.00001 )
    zero_array = np.zeros(N_input,dtype='float64')
    p   = zero_array + np.float64(1)
    d   = zero_array.copy()
    bbr = zero_array.copy()
    bbt = zero_array.copy()
    bbf = zero_array.copy()
    x = zero_array.copy()
    y = p.copy() # 1s
    
    for m in range(1, k_plus1):
        bm  = (m != 1)
        if bm:
            m_minus_1 = np.float64(m - 1)
            w = x.copy()
            x = w *cos_phi + y *sin_phi
            y = y *cos_phi - w *sin_phi
        q = p.copy()
        z = d.copy()
        bi = zero_array.copy()
        p2 = zero_array.copy()
        d2 = zero_array.copy()
        for n in range(m, k_plus1):
            mn = int( n*(n-1)/2 + m )
            w  = g[mn]*y + h[mn]*x
            bbr += b[:,n]*w*q
            bbt -= a[:,n]*w*z
            if bm:
                qq = q.copy()
                ind = np.where(bk)[0]
                if (len(ind) != 0):
                    qq[ind] = z[ind]
                bi += a[:,n] * (g[mn]*x-h[mn]*y) * qq
            xk = rec[mn] # faster to write this to xk, to use below twice
            dp = cos_theta *z - sin_theta *q - d2*xk
            pm = cos_theta *q                - p2*xk
            d2 = z.copy()
            p2 = q.copy()
            z = dp.copy()
            q = pm.copy()
        d = sin_theta *d + cos_theta *p
        p = sin_theta *p
        if bm:
            bi  *= m_minus_1
            bbf += bi
    

    bf = bbf.copy() # set size of array and do the 3rd case
    ind = np.where((bk == 1) & (cos_theta < 0))[0]
    if (len(ind) != 0):
        bf[ind] = -bbf[ind]
    ind = np.where(bk == 0)[0]
    if (len(ind) != 0):
        bf[ind] =  bbf[ind]/sin_theta[ind]
            
            

    BR=bbr
    BT=bbt
    BP=bf

    return BR[0], BT[0], BP[0] 


############################################################################### 
#                          MAIN PROGRAM                                       # 
############################################################################### 


def kk_2005(t,x1,x2,x3,xyz_in,Bxyz_Out):
    # ----------------------------------------------------------- # 
    # Translated from procedure kk_2009
    #
    # Calling the functions to calculate the total magnetic field 
    # 
    # Adapted to have inputs coordinates either in spherical or 
    # cartesian coordinates (S3RH)
    #   - xyz_in = True : x,y,z (Rj)
    #   - xyz_in = False : r,t,p (Rj,deg,deg)
    #
    #   - Bxyz_out = True : Bx,By,Bz (nT)
    #   - Bxyz_out = False : Br, Bt, Bp (nT)
    # ----------------------------------------------------------- #  


    # constants
    parmod = np.zeros(10)
    M = 8
    pi = np.pi
    Mode = 7

    """
    # Not used :   
    M1 = 64
    M2 = 16
    radian = pi/180
    """


    if xyz_in == True :
        xs3 = x1 
        ys3 = x2
        zs3 = x3 
     
        r, theta, phi = CAR2SPH_pos(xs3, ys3, zs3)
        rho = np.sqrt(xs3*xs3 + ys3*ys3)    
        
    if xyz_in == False : 
        r = x1
        theta = np.radians(x2)
        phi = np.radians(x3)
        
        rho = r*np.sin(theta)
        xs3 = rho*np.cos(phi)
        ys3 = rho*np.sin(phi)
        zs3 = r*np.cos(theta)

    
    # given time, stheta, sphi, and phase are returned
    stheta, sphi, phase = JSUN(t) 

    
    
    # Now define the nine derivatives
    # These are calculated at the original location
    # First get the unit normals of the zp axis
    RNx, RNy, RNz = csheet_N(t,rho,phi,zs3,stheta)

    
    # The z axis of the mapped location is given by
    zpx = RNx
    zpy = RNy
    zpz = RNz
    zp = xs3*zpx + ys3*zpy + zs3*zpz
    
    # The y axis of the mapped location is given by
    ypx = zpy
    ypy = -zpx
    ypz = 0
    yp1 = np.sqrt(ypx*ypx + ypy*ypy + ypz*ypz)
    ypx = ypx/yp1
    ypy = ypy/yp1
    ypz = ypz/yp1
    yp = xs3*ypx + ys3*ypy + zs3*ypz
    

    
    # The x axis of the mapped location is given by
    xpx = ypy*zpz - ypz*zpy
    xpy = ypz*zpx - ypx*zpz
    xpz = ypx*zpy - ypy*zpx
    xp = xs3*xpx + ys3*xpy + zs3*xpz

    # Now define the nine derivatives
    dxpdx = xpx
    dypdx = ypx
    dzpdx = zpx
    dxpdy = xpy
    dypdy = ypy
    dzpdy = zpy
    dxpdz = xpz
    dypdz = ypz
    dzpdz = zpz
    
    R = np.sqrt(xs3*xs3 + ys3*ys3 + zs3*zs3)
    Rp = np.sqrt(xp*xp + yp*yp + zp*zp)

    # Now calculate the T matrix
    Txx = (dypdy*dzpdz - dypdz*dzpdy)
    Txy = (dxpdz*dzpdy - dxpdy*dzpdz)
    Txz = (dxpdy*dypdz - dxpdz*dypdy)
    Tyx = (dypdz*dzpdx - dypdx*dzpdz)
    Tyy = (dxpdx*dzpdz - dxpdz*dzpdx)
    Tyz = (dxpdz*dypdx - dxpdx*dypdz)
    Tzx = (dypdx*dzpdy - dypdy*dzpdx)
    Tzy = (dxpdy*dzpdx - dxpdx*dzpdy)
    Tzz = (dxpdx*dypdy - dxpdy*dypdx)
    
    
    # The mapped location is:
    vecin = [xs3, ys3, zs3]
    vecou = JROT(t,vecin,'s3c', 'jso')
    
    XJSO = vecou[0]
    YJSO = vecou[1]
    ZJSO = vecou[2]
    
    
    ZNS3 = csheet_struc(rho,phi,stheta,XJSO,YJSO)
    
    RLT = np.arctan2(YJSO, XJSO)
    rmap = np.sqrt(xp*xp + yp*yp)
    pmap = np.arctan2(yp, xp)
    zmap = zs3 - ZNS3

    # Now calculate the field at the mapped location in cylindrical coordinates
    scol = pi/2 - stheta
    
    scolOut, sphiOut = get_mapped_sunangle(scol, sphi, xpx, xpy, xpz, ypx, ypy, ypz, zpx, zpy, zpz)    
    
    Brds, Bpds, Bzds = dipole_shiel_cyl_S3(rmap, pmap, zmap, parmod, sphiOut)
    
    Brcs, Bpcs, Bzcs = tail_mag_notilt_all_modes(rmap, pmap, zmap, RLT, Mode)
    
    Brcss, Bpcss, Bzcss = tail_mag_shield_cyl_S3(rmap, pmap, zmap, sphiOut, M, Mode)
    
    Brmap = Brds + Brcs + Brcss
    Bpmap = Bpds + Bpcs + Bpcss
    Bzmap = Bzds + Bzcs + Bzcss
     
    BXcarMap, BYcarMap, BZcarMap = cyl2car_vect(pmap, Brmap, Bpmap, Bzmap)
    BXdsMap, BYdsMap, BZdsMap = cyl2car_vect(pmap, Brds, Bpds, Bzds)
    BXcsMap, BYcsMap, BZcsMap = cyl2car_vect(pmap, Brcs, Bpcs, Bzcs)
    BXcssMap, BYcssMap, BZcssMap = cyl2car_vect(pmap,Brcss, Bpcss, Bzcss)
    
    Bxfinal =  Txx*BXcarMap + Txy*BYcarMap + Txz*BZcarMap
    Byfinal =  Tyx*BXcarMap + Tyy*BYcarMap + Tyz*BZcarMap
    Bzfinal =  Tzx*BXcarMap + Tzy*BYcarMap + Tzz*BZcarMap

    Bxdsfinal =  Txx*BXdsMap + Txy*BYdsMap + Txz*BZdsMap
    Bydsfinal =  Tyx*BXdsMap + Tyy*BYdsMap + Tyz*BZdsMap
    Bzdsfinal =  Tzx*BXdsMap + Tzy*BYdsMap + Tzz*BZdsMap

    Bxcsfinal =  Txx*BXcsMap + Txy*BYcsMap + Txz*BZcsMap
    Bycsfinal =  Tyx*BXcsMap + Tyy*BYcsMap + Tyz*BZcsMap
    Bzcsfinal =  Tzx*BXcsMap + Tzy*BYcsMap + Tzz*BZcsMap

    Bxcssfinal =  Txx*BXcssMap + Txy*BYcssMap + Txz*BZcssMap
    Bycssfinal =  Tyx*BXcssMap + Tyy*BYcssMap + Tyz*BZcssMap
    Bzcssfinal =  Tzx*BXcssMap + Tzy*BYcssMap + Tzz*BZcssMap
    
    
    Brm, Btm, Bpm = CAR2SPH_MAG(theta,phi, Bxfinal, Byfinal, Bzfinal)
    Brdsm, Btdsm, Bpdsm = CAR2SPH_MAG(theta, phi, Bxdsfinal, Bydsfinal, Bzdsfinal)
    Brcsm, Btcsm, Bpcsm = CAR2SPH_MAG(theta, phi, Bxcsfinal, Bycsfinal, Bzcsfinal)   
    Brcssm, Btcssm, Bpcssm = CAR2SPH_MAG(theta, phi, Bxcssfinal, Bycssfinal, Bzcssfinal)
    
    BR, BT, BF = JRM33_no_dipole(r, theta, phi)
    
    Brm = Brm + BR
    Btm = Btm + BT
    Bpm = Bpm + BF

    Brdsm = Brdsm + BR
    Btdsm = Btdsm + BT
    Bpdsm = Bpdsm + BF
    
    
    # Now calculate the solar wind magnetic field
    BrS3IMF, BtS3IMF, BpS3IMF = getBimf(t, theta, phi)
    
    mpCheck = CheckIfInsideMappedMp(t, xs3, ys3, zs3, ZNS3)
    
    if (mpCheck == 1) :
        Brm = Brm + BrS3IMF
        Btm = Btm + BtS3IMF
        Bpm = Bpm + BpS3IMF
    else :
        Brm = BrS3IMF
        Btm = BtS3IMF
        Bpm = BpS3IMF
      
    if Bxyz_Out == False :    
        return Brm, Btm, Bpm, mpCheck
    
    if Bxyz_Out == True : 
        Bx, By, Bz = SPH2CAR_MAG(theta, phi, Brm, Btm, Bpm)
        
        return Bx, By, Bz, mpCheck
    
    
    
    
    
    
    