function [Ap,Bp,Cp,Ep] = Loewner_sylvester(s,Hs,nmax,epsilon)
% Calculates Loewner interpolant to the data in s, Hs
% Assumes data comes in complex conjugate pairs.  Solves for system
% matrices via two sylvester equations.
%%%%%%%%%%%% Inputs %%%%%%%%%%%%
% s: interpolation points (must be even length)
% Hs: values of H at s
% nmax: maximal allowed model order
% epsilon: tolerance for determining rank of Loewner pencil

%%%%%%%%%%%% Outputs %%%%%%%%%%%%
% Ap Bp Cp Ep are system matrices such that the function
%     H(x) = Cp*((x*Ep-Ap)^(-1)*Bp
% satisfies
%     H(s(i)) = Hs(i)
% returned matrices may not exactly interpolate data if the model is
% truncated to an order p < r = length(s) model, due to rank deficiences in
% A and E matrices.

%  References:
%  	A. J. Mayo, A. C. Antoulas, A framework for the solution of the 
%     generalized realization problem, Linear Algebra and its Applications
%     425, pp. 634?662 (2007). (DOI: 10.1016/j.laa.2007.03.008)

%   A. Antoulas, C. Beattie, and Gugercin, Interpolary Methods for Model 
%     Reduction, Computational Science and Engineering, SIAM, 2020.



r = length(s)/2;
s1 = s(1:r); s2 = s(r+1:end);
Hs1 = Hs(1:r); Hs2 = Hs(r+1:end);

% Data must come in complex pairs!
s1_cplx_check = sum(abs(imag(s1(1:2:end)) + imag(s1(2:2:end)))) > 0;
s2_cplx_check = sum(abs(imag(s2(1:2:end)) + imag(s2(2:2:end)))) > 0;
Hs1_cplx_check = sum(abs(imag(Hs1(1:2:end)) + imag(Hs1(2:2:end)))) > 0;
Hs2_cplx_check = sum(abs(imag(Hs2(1:2:end)) + imag(Hs2(2:2:end)))) > 0;

if s1_cplx_check || s2_cplx_check || Hs1_cplx_check || Hs2_cplx_check
    fprintf('Data was not in complex pairs!\n')
end

% Solve via Sylvester equations
Z_R = [];
Y_R = [];
Sigma_R = [];
Theta_R = [];
B_R = [];
C_R = [];
index = 1;
while length(Sigma_R) < r
    if isreal(s1(index))
        Sigma_R = blkdiag(Sigma_R,s1(index));
        Y_R = [Y_R,Hs1(index)];
        B_R = [B_R,1];
        index = index + 1;
    else
        T = [real(s1(index)), imag(s1(index)); -imag(s1(index)), real(s1(index))];
        Sigma_R = blkdiag(Sigma_R,T);
        Y_R = [Y_R,[real(Hs1(index)),imag(Hs1(index))]];
        B_R = [B_R,[1,0]];
        index = index + 2;
    end
end
index = 1;
while length(Theta_R) < r
    if isreal(s2(index))
        Theta_R = blkdiag(Theta_R,s2(index));
        Z_R = [Z_R,Hs2(index)];
        C_R = [C_R,1];
        index = index + 1;
    else
        T = [real(s2(index)), imag(s2(index)); -imag(s2(index)), real(s2(index))];
        Theta_R = blkdiag(Theta_R,T);
        Z_R = [Z_R,[real(Hs2(index)),-imag(Hs2(index))]];
        C_R = [C_R,[1,0]];
        index = index + 2;
    end
end

X_L = (C_R.'*Y_R-Z_R.'*B_R);
X_M = (C_R.'*Y_R*Sigma_R-Theta_R*Z_R.'*B_R);

L_R = sylvester(-Theta_R,Sigma_R,X_L);

M_R = sylvester(-Theta_R,Sigma_R,X_M);



%%%%%%%%%%%%%%%%%%%%%
B = reshape(Z_R,r,1);
C = reshape(Y_R,1,r);
E = -L_R;
A = -M_R;



%%%%%%%%%%%%%%%%%%%%%
%% remove redundencies
[Y,theta1,~] = svd([E A]);
[~,theta2,X] = svd([E;A]);

sVal1 = diag(theta1);
sValScaled1 = sVal1/sVal1(1);
sVal2 = diag(theta2);
sValScaled2 = sVal2/sVal2(1);

pHat1 = find(sValScaled1 < epsilon,1)-1;
pHat2 = find(sValScaled2 < epsilon,1)-1;
p = min([pHat1,pHat2,nmax]);%truncate at max allowed degree or at tolerence level

%only remove reduncencies if they are present
if p < r
    Ep = Y(:,1:p)'*E*X(:,1:p);
    Ap = Y(:,1:p)'*A*X(:,1:p);
    Bp = Y(:,1:p)'*B;
    Cp = C*X(:,1:p);
else
    Ep = E;
    Ap = A;
    Bp = B;
    Cp = C;
end