
function [Ap,Bp,Cp,Ep] = HermiteLoewner(s,Hs,Hps,nmax,epsilon)
% computes Hermite Loewner interpolant to the data is Hs and Hps at s.
%%%%%%%%%%%% INPUTS %%%%%%%%%%%%
% s:  sample points
% Hs:  values at s
% Hps:  derivative values at s
% nmax:  maximal order of resulting model
% epsilon: tolerance for determining rank of Loewner pencil
%%%%%%%%%%%% Outputs %%%%%%%%%%%%
% Ap Bp Cp Ep are system matrices such that the function
%     H(x) = Cp*((x*Ep-Ap)^(-1)*Bp
% satisfies
%     H(s(i)) = Hs(i)   and   H'(s(i)) = Hps(s(i)).
% returned matrices may not exactly interpolate data if the model is
% truncated to an order p < r = length(s) model, due to rank deficiences in
% A and E matrices.

%  References:
%  	A. J. Mayo, A. C. Antoulas, A framework for the solution of the 
%     generalized realization problem, Linear Algebra and its Applications
%     425, pp. 634?662 (2007). (DOI: 10.1016/j.laa.2007.03.008)

%   A. Antoulas, C. Beattie, and Gugercin, Interpolary Methods for Model 
%     Reduction, Computational Science and Engineering, SIAM, 2020.


r = length(s);

% reshape to column vectors

s = reshape(s,[r 1]);
Hs = reshape(Hs,[r 1]);
Hps = reshape(Hps,[r 1]);

% sort in to complex pairs if possible
try
    s_temp = cplxpair(s);
    [~,idx] = ismember(s_temp,s);
    s = s(idx);
    Hs = Hs(idx);
    Hps = Hps(idx);
    is_real = true;
catch
    warning('Interpolation points not closed under conjugation, realization will be complex.')
    is_real = false;
end

Mi = zeros(r); Li = zeros(r);
Bi = Hs;
Ci = Hs.';

for i = 1:r
    for j = 1:r
        if s(i) == s(j)
            Li(i,j) = Hps(i);
            Mi(i,j) = (Hs(i)+s(i)*Hps(i));
        else
            Li(i,j) = (Hs(i)-Hs(j))/(s(i)-s(j));
            Mi(i,j) = (s(i)*Hs(i)-s(j)*Hs(j))/(s(i)-s(j));
        end
    end
end

Ei = -Li;
Ai = -Mi;

%% Keep realization real if possible
if is_real
    Tblk = 1/sqrt(2)*[1 -1i; 1 1i];
    
    index = 1;
    T_blks = {};
    while index <= r
        if index + 1 <= r && s(index) == conj(s(index + 1))
            T_blks = [T_blks Tblk];
            index = index + 2;
        else
            T_blks = [T_blks 1];
            index = index + 1;
        end
    end
    
    T = blkdiag(T_blks{:});
    
    E = real(T'*Ei*T);
    A = real(T'*Ai*T);
    B = real(T'*Bi);
    C = real(Ci*T);
else
    E = Ei;
    A = Ai;
    B = Bi;
    C = Ci;
end

%% remove redundencies
[Y,theta1,~] = svd([E A]);
[~,theta2,X] = svd([E;A]);

sVal1 = diag(theta1);
sValScaled1 = sVal1/sVal1(1);
sVal2 = diag(theta2);
sValScaled2 = sVal2/sVal2(1);

pHat1 = find(sValScaled1 < epsilon,1)-1;
pHat2 = find(sValScaled2 < epsilon,1)-1;
p = min([pHat1,pHat2,nmax]);%truncate at max allowed degree or at tolerence level

%only remove reduncencies if they are present
if p < r
    Ep = Y(:,1:p)'*E*X(:,1:p);
    Ap = Y(:,1:p)'*A*X(:,1:p);
    Bp = Y(:,1:p)'*B;
    Cp = C*X(:,1:p);
else
    Ep = E;
    Ap = A;
    Bp = B;
    Cp = C;
end
