
function [Mj,cond_num,res] = moment_match(s,n,W,k,tau1,tau2,skip_cond_comp)
% Function to find the value of the transfer function of dynamical system
% from time domain data 

%%%%%% INPUTS %%%%%
% s: point in C to recover H(s)
% W: is an orthogonal subspace for the current window of data (see
% subsection 3.1
% n is (approximate) order of the system
% k number of moments to match
% tau1: tolerance on uniqueness condition
% tau2: tolerance on existence condition
% skip_cond_comp: skips computation of condition numbers of linear systems
%           used to solve for frequency data.  Recommended if n is large

%%%%% OUTPUTS %%%%%
% Mj:  the values and derivatives recovered
% cond_num: the condition number of the linear system coefficient matrix
%           used to calculate frequency data
% res: the relative residual of the solution to the linear system (for
%           recovering value of transfer function only)

% This file is part of Code and Data for the numerical experiments in 
% "Frequency-Based Reduced Models from Purely Time-Domain Data via Data Informativity"
% Copyright (C) 2023 Michael S. Ackermann
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%% Check interpolation conditions
% Check existence and uniqueness conditions
gamma_sig = calc_gamma(s,n,0);
b = [gamma_sig;zeros(n+1,1)];
z = [zeros(n+1,1);gamma_sig];

[unique_cond, exist_cond] = check_rank_conds(W,z,b,tau1,tau2);
%if we cant interpolate, quit
if ~unique_cond || ~exist_cond
    Mj = NaN(k+1,1);
    cond_num = NaN;
    res = NaN;
    return
end

%% Calculate value of transfer function
% fast QR decomposition
[Q,R] = orthogonalize_Wz(W,-z);
x = R\(Q'*b);
M0 = x(end);

%Calculate the relative residual
res = norm([W -z]*x - b)/norm(b);
if ~skip_cond_comp
    A = [W -z];
    cond_num = cond(A);
else
    cond_num = nan;
end

%% Calculate derivatives
Mj = NaN(k+1,1);
Mj(1) = M0;

for i = 1:k
    % Make the right hand side of the linear system
    b = Make_RHS(i,Mj(1:i),n,s);
    % uniqueness condition is same as for value calculation
    [~, exist_cond_der] = check_rank_conds(W,z,b,tau1,tau2);
    %if Mk is not unique, quit
    if ~exist_cond_der
        return
    end
    x = R\(Q'*b);
    Mi = x(end);
    Mj(i+1) = Mi;
end