function [Hs,n_std_Hs,cond_nums,residuals,opts] = ...
    CalculateTFVals(U,Y,z_vec,opts)
% Finds values (and derivatives) of underlying dynamical system that
% produces output Y given input U.

%%%%%%%%% INPUTS %%%%%%%%%
% U: vector of time domain input
% Y: vector of time domain output corresponding to U
% z_vec:  vector of complex numbers where we would like to learn frequency
%      informaion
% opts.der_order:  number of derivatives to match
% opts.num_est:  number of estimates of frequency information to calculate
% opts.num_windows:  total number of estimates to the transfer function
%          value to calculate (K in paper)
% opts.num_windows_keep:  subset of total windows to use for calculating
%          transfer function value (W in paper)
% opts.tau1:  tolerance on uniqueness rank condition
% opts.tau2:  tolerance on existence rank condition
% opts.n:  is (an approximation of) order of underlying dynamical system
% opts.skip_condition_numbers:  skips computation of linear system condition
%          numbers (for large n, this is a computational bottle neck)
% opts.t0:  first component in U and Y to use to calculate moments.
%          Increase to reduce influence of transient response
% opts.W:  cell array of precomputed orthogonal bases for range of data
%          windows

%%%%%%%%% OUTPUTS %%%%%%%%%
% Hs:  discovered frequency information
% n_std_hs:  normalized standard deviation in the estimates for components
%          of Hs (s_W in paper)
% cond_nums:  condition numbers of the matrix used to calculate values of
%          transfer function of underlying dynamical system
% residuals:  vector of average residuals from linear system used to
%           calculate values of transfer function of underlying dynamical
%           system

% This file is part of Code and Data for the numerical experiments in 
% "Frequency-Based Reduced Models from Purely Time-Domain Data via Data Informativity"
% Copyright (C) 2023 Michael S. Ackermann
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% set default option values
default.der_order = 0;
default.num_windows = 20;
default.num_windows_keep = 10;
default.n = nan;
default.tau1 = 10^-10;
default.tau2 = 10^-10;
default.t0 = 1;
default.W = nan;
default.skip_condition_numbers = false;
if nargin<4
    opts=default;
else
    %Merge default values into opts
    names=fieldnames(default);
    for m=1:length(names)
        if ~isfield(opts,names(m))
            opts.(names{m}) = default.(names{m});
        end
    end
end
% if n is not supplied, discover N using MOESP
if isnan(opts.n)
    if length(U) > 1000
        n = MOESP(U(1:1000),Y(1:1000));
    else
        n = MOESP(U,Y);
    end
else
    n = opts.n;
end
fprintf('Using n = %d\n',n)
num_windows = opts.num_windows;
num_windows_keep = opts.num_windows_keep;
t = 3*n;
T = length(U);


%% Precompute Orthogonal Subspaces
if iscell(opts.W)
    W_cell = opts.W;
elseif num_windows == 1
    Hu = HankMat(U(1:t+1),n);
    Hy = HankMat(Y(1:t+1),n);
    W = orth([Hu;Hy]);
    W_cell{1} = W;
else
    t0 = opts.t0;
    num_skip = floor((T-t-t0)/num_windows);
    %cell array for storing the orthogonal bases for each window
    W_cell = cell(num_windows,1);
    count = 1;
    for t_start = t0:num_skip:T-t
        t_end = t_start + t;
        U_i = U(t_start:t_end);
        Y_i = Y(t_start:t_end);
    
        Hu = HankMat(U_i,n);
        Hy = HankMat(Y_i,n);
        W = orth([Hu;Hy]);
        W_cell{count} = W;
        count = count + 1;
    end
end
opts.W = W_cell;
%% Calculate the transfer function values and derivatives
skip_cond_comp = opts.skip_condition_numbers;
num = length(z_vec);
der_order = opts.der_order;
tau1 = opts.tau1;
tau2 = opts.tau2;
Hs = zeros(num,1 + der_order); %stores accepted TF vals and derivatives
n_std_Hs = zeros(num,1 + der_order); %stores average normalized standard devs
cond_nums = zeros(num,1); %stores average condition numbers of coefficient matrix [W z]
residuals = zeros(num,1); %stores the average residual of the LS problem
for k = 1:num
    z = z_vec(k);
    Mj_est_vec = zeros(num_windows,1+der_order); %stores all estimates of val and ders
    cond_vec = zeros(num_windows,1); %stores all condition numbers
    res_vec = zeros(num_windows,1); %Least Squares residual
    parfor current_SS = 1:num_windows
        W = W_cell{current_SS}; %get precomputed subspace
        %calculate estimates
        [Mj, cond_num,res] = moment_match(z,n,W,der_order,tau1,tau2,skip_cond_comp);
        Mj_est_vec(current_SS,:) = Mj.';
        cond_vec(current_SS) = cond_num;
        res_vec(current_SS) = res;
    end

    for i = 0:der_order
        %take out all instances where we failed to calculate a moment
        idx = ~isnan(Mj_est_vec(:,i+1));
        thin_data = Mj_est_vec(idx,i+1);
        thin_cond = cond_vec(idx);
        thin_res = res_vec(idx);
        if isempty(thin_data)
            thin_data = NaN;
            thin_cond = NaN;
        else
            % Accept maximally num_windows_keep estimates
            num_to_accept = min([num_windows_keep,length(thin_data)]);
            [~,accepted_res] = sort(thin_res);
            accepted_res = accepted_res(1:num_to_accept);
            thin_data = thin_data(accepted_res);
            thin_res = thin_res(accepted_res);
            thin_cond = thin_cond(accepted_res);
        end        
        if isempty(thin_data)
            avg_Mj = NaN;
            avg_cond = NaN;
            avg_res = NaN;
            std_norm = NaN;
        else
            %set the accepted M0 to be the mean of the calculated M0 values            
            avg_Mj = mean(thin_data);
            std_dev = std(thin_data);
            std_norm = std_dev/abs(avg_Mj);
            avg_cond = mean(thin_cond);
            avg_res = mean(thin_res);
        end
        cond_nums(k) = avg_cond;
        residuals(k) = avg_res;
        Hs(k,i+1) = avg_Mj;
        n_std_Hs(k,i+1) = std_norm;     
    end
end

