% demo_msd_polyAA.m -> a demo file used to test the modified 
% Antoulas-Anderson (AA) approach (for SISO systems with TFs that have 
% polynomial terms), applied to the MSD test case
%
% Last revised: October 31, 2023
% 
% Author : Ion Victor Gosea, CSC Group, Max Planck Magdeburg
%

% clean up
clear all, close all, format compact, format short e, clc
% setup plots
set(0,'DefaultFigurePosition', [100 100 1000 400]);
set(0,'defaultlinelinewidth',3)
set(0,'defaultlinemarkersize',20)
set(0,'defaultaxesfontsize',24)
% set(0,'defaulttextinterpreter','latex')

problem = 'NSE';
problem = 'MSD';

[H, m, p, P0, P1] = problem_setup(problem);

if strcmp(problem, 'MSD')
    disp('MSD problem')

elseif strcmp(problem, 'NSE')
    disp('NSE problem')
end

N = 60;
w = logspace(-3,5,N)*1i;

FR = zeros(1,N);

for ii = 1:N
    FR(ii) = H(w(ii));
end

figure(1);
loglog(imag(w),abs(FR),'k-.');
title('Frequency response');
xlabel('Frequency');
ylabel('Magnitude');
axis tight;

wr = w(1:5:end);
wl = setdiff(w,wr);

make_real = 1;

if(make_real)
    wrc = zeros(1,2*length(wr));
    wrc(1:2:end) = wr;
    wrc(2:2:end) = conj(wr);
    
    wlc = zeros(1,2*length(wl));
    wlc(1:2:end) = wl;
    wlc(2:2:end) = conj(wl);
else
    wrc = wr;
    wlc = wl;
end

LL = zeros(length(wlc),length(wrc));

tic;
% %% precomputing the H evaluations
Hwlc = NaN(size(wlc));
Hwrc = NaN(size(wrc));
for ii = 1:length(wlc)
    Hwlc(ii) = H(wlc(ii));
end
for jj = 1:length(wrc)
    Hwrc(jj) = H(wrc(jj));
end
% %% generate the Loewner matrix
for ii = 1:length(wlc)
    for jj = 1:length(wrc)
        LL(ii,jj) = (Hwlc(ii)-Hwrc(jj))/(wlc(ii)-wrc(jj));
    end
end
TLL = toc;
fprintf('time for set up Loewner = %d H evaluations: %.2e\n', N, TLL)

LL = [LL -ones(length(wlc),1)];

ss = svd(LL); ss = ss/ss(1);

figure(2);
semilogy(ss,'r.--','markersize',36);
axis tight;
title('Singular value decay of the Loewner matrix');

[U,S,V] = svd(LL);

tol = 1e-14;

indx = length(find(ss>tol));

sing_vec = V(:,indx);

weights = sing_vec(1:length(wrc));
b_coeff = sing_vec(end);

h = zeros(1,length(wrc));

for ii = 1:length(wrc)
    h(ii) = Hwrc(ii);
end

P0
P0_est = (h*weights-b_coeff*sum(wrc))/sum(weights)

P1
P1_est = b_coeff/sum(weights)


% reconstruct the fitted transfer function
N = @(s) b_coeff+(h./(s-wrc))*weights;
D = @(s) (1./(s-wrc))*weights;

freq = logspace(-4,6,300)*1i;

for ii = 1:300
    Horig(ii) = H(freq(ii));
    Hfit(ii) = N(freq(ii))/D(freq(ii));
end

figure(3);
loglog(imag(freq),abs(Horig),'b-'); hold on;
loglog(imag(freq),abs(Hfit),'y-.');
legend('original','fitted');
xlabel('Frequency');
ylabel('Magnitude');
axis tight;

figure(4);
loglog(imag(freq),abs(Horig),'b-'); hold on;
loglog(imag(freq),abs(Hfit),'y-.');
loglog(imag(freq),abs(Horig-Hfit)./abs(Horig),'g--');
legend('original','fitted','error');
xlabel('Frequency');
ylabel('Magnitude');
axis tight;

save(['data/CAA_' problem '.mat'], 'Horig', 'Hfit', 'freq');
