% demo_msd_Loewner.m -> a demo file used to test the modified 
% Loewner linear framework (for MIMO systems), applied to the MSD test case
% A post-processing algorithm is applied to the data in
% order to extract the (estimated) polynomial coefficient matrices.
%
% Last revised: October 31, 2023
% 
% Author : Ion Victor Gosea, CSC Group, Max Planck Magdeburg
%

% clean up
clear 
close
% setup plots
set(0,'DefaultFigurePosition', [100 100 1000 400]);
set(0,'defaultlinelinewidth',3)
set(0,'defaultlinemarkersize',20)
set(0,'defaultaxesfontsize',20)
% set(0,'defaulttextinterpreter','latex')

% set(0,'DefaultFigureColormap', turbo)
syscolors = parula(6);  % FOM, Loewner, Loewner-new
syscolors = syscolors(3:5, :);
errcolors = cool(2);  % Loewner, Loewner-new
svcolors = hot(1);  % for SV plots


problem = 'NSE';
problem = 'MSD';

[H_true, m, p, P0, P1] = problem_setup(problem);

P_true{1} = P0;
P_true{2} = P1;
P_true{3} = zeros(p,m);

if  strcmp(problem,'MSD')
    % number of interpolation points for the high/low frequency range
    nLhi = 20;
    nLlo = 200;

elseif  strcmp(problem,'NSE')
    % number of interpolation points for the high/low frequency range
    nLhi = 10;
    nLlo = 40;
end

% high frequency range 
omega_hi = logspace(7,9,nLhi)*1i;

H_hi = zeros(p,m, nLhi);

% sample the transfer function on the frequency range omega
for kk = 1:nLhi
    H_hi(:,:,kk) = H_true(omega_hi(kk));
end

figure
colororder(syscolors)

loglog(imag(omega_hi),reshape(abs(H_hi(1,1,:)),1, nLhi),'b.-','markersize',6)
title('Frequency response'); 
axis tight

% Loewner procedure in short - using the routine loewner.m
[LL_hi,sLL_hi,~,~,V_hi,W_hi,L,R] = loewner(omega_hi,reshape(H_hi(1,1,:),1, nLhi));

[~,sing1_hi,~]=svd([LL_hi sLL_hi],'econ');
[~,sing2_hi,~]=svd([LL_hi;sLL_hi]);

ss1 = diag(sing1_hi); ss1 = ss1/ss1(1);
ss2 = diag(sing2_hi); ss2 = ss2/ss2(1);

figure;
colororder(svcolors)
semilogy(ss1,'k.-','markersize',20); hold on;
semilogy(ss2,'k--','markersize',12);
%title('Singular value decay of the Loewner matrix - high frequency range');
grid on;axis tight;

P_est{2} = pinv(L)*LL_hi*pinv(R);
P_est{1} = real(pinv(L)*sLL_hi*pinv(R));
P_est{3} = zeros(p,m);

norm(P_est{1}-P_true{1})
norm(P_est{2}-P_true{2})

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% low frequency range 
omega_lo = logspace(-4, 2, nLlo)*j;

H_lo = zeros(p,m,length(omega_lo));

for jj = 1:length(omega_lo)
    H_lo(:,:,jj) =  H_true(omega_lo(jj));  % C*((omega_lo(jj)*E-A)\B)+P_true{2}*omega_lo(jj);
end


[LL_lo,sLL_lo,mu,la,V_lo,W_lo] = loewner(omega_lo,reshape(H_lo(1,1,:),1,length(omega_lo)),'real');

[u1,sing1_lo,~]=svd([LL_lo sLL_lo],'econ');
[~,sing2_lo,v2]=svd([LL_lo;sLL_lo]);

ss1 = diag(sing1_lo); ss1 = ss1/ss1(1);
ss2 = diag(sing2_lo); ss2 = ss2/ss2(1);

figure;
colororder(svcolors)
semilogy(ss1,'m.-','markersize',20); hold on;
semilogy(ss2,'m--','markersize',12);
%title('Singular value decay of the Loewner matrices - low frequency range');
grid on;axis tight;

%tol = 10^(-15);
tol = 10^(-10);
%tol = 10^(-5);
r = length(find(ss1>tol));


% Reduced order for SVD-based Loewner approximants
x=u1(:,1:r);
y=v2(:,1:r);
Er=-x.'*LL_lo*y;
Ar=-x.'*sLL_lo*y;
Br=x.'*V_lo;
Cr=W_lo*y;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% the complete frequency range
% omega_all = logspace(-2,6,)*j;
omega_all = logspace(-4,6,nLlo+nLhi)*j;
%omega_all = logspace(-2,10,600)*j;

% omega_all = [omega_lo, omega_hi];
% H1 = cat(3, H_lo, H_hi);

H1 = zeros(p,m,length(omega_all));

for jj = 1:length(omega_all)
    H1(:,:,jj) =  H_true(omega_all(jj));  % C*((omega_all(jj)*E-A)\B)+P_true{2}*omega_all(jj);
end

% sysl = dss(A1,B1,C1,0,E1);
% 
% sysr = dss(Ar,Br,Cr,0,Er);

H2 = zeros(p,m,length(omega_all));

for jj = 1:length(omega_all)
    H2(:,:,jj) = Cr*((omega_all(jj)*Er-Ar)\Br);
end

% figure
% subplot(1,2,1)
% colororder(syscolors)
% h1 = loglog(imag(omega_all),abs(reshape(H1(1,1,:),1,length(omega_all)))); hold on
% h2 = loglog(imag(omega_all),abs(reshape(H2(1,1,:),1,length(omega_all))));
% title('Frequency response');
% legend([h1 h2],'Original model',['Loewner model - r = ',num2str(r)]);
% axis tight
% xt={'10^{-2}'; '10^0'; '10^2'; '10^4'; '10^6'} ; 
% set(gca,'xtick',[10^(-2) 10^(0) 10^2 10^4 10^6]); 
% set(gca,'xticklabel',xt);
% xlabel('Frequency');
% 
% subplot(1,2,2)
% colororder(errcolors)
% loglog(imag(omega_all),abs(abs(reshape(H1(1,1,:),1,length(omega_all)))-abs(reshape(H2(1,1,:),1,length(omega_all))))...
% ./abs(abs(reshape(H1(1,1,:),1,length(omega_all)))));
% title('Absolute error');
% axis tight
% xt={'10^{-2}'; '10^0'; '10^2'; '10^4'; '10^6'} ; 
% set(gca,'xtick',[10^(-2) 10^(0) 10^2 10^4 10^6]); 
% set(gca,'xticklabel',xt);
% xlabel('Frequency');



for jj =1:length(omega_lo)
 polypart(:,:,jj) = cell2mat(P_est)*[eye(m) ; omega_lo(jj)*eye(m) ; omega_lo(jj)^2*eye(m)];
end

% subtract the polynomial part from the data
Hh = H_lo-polypart; 

[LL1,sLL1,mu,la,V1,W1] = loewner(omega_lo,reshape(Hh(1,1,:),1,length(omega_lo)),'real');

% ss1 = svd(LL1);ss1 = ss1/ss1(1);
% 
% [u1,sing1,~]=svd([LL1 sLL1],'econ');
% [~,sing2,v2]=svd([LL1;sLL1]);

% Reduced order for SVD-based Loewner approximants
x=u1(:,1:r);
y=v2(:,1:r);
Er1=-x.'*LL1*y;
Ar1=-x.'*sLL1*y;
Br1=x.'*V1;
Cr1=W1*y;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

%reattach the polynomial part 
J = kron([0 1;0 0],eye(m));
A3 = blkdiag(Ar1,eye(2*m));
B3 = [Br1;kron([0 1]',eye(m))];
C3 = [Cr1 -[P_est{2} P_est{1}]];
E3 = blkdiag(Er1,J);

% sys3 = dss(A3,B3,C3,0,E3);

H3 = zeros(p,m,length(omega_all));

for jj = 1:length(omega_all)
    H3(:,:,jj) = C3*((omega_all(jj)*E3-A3)\B3);
end

figure
colororder(syscolors)
h1 = loglog(imag(omega_all),abs(reshape(H1(1,1,:),1,length(omega_all)))); hold on
h2 = loglog(imag(omega_all),abs(reshape(H2(1,1,:),1,length(omega_all))),'-.');
h3 = loglog(imag(omega_all),abs(reshape(H3(1,1,:),1,length(omega_all))),'--');
%title('Frequency response');
legend([h1 h2 h3],'Original model',['Loewner - r = ',num2str(r)],...
    ['Loewner new - r = ',num2str(r)],'Location','northwest');
xlabel('Frequency');

axis tight

xt={'10^{-2}'; '10^0'; '10^2'; '10^4'; '10^6'} ; 
set(gca,'xtick',[10^(-2) 10^(0) 10^2 10^4 10^6]); 
set(gca,'xticklabel',xt);

figure
colororder(errcolors)
loglog(imag(omega_all),abs(reshape(H1(1,1,:),1,length(omega_all))-reshape(H2(1,1,:),1,length(omega_all)))...
./abs(reshape(H1(1,1,:),1,length(omega_all))),'-.'); hold on;
loglog(imag(omega_all),abs(reshape(H1(1,1,:),1,length(omega_all))-reshape(H3(1,1,:),1,length(omega_all)))...
./abs(reshape(H1(1,1,:),1,length(omega_all))),'--'); hold on;
title('Relative approximation error');
legend('Loewner','Loewner new','Location','northwest');
xlabel('Frequency');

axis tight
% 
xlim([10^(-2) 10^5]);
% ylim([10^(-8) 10^0]);
% 
xt={'10^{-2}'; '10^0'; '10^2'; '10^4'; '10^6'} ; 
set(gca,'xtick',[10^(-2) 10^(0) 10^2 10^4 10^6]); 
set(gca,'xticklabel',xt);
% 
yt={'10^{-10}';'10^{-8}'; '10^{-6}'; '10^{-4}'; '10^{-2}'; '10^0'} ; 
set(gca,'ytick',[10^(-10) 10^(-8) 10^(-6) 10^(-4) 10^(-2) 10^0]); 
set(gca,'yticklabel',yt);

save(['data/Loewner_' problem '.mat'], 'H1', 'H2', 'H3', 'omega_all', 'r')
