import numpy as np
import seaborn as sns
import matplotlib.pyplot as plt
from phyafb_api import read_hr_file, read_playing_file, read_game_file


def figure_hr_player(path: str):
    fig = plt.figure()
    data = np.zeros((10, 10))
    for player in range(1, 11):
        for game in range(1, 11):
            hr_data = read_hr_file(path, game, player)
            if len(hr_data) > 0:
                data[player - 1][game - 1] = np.mean(hr_data)
            else:
                data[player - 1][game - 1] = np.nan
    print(data)
    sns.boxplot(data=data, orient="v")
    plt.xlabel("Player")
    plt.ylabel("HR (bpm)")
    plt.xticks(range(10), range(1, 11))
    fig.savefig("boxplot_players.png")


def figure_wg_tt_lv(path: str, player: int):
    wg = np.zeros((1, 10))
    tt = np.zeros((1, 10))
    lt = np.zeros((1, 10))

    for game in range(1, 11):
        hr_data = read_hr_file(path, game, player)
        playing_data = read_playing_file(path, game, player)
        analysis_data = read_game_file(path, game, "ANALYSIS")

        if len(hr_data) == 0:
            # if the player did not play in the game, the HR vectors are set to NaN
            wg[0, game - 1] = np.nan
            tt[0, game - 1] = np.nan
            lt[0, game - 1] = np.nan
        else:
            idx1 = playing_data == 1   # indices of the HRs when the pl. was on the court
            idx2 = analysis_data == 1  # indices of the HRs when the pl. was on the court and the clock is running

            # Whole game: It is the mean of all the HRs of the player in the game
            wg[0, game - 1] = np.mean(hr_data)

            # Total game: It is the mean of the HRs of the player when she was on the court
            if sum(idx2) == 0:
                tt[0, game - 1] = np.nan
            else:
                tt[0, game - 1] = np.mean(hr_data[idx2])

            # Live time: It is the mean of the HRs of the player when she was on the court and the clock was running
            if sum(idx1) == 0:
                lt[0, game - 1] = np.nan
            else:
                lt[0, game - 1] = np.mean(hr_data[idx1])

    # Plotting a boxplot with the three HR vectors
    data = np.concatenate((wg, tt, lt), axis=0)
    data = np.transpose(data)
    fig = plt.figure()
    sns.boxplot(data=data, orient="v")
    plt.xlabel("Type")
    plt.ylabel("HR (bpm)")
    plt.xticks(range(3), ["WG", "TT", "LT"])
    fig.savefig("boxplot_wgttlt" + str(player) + ".png")


def figure_attack_defense(path: str, player: int):
    hr_at = np.zeros((1, 10))
    hr_df = np.zeros((1, 10))

    for game in range(1, 11):
        hr_data = read_hr_file(path, game, player)
        playing_data = read_playing_file(path, game, player)
        analysis_data = read_game_file(path, game, "ANALYSIS")
        attacking_data = read_game_file(path, game, "ATTACK_DEF")

        if len(hr_data) == 0:
            # if the player did not play in the game, the HR vectors are set to NaN
            hr_at[0, game - 1] = np.nan
            hr_df[0, game - 1] = np.nan
        else:
            idx1 = playing_data == 1  # indices of the HRs when the pl. was on the court
            idx2 = analysis_data == 1  # indices of the HRs when the pl. was on the court and the clock is running
            idx_at = attacking_data == 1  # indices of the HRs when the pl. was attacking
            idx_df = attacking_data == 2  # indices of the HRs when the pl. was defending

            idx = idx1 & idx2  # indices of the HRs when the pl. was on the court and the clock is running
            idx = idx & idx_at
            if sum(idx) == 0:
                hr_at[0, game - 1] = np.nan
            else:
                hr_at[0, game - 1] = np.mean(hr_data[idx])

            idx = idx1 & idx2  # indices of the HRs when the pl. was on the court and the clock is running
            idx = idx & idx_df
            if sum(idx) == 0:
                hr_df[0, game - 1] = np.nan
            else:
                hr_df[0, game - 1] = np.mean(hr_data[idx])

    # Plotting a boxplot with the three HR vectors
    data = np.concatenate((hr_at, hr_df), axis=0)
    data = np.transpose(data)
    fig = plt.figure()
    sns.boxplot(data=data, orient="v")
    plt.xlabel("Game situation")
    plt.ylabel("HR (bpm)")
    plt.xticks(range(2), ["Attacking", "Defending"])
    fig.savefig("boxplot_atdf" + str(player) + ".png")


if __name__ == '__main__':
    path = "./data/PHYAFBDataset/"
    figure_hr_player(path)
    player = 1
    figure_wg_tt_lv(path, player)
    figure_attack_defense(path, player)

