from dataclasses import dataclass


@dataclass
class LegacyLUT:
    title: str
    lut_color: str
    lut_alpha: str
    lut_range: list[int]
    color_interpolation: str
    lut_relative: bool
    lut_color_invert: str = ""
    lut_alpha_invert: str = ""
    color_interpolation_invert: str = "RGB"


LEGACY_LUTS = [
    LegacyLUT(
        title="LABEL",
        lut_color="[0 0 0 0.976563, 1 0 0.878906 0.976563, 2 0 0.976563 0.195313, 3 0.683594 0.976563 0, 4 0.976563 0.390625 0, 5 0.976563 0 0.488281, 6 0.585938 0 0.976563, 7 0 0 0.808594, 8 0 0.726563 0.808594, 9 0 0.808594 0.160156, 10 0.5625 0.808594 0, 11 0.808594 0.320313 0, 12 0.808594 0 0.402344, 13 0.484375 0 0.808594, 14 0 0 0.636719, 15 0 0.570313 0.636719, 16 0 0.636719 0.125, 17 0.445313 0.636719 0, 18 0.636719 0.253906 0, 19 0.636719 0 0.316406, 20 0.378906 0 0.636719, 21 0 0 0.464844, 22 0 0.417969 0.464844, 23 0 0.464844 0.0898438, 24 0.324219 0.464844 0, 25 0.464844 0.183594 0, 26 0.464844 0 0.230469, 27 0.277344 0 0.464844, 28 0.390625 0.390625 0.976563, 29 0.390625 0.917969 0.976563, 30 0.390625 0.976563 0.507813, 31 0.800781 0.976563 0.390625, 32 0.976563 0.625 0.390625, 33 0.976563 0.390625 0.683594, 34 0.742188 0.390625 0.976563, 35 0.320313 0.320313 0.808594, 36 0.320313 0.757813 0.808594, 37 0.320313 0.808594 0.417969, 38 0.660156 0.808594 0.320313, 39 0.808594 0.515625 0.320313, 40 0.808594 0.320313 0.5625, 41 0.613281 0.320313 0.808594, 42 0.253906 0.253906 0.636719, 43 0.253906 0.597656 0.636719, 44 0.253906 0.636719 0.328125, 45 0.519531 0.636719 0.253906, 46 0.636719 0.40625 0.253906, 47 0.636719 0.253906 0.445313, 48 0.480469 0.253906 0.636719, 49 0.183594 0.183594 0.464844, 50 0.183594 0.4375 0.464844, 51 0.183594 0.464844 0.238281, 52 0.378906 0.464844 0.183594, 53 0.464844 0.296875 0.183594, 54 0.464844 0.183594 0.324219, 55 0.351563 0.183594 0.464844, 56 0.683594 0.683594 0.976563, 57 0.683594 0.949219 0.976563, 58 0.683594 0.976563 0.742188, 59 0.890625 0.976563 0.683594, 60 0.976563 0.800781 0.683594, 61 0.976563 0.683594 0.832031, 62 0.859375 0.683594 0.976563, 63 0.5625 0.5625 0.808594, 64 0.5625 0.78125 0.808594, 65 0.5625 0.808594 0.613281, 66 0.734375 0.808594 0.5625, 67 0.808594 0.660156 0.5625, 68 0.808594 0.5625 0.6875, 69 0.710938 0.5625 0.808594, 70 0.445313 0.445313 0.636719, 71 0.445313 0.617188 0.636719, 72 0.445313 0.636719 0.480469, 73 0.578125 0.636719 0.445313, 74 0.636719 0.519531 0.445313, 75 0.636719 0.445313 0.539063, 76 0.558594 0.445313 0.636719, 77 0.324219 0.324219 0.464844, 78 0.324219 0.449219 0.464844, 79 0.324219 0.464844 0.351563, 80 0.421875 0.464844 0.324219, 81 0.464844 0.378906 0.324219, 82 0.464844 0.324219 0.394531, 83 0.40625 0.324219 0.464844, 84 0.878906 0.878906 0.976563, 85 0.878906 0.96875 0.976563, 86 0.878906 0.976563 0.898438, 87 0.949219 0.976563 0.878906, 88 0.976563 0.917969 0.878906, 89 0.976563 0.878906 0.929688, 90 0.9375 0.878906 0.976563, 91 0.726563 0.726563 0.808594, 92 0.726563 0.800781 0.808594, 93 0.726563 0.808594 0.742188, 94 0.78125 0.808594 0.726563, 95 0.808594 0.757813 0.726563, 96 0.808594 0.726563 0.765625, 97 0.773438 0.726563 0.808594, 98 0.570313 0.570313 0.636719, 99 0.570313 0.628906 0.636719, 100 0.570313 0.636719 0.585938, 101 0.617188 0.636719 0.570313, 102 0.636719 0.597656 0.570313, 103 0.636719 0.570313 0.601563, 104 0.609375 0.570313 0.636719, 105 0.417969 0.417969 0.464844, 106 0.417969 0.460938 0.464844, 107 0.417969 0.464844 0.425781, 108 0.449219 0.464844 0.417969, 109 0.464844 0.4375 0.417969, 110 0.464844 0.417969 0.441406, 111 0.445313 0.417969 0.464844, 112 0 0 0.976563, 113 0 0.878906 0.976563, 114 0 0.976563 0.195313, 115 0.683594 0.976563 0, 116 0.976563 0.390625 0, 117 0.976563 0 0.488281, 118 0.585938 0 0.976563, 119 0 0 0.808594, 120 0 0.726563 0.808594, 121 0 0.808594 0.160156, 122 0.5625 0.808594 0, 123 0.808594 0.320313 0, 124 0.808594 0 0.402344, 125 0.484375 0 0.808594, 126 0 0 0.636719, 127 0 0.570313 0.636719, 128 0 0.636719 0.125, 129 0.445313 0.636719 0, 130 0.636719 0.253906 0, 131 0.636719 0 0.316406, 132 0.378906 0 0.636719, 133 0 0 0.464844, 134 0 0.417969 0.464844, 135 0 0.464844 0.0898438, 136 0.324219 0.464844 0, 137 0.464844 0.183594 0, 138 0.464844 0 0.230469, 139 0.277344 0 0.464844, 140 0.390625 0.390625 0.976563, 141 0.390625 0.917969 0.976563, 142 0.390625 0.976563 0.507813, 143 0.800781 0.976563 0.390625, 144 0.976563 0.625 0.390625, 145 0.976563 0.390625 0.683594, 146 0.742188 0.390625 0.976563, 147 0.320313 0.320313 0.808594, 148 0.320313 0.757813 0.808594, 149 0.320313 0.808594 0.417969, 150 0.660156 0.808594 0.320313, 151 0.808594 0.515625 0.320313, 152 0.808594 0.320313 0.5625, 153 0.613281 0.320313 0.808594, 154 0.253906 0.253906 0.636719, 155 0.253906 0.597656 0.636719, 156 0.253906 0.636719 0.328125, 157 0.519531 0.636719 0.253906, 158 0.636719 0.40625 0.253906, 159 0.636719 0.253906 0.445313, 160 0.480469 0.253906 0.636719, 161 0.183594 0.183594 0.464844, 162 0.183594 0.4375 0.464844, 163 0.183594 0.464844 0.238281, 164 0.378906 0.464844 0.183594, 165 0.464844 0.296875 0.183594, 166 0.464844 0.183594 0.324219, 167 0.351563 0.183594 0.464844, 168 0.683594 0.683594 0.976563, 169 0.683594 0.949219 0.976563, 170 0.683594 0.976563 0.742188, 171 0.890625 0.976563 0.683594, 172 0.976563 0.800781 0.683594, 173 0.976563 0.683594 0.832031, 174 0.859375 0.683594 0.976563, 175 0.5625 0.5625 0.808594, 176 0.5625 0.78125 0.808594, 177 0.5625 0.808594 0.613281, 178 0.734375 0.808594 0.5625, 179 0.808594 0.660156 0.5625, 180 0.808594 0.5625 0.6875, 181 0.710938 0.5625 0.808594, 182 0.445313 0.445313 0.636719, 183 0.445313 0.617188 0.636719, 184 0.445313 0.636719 0.480469, 185 0.578125 0.636719 0.445313, 186 0.636719 0.519531 0.445313, 187 0.636719 0.445313 0.539063, 188 0.558594 0.445313 0.636719, 189 0.324219 0.324219 0.464844, 190 0.324219 0.449219 0.464844, 191 0.324219 0.464844 0.351563, 192 0.421875 0.464844 0.324219, 193 0.464844 0.378906 0.324219, 194 0.464844 0.324219 0.394531, 195 0.40625 0.324219 0.464844, 196 0.878906 0.878906 0.976563, 197 0.878906 0.96875 0.976563, 198 0.878906 0.976563 0.898438, 199 0.949219 0.976563 0.878906, 200 0.976563 0.917969 0.878906, 201 0.976563 0.878906 0.929688, 202 0.9375 0.878906 0.976563, 203 0.726563 0.726563 0.808594, 204 0.726563 0.800781 0.808594, 205 0.726563 0.808594 0.742188, 206 0.78125 0.808594 0.726563, 207 0.808594 0.757813 0.726563, 208 0.808594 0.726563 0.765625, 209 0.773438 0.726563 0.808594, 210 0.570313 0.570313 0.636719, 211 0.570313 0.628906 0.636719, 212 0.570313 0.636719 0.585938, 213 0.617188 0.636719 0.570313, 214 0.636719 0.597656 0.570313, 215 0.636719 0.570313 0.601563, 216 0.609375 0.570313 0.636719, 217 0.417969 0.417969 0.464844, 218 0.417969 0.460938 0.464844, 219 0.417969 0.464844 0.425781, 220 0.449219 0.464844 0.417969, 221 0.464844 0.4375 0.417969, 222 0.464844 0.417969 0.441406, 223 0.445313 0.417969 0.464844, 224 0 0 0.976563, 225 0 0.878906 0.976563, 226 0 0.976563 0.195313, 227 0.683594 0.976563 0, 228 0.976563 0.390625 0, 229 0.976563 0 0.488281, 230 0.585938 0 0.976563, 231 0 0 0.808594, 232 0 0.726563 0.808594, 233 0 0.808594 0.160156, 234 0.5625 0.808594 0, 235 0.808594 0.320313 0, 236 0.808594 0 0.402344, 237 0.484375 0 0.808594, 238 0 0 0.636719, 239 0 0.570313 0.636719, 240 0 0.636719 0.125, 241 0.445313 0.636719 0, 242 0.636719 0.253906 0, 243 0.636719 0 0.316406, 244 0.378906 0 0.636719, 245 0 0 0.464844, 246 0 0.417969 0.464844, 247 0 0.464844 0.0898438, 248 0.324219 0.464844 0, 249 0.464844 0.183594 0, 250 0.464844 0 0.230469, 251 0.277344 0 0.464844, 252 0.390625 0.390625 0.976563, 253 0.390625 0.917969 0.976563, 254 0.390625 0.976563 0.507813, 255 0 0 0]",
        lut_alpha="[0 0, 1 1, 2 1, 3 1, 4 1, 5 1, 6 1, 7 1, 8 1, 9 1, 10 1, 11 1, 12 1, 13 1, 14 1, 15 1, 16 1, 17 1, 18 1, 19 1, 20 1, 21 1, 22 1, 23 1, 24 1, 25 1, 26 1, 27 1, 28 1, 29 1, 30 1, 31 1, 32 1, 33 1, 34 1, 35 1, 36 1, 37 1, 38 1, 39 1, 40 1, 41 1, 42 1, 43 1, 44 1, 45 1, 46 1, 47 1, 48 1, 49 1, 50 1, 51 1, 52 1, 53 1, 54 1, 55 1, 56 1, 57 1, 58 1, 59 1, 60 1, 61 1, 62 1, 63 1, 64 1, 65 1, 66 1, 67 1, 68 1, 69 1, 70 1, 71 1, 72 1, 73 1, 74 1, 75 1, 76 1, 77 1, 78 1, 79 1, 80 1, 81 1, 82 1, 83 1, 84 1, 85 1, 86 1, 87 1, 88 1, 89 1, 90 1, 91 1, 92 1, 93 1, 94 1, 95 1, 96 1, 97 1, 98 1, 99 1, 100 1, 101 1, 102 1, 103 1, 104 1, 105 1, 106 1, 107 1, 108 1, 109 1, 110 1, 111 1, 112 1, 113 1, 114 1, 115 1, 116 1, 117 1, 118 1, 119 1, 120 1, 121 1, 122 1, 123 1, 124 1, 125 1, 126 1, 127 1, 128 1, 129 1, 130 1, 131 1, 132 1, 133 1, 134 1, 135 1, 136 1, 137 1, 138 1, 139 1, 140 1, 141 1, 142 1, 143 1, 144 1, 145 1, 146 1, 147 1, 148 1, 149 1, 150 1, 151 1, 152 1, 153 1, 154 1, 155 1, 156 1, 157 1, 158 1, 159 1, 160 1, 161 1, 162 1, 163 1, 164 1, 165 1, 166 1, 167 1, 168 1, 169 1, 170 1, 171 1, 172 1, 173 1, 174 1, 175 1, 176 1, 177 1, 178 1, 179 1, 180 1, 181 1, 182 1, 183 1, 184 1, 185 1, 186 1, 187 1, 188 1, 189 1, 190 1, 191 1, 192 1, 193 1, 194 1, 195 1, 196 1, 197 1, 198 1, 199 1, 200 1, 201 1, 202 1, 203 1, 204 1, 205 1, 206 1, 207 1, 208 1, 209 1, 210 1, 211 1, 212 1, 213 1, 214 1, 215 1, 216 1, 217 1, 218 1, 219 1, 220 1, 221 1, 222 1, 223 1, 224 1, 225 1, 226 1, 227 1, 228 1, 229 1, 230 1, 231 1, 232 1, 233 1, 234 1, 235 1, 236 1, 237 1, 238 1, 239 1, 240 1, 241 1, 242 1, 243 1, 244 1, 245 1, 246 1, 247 1, 248 1, 249 1, 250 1, 251 1, 252 1, 253 1, 254 1, 255 0.5]",
        lut_range=[0, 255],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="JOINTHIST",
        lut_color="[0 0 0 0, 1 1 0 0, 2 1 1 0, 3 0 1 0, 4 1 1 1]",
        lut_alpha="[0 0, 1 1, 2 1, 3 1, 4 1]",
        lut_range=[0, 4],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="DEFAULT",
        lut_color="[ 0 0 0 0, 255 1 1 1 ]",
        lut_alpha="[ 0 0, 255 1 ]",
        lut_color_invert="[ 255 0 0 0, 0 1 1 1 ]",
        lut_alpha_invert="[ 255 0, 0 1 ]",
        lut_range=[0, 255],
        color_interpolation="InterpolateRGB",
        color_interpolation_invert="InterpolateRGB",
        lut_relative=True,
    ),
    LegacyLUT(
        title="RAINBOW",
        lut_color="[ 0 0.25098 0 1, 84.9544 0.145147 0 1, 84.9585 0.0745912 0 1, 4095 1 0 0 ]",
        lut_alpha="[ 0 0, 84.9544 0, 84.9585 0.5, 4095 0.5 ]",
        lut_color_invert="[ 0 1 0 0, 4010.9585 0.0745912 0 1, 4010.9544 0.145147 0 1, 4095 0.25098 0 1]",
        lut_alpha_invert="[ 0 0.5, 4010.9585 0.5, 4010.9544 0, 4095 0 ]",
        lut_range=[0, 4095],
        color_interpolation="InterpolateHLSneg",
        color_interpolation_invert="InterpolateHLSpos",
        lut_relative=True,
    ),
    LegacyLUT(
        title="BIPOLAR",
        lut_color="[ -100 0 0 1, 0 0 0 1, 0 1 0 0, 100 1 0 0]",
        lut_alpha="[ -100 1, 0 0, 0 0, 100 1]",
        lut_color_invert="[ -100 1 0 0, 0 1 0 0, 0 0 0 1, 100 0 0 1]",
        lut_alpha_invert="[ -100 1, 0 0, 0 0, 100 1]",
        lut_range=[-100, 100],
        color_interpolation="InterpolateRGB",
        color_interpolation_invert="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="ADC_HOTIRON",
        lut_color="[ 0 0 0 0, 40 1 .65 0.16]",
        lut_alpha="[ 0 0, 40 1]",
        lut_color_invert="[ 0 1 .65 0.16, 40 0 0 0]",
        lut_alpha_invert="[ 0 1, 40 0]",
        lut_range=[0, 40],
        color_interpolation="InterpolateRGB",
        color_interpolation_invert="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="TRAFFICLIGHT",
        lut_color="[ 0 0 0 0, 144.425 0.0195265 0 0, 144.429 0 1 0, 4095 1 0 0 ]",
        lut_alpha="[ 0 0, 144.425 0, 144.429 1, 4095 1]",
        lut_color_invert="[ 0 0 0 0, 144.425 0 0 0, 144.429 1 0 0, 4095 0 1 0 ]",
        lut_alpha_invert="[ 4095 0, 3950.429 0, 3950.425 1, 0 1]",
        lut_range=[0, 4095],
        color_interpolation="InterpolateHLSneg",
        color_interpolation_invert="InterpolateHLSpos",
        lut_relative=True,
    ),
    LegacyLUT(
        title="TEMPERATURE",
        lut_color="[ 0 0 0 0, 144.425 0.0 0 0, 144.429 0 0 1, 4095 1 0 0 ]",
        lut_alpha="[ 0 0, 144.425 0, 144.429 1, 4095 1 ]",
        lut_color_invert="[ 0 0 0 0, 144.425 0 0 0, 144.429 1 0 0, 4095 0 0 1]",
        lut_alpha_invert="[ 4095 0, 3950.429 0, 3950.425 1, 0 1 ]",
        lut_range=[0, 4095],
        color_interpolation="InterpolateHLSpos",
        color_interpolation_invert="InterpolateHLSneg",
        lut_relative=True,
    ),
    LegacyLUT(
        title="HOTIRON",
        lut_color="[ 0 0 0 0, 161.421 1 0.729387 0.00989342, 4095 1 0.25098 0.25098 ]",
        lut_alpha="[ 0 0, 161.421 0.226994, 4095 1 ]",
        lut_color_invert="[ 0 1 0.25098 0.25098, 3933.421 1 0.729387 0.00989342, 4095 0 0 0]",
        lut_alpha_invert="[ 0 1, 3933.421 0.226994, 4095 0 ]",
        lut_range=[0, 4095],
        color_interpolation="InterpolateRGB",
        color_interpolation_invert="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="PERFUSION",
        lut_color="[ 0 1 0 0, 500 0 0 1, 501 0 0 0, 800 0 0 0 ]",
        lut_alpha="[ 0 1, 500 0, 501 0, 800 0 ]",
        lut_color_invert="[0 0 0 1, 500 1 0 0, 501 0 0 0, 800 0 0 0 ]",
        lut_alpha_invert="[ 0 0 , 500 1, 501 0,800 0]",
        lut_range=[0, 800],
        color_interpolation="InterpolateHLSpos",
        color_interpolation_invert="InterpolateHLSneg",
        lut_relative=True,
    ),
    LegacyLUT(
        title="CAD4TB",
        lut_color="[-1 0 0 1, -0.01 0 0 1, 0 0 0 1, 1 1 0 0 ]",
        lut_alpha="[ -1 0, -0.01 0, 0 0.2, 1 0.6 ]",
        lut_color_invert="[-1 1 0 0, -0.01 1 0 0, 0 1 0 0, 1 0 0 1]",
        lut_alpha_invert="[-1 0, -0.01 0, 0 0.2, 1 0.6 ]",
        lut_range=[-1, 1],
        color_interpolation="InterpolateHLSneg",
        color_interpolation_invert="InterpolateHLSpos",
        lut_relative=False,
    ),
    LegacyLUT(
        title="EMPH",
        lut_color="[ 0 0 0 0, 0.9 0 0 0, 1 0 1 1, 151 0 1 1, 151.1 0 0 0, 501.9 0 0 0, 502 0 1 0, 652 0 1 0, 652.1 0 0 0, 1002.9 0 0 0, 1003 1 0 0, 1153 1 0 0, 1153.1 0 0 0, 1503.9 0 0 0, 1504 1 1 0, 1654 1 1 0, 1654.1 0 0 0, 2004.9 0 0 0, 2005 1 0 1, 2155 1 0 1, 2155.1 0 0 0, 2506 0 0 0 ]",
        lut_alpha="[ 0 0, 0.9 0, 1 1, 151 1, 151.1 0, 501.9 0, 502 1, 652 1, 652.1 0, 1002.9 0, 1003 1, 1153 1, 1153.1 0, 1503.9 0, 1504 1, 1654 1, 1654.1 0, 2004.9 0, 2005 1, 2155 1, 2155.1 0, 2506 0 ]",
        lut_range=[0, 2506],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="LOLA11",
        lut_color="[0 0 0 0.976563, 1 0.423529 1 0.592157, 2 0.458824 0.447059 1, 3 0.976471 0.286275 0, 4 0.976563 0.390625 0, 5 0.976563 0 0.488281, 6 0.585938 0 0.976563, 7 0 0 0.808594, 8 0 0.726563 0.808594, 9 0 0.808594 0.160156, 10 0.5625 0.808594 0, 11 0.808594 0.320313 0, 12 0.808594 0 0.402344, 13 0.484375 0 0.808594, 14 0 0 0.636719, 15 0 0.570313 0.636719, 16 0 0.636719 0.125, 17 0.445313 0.636719 0, 18 0.636719 0.253906 0, 19 0.636719 0 0.316406, 20 0.378906 0 0.636719, 21 0 0 0.464844, 22 0 0.417969 0.464844, 23 0 0.464844 0.0898438, 24 0.324219 0.464844 0, 25 0.464844 0.183594 0, 26 0.464844 0 0.230469, 27 0.277344 0 0.464844, 28 0.390625 0.390625 0.976563, 29 0.390625 0.917969 0.976563, 30 0.390625 0.976563 0.507813, 31 0.800781 0.976563 0.390625, 32 0.976563 0.625 0.390625, 33 0.976563 0.390625 0.683594, 34 0.742188 0.390625 0.976563, 35 0.320313 0.320313 0.808594, 36 0.320313 0.757813 0.808594, 37 0.320313 0.808594 0.417969, 38 0.660156 0.808594 0.320313, 39 0.808594 0.515625 0.320313, 40 0.808594 0.320313 0.5625, 41 0.613281 0.320313 0.808594, 42 0.253906 0.253906 0.636719, 43 0.253906 0.597656 0.636719, 44 0.253906 0.636719 0.328125, 45 0.519531 0.636719 0.253906, 46 0.636719 0.40625 0.253906, 47 0.636719 0.253906 0.445313, 48 0.480469 0.253906 0.636719, 49 0.183594 0.183594 0.464844, 50 0.183594 0.4375 0.464844, 51 0.183594 0.464844 0.238281, 52 0.378906 0.464844 0.183594, 53 0.464844 0.296875 0.183594, 54 0.464844 0.183594 0.324219, 55 0.351563 0.183594 0.464844, 56 0.683594 0.683594 0.976563, 57 0.683594 0.949219 0.976563, 58 0.683594 0.976563 0.742188, 59 0.890625 0.976563 0.683594, 60 0.976563 0.800781 0.683594, 61 0.976563 0.683594 0.832031, 62 0.859375 0.683594 0.976563, 63 0.5625 0.5625 0.808594, 64 0.5625 0.78125 0.808594, 65 0.5625 0.808594 0.613281, 66 0.734375 0.808594 0.5625, 67 0.808594 0.660156 0.5625, 68 0.808594 0.5625 0.6875, 69 0.710938 0.5625 0.808594, 70 0.445313 0.445313 0.636719, 71 0.445313 0.617188 0.636719, 72 0.445313 0.636719 0.480469, 73 0.578125 0.636719 0.445313, 74 0.636719 0.519531 0.445313, 75 0.636719 0.445313 0.539063, 76 0.558594 0.445313 0.636719, 77 0.324219 0.324219 0.464844, 78 0.324219 0.449219 0.464844, 79 0.324219 0.464844 0.351563, 80 0.421875 0.464844 0.324219, 81 0.464844 0.378906 0.324219, 82 0.464844 0.324219 0.394531, 83 0.40625 0.324219 0.464844, 84 0.878906 0.878906 0.976563, 85 0.878906 0.96875 0.976563, 86 0.878906 0.976563 0.898438, 87 0.949219 0.976563 0.878906, 88 0.976563 0.917969 0.878906, 89 0.976563 0.878906 0.929688, 90 0.9375 0.878906 0.976563, 91 0.726563 0.726563 0.808594, 92 0.726563 0.800781 0.808594, 93 0.726563 0.808594 0.742188, 94 0.78125 0.808594 0.726563, 95 0.808594 0.757813 0.726563, 96 0.808594 0.726563 0.765625, 97 0.773438 0.726563 0.808594, 98 0.570313 0.570313 0.636719, 99 0.570313 0.628906 0.636719, 100 0.570313 0.636719 0.585938, 101 0.617188 0.636719 0.570313, 102 0.636719 0.597656 0.570313, 103 0.636719 0.570313 0.601563, 104 0.609375 0.570313 0.636719, 105 0.417969 0.417969 0.464844, 106 0.417969 0.460938 0.464844, 107 0.417969 0.464844 0.425781, 108 0.449219 0.464844 0.417969, 109 0.464844 0.4375 0.417969, 110 0.464844 0.417969 0.441406, 111 0.445313 0.417969 0.464844, 112 0 0 0.976563, 113 0 0.878906 0.976563, 114 0 0.976563 0.195313, 115 0.683594 0.976563 0, 116 0.976563 0.390625 0, 117 0.976563 0 0.488281, 118 0.585938 0 0.976563, 119 0 0 0.808594, 120 0 0.726563 0.808594, 121 0 0.808594 0.160156, 122 0.5625 0.808594 0, 123 0.808594 0.320313 0, 124 0.808594 0 0.402344, 125 0.484375 0 0.808594, 126 0 0 0.636719, 127 0 0.570313 0.636719, 128 0 0.636719 0.125, 129 0.445313 0.636719 0, 130 0.636719 0.253906 0, 131 0.636719 0 0.316406, 132 0.378906 0 0.636719, 133 0 0 0.464844, 134 0 0.417969 0.464844, 135 0 0.464844 0.0898438, 136 0.324219 0.464844 0, 137 0.464844 0.183594 0, 138 0.464844 0 0.230469, 139 0.277344 0 0.464844, 140 0.390625 0.390625 0.976563, 141 0.390625 0.917969 0.976563, 142 0.390625 0.976563 0.507813, 143 0.800781 0.976563 0.390625, 144 0.976563 0.625 0.390625, 145 0.976563 0.390625 0.683594, 146 0.742188 0.390625 0.976563, 147 0.320313 0.320313 0.808594, 148 0.320313 0.757813 0.808594, 149 0.320313 0.808594 0.417969, 150 0.660156 0.808594 0.320313, 151 0.808594 0.515625 0.320313, 152 0.808594 0.320313 0.5625, 153 0.613281 0.320313 0.808594, 154 0.253906 0.253906 0.636719, 155 0.253906 0.597656 0.636719, 156 0.253906 0.636719 0.328125, 157 0.519531 0.636719 0.253906, 158 0.636719 0.40625 0.253906, 159 0.636719 0.253906 0.445313, 160 0.480469 0.253906 0.636719, 161 0.183594 0.183594 0.464844, 162 0.183594 0.4375 0.464844, 163 0.183594 0.464844 0.238281, 164 0.378906 0.464844 0.183594, 165 0.464844 0.296875 0.183594, 166 0.464844 0.183594 0.324219, 167 0.351563 0.183594 0.464844, 168 0.683594 0.683594 0.976563, 169 0.683594 0.949219 0.976563, 170 0.683594 0.976563 0.742188, 171 0.890625 0.976563 0.683594, 172 0.976563 0.800781 0.683594, 173 0.976563 0.683594 0.832031, 174 0.859375 0.683594 0.976563, 175 0.5625 0.5625 0.808594, 176 0.5625 0.78125 0.808594, 177 0.5625 0.808594 0.613281, 178 0.734375 0.808594 0.5625, 179 0.808594 0.660156 0.5625, 180 0.808594 0.5625 0.6875, 181 0.710938 0.5625 0.808594, 182 0.445313 0.445313 0.636719, 183 0.445313 0.617188 0.636719, 184 0.445313 0.636719 0.480469, 185 0.578125 0.636719 0.445313, 186 0.636719 0.519531 0.445313, 187 0.636719 0.445313 0.539063, 188 0.558594 0.445313 0.636719, 189 0.324219 0.324219 0.464844, 190 0.324219 0.449219 0.464844, 191 0.324219 0.464844 0.351563, 192 0.421875 0.464844 0.324219, 193 0.464844 0.378906 0.324219, 194 0.464844 0.324219 0.394531, 195 0.40625 0.324219 0.464844, 196 0.878906 0.878906 0.976563, 197 0.878906 0.96875 0.976563, 198 0.878906 0.976563 0.898438, 199 0.949219 0.976563 0.878906, 200 0.976563 0.917969 0.878906, 201 0.976563 0.878906 0.929688, 202 0.9375 0.878906 0.976563, 203 0.726563 0.726563 0.808594, 204 0.726563 0.800781 0.808594, 205 0.726563 0.808594 0.742188, 206 0.78125 0.808594 0.726563, 207 0.808594 0.757813 0.726563, 208 0.808594 0.726563 0.765625, 209 0.773438 0.726563 0.808594, 210 0.570313 0.570313 0.636719, 211 0.570313 0.628906 0.636719, 212 0.570313 0.636719 0.585938, 213 0.617188 0.636719 0.570313, 214 0.636719 0.597656 0.570313, 215 0.636719 0.570313 0.601563, 216 0.609375 0.570313 0.636719, 217 0.417969 0.417969 0.464844, 218 0.417969 0.460938 0.464844, 219 0.417969 0.464844 0.425781, 220 0.449219 0.464844 0.417969, 221 0.464844 0.4375 0.417969, 222 0.464844 0.417969 0.441406, 223 0.445313 0.417969 0.464844, 224 0 0 0.976563, 225 0 0.878906 0.976563, 226 0 0.976563 0.195313, 227 0.683594 0.976563 0, 228 0.976563 0.390625 0, 229 0.976563 0 0.488281, 230 0.585938 0 0.976563, 231 0 0 0.808594, 232 0 0.726563 0.808594, 233 0 0.808594 0.160156, 234 0.5625 0.808594 0, 235 0.808594 0.320313 0, 236 0.808594 0 0.402344, 237 0.484375 0 0.808594, 238 0 0 0.636719, 239 0 0.570313 0.636719, 240 0 0.636719 0.125, 241 0.445313 0.636719 0, 242 0.636719 0.253906 0, 243 0.636719 0 0.316406, 244 0.378906 0 0.636719, 245 0 0 0.464844, 246 0 0.417969 0.464844, 247 0 0.464844 0.0898438, 248 0.324219 0.464844 0, 249 0.464844 0.183594 0, 250 0.464844 0 0.230469, 251 0.277344 0 0.464844, 252 0.390625 0.390625 0.976563, 253 0.390625 0.917969 0.976563, 254 0.390625 0.976563 0.507813, 255 0 0 0]",
        lut_alpha="[0 0, 1 1, 2 1, 3 1, 4 1, 5 1, 6 1, 7 1, 8 1, 9 1, 10 1, 11 1, 12 1, 13 1, 14 1, 15 1, 16 1, 17 1, 18 1, 19 1, 20 1, 21 1, 22 1, 23 1, 24 1, 25 1, 26 1, 27 1, 28 1, 29 1, 30 1, 31 1, 32 1, 33 1, 34 1, 35 1, 36 1, 37 1, 38 1, 39 1, 40 1, 41 1, 42 1, 43 1, 44 1, 45 1, 46 1, 47 1, 48 1, 49 1, 50 1, 51 1, 52 1, 53 1, 54 1, 55 1, 56 1, 57 1, 58 1, 59 1, 60 1, 61 1, 62 1, 63 1, 64 1, 65 1, 66 1, 67 1, 68 1, 69 1, 70 1, 71 1, 72 1, 73 1, 74 1, 75 1, 76 1, 77 1, 78 1, 79 1, 80 1, 81 1, 82 1, 83 1, 84 1, 85 1, 86 1, 87 1, 88 1, 89 1, 90 1, 91 1, 92 1, 93 1, 94 1, 95 1, 96 1, 97 1, 98 1, 99 1, 100 1, 101 1, 102 1, 103 1, 104 1, 105 1, 106 1, 107 1, 108 1, 109 1, 110 1, 111 1, 112 1, 113 1, 114 1, 115 1, 116 1, 117 1, 118 1, 119 1, 120 1, 121 1, 122 1, 123 1, 124 1, 125 1, 126 1, 127 1, 128 1, 129 1, 130 1, 131 1, 132 1, 133 1, 134 1, 135 1, 136 1, 137 1, 138 1, 139 1, 140 1, 141 1, 142 1, 143 1, 144 1, 145 1, 146 1, 147 1, 148 1, 149 1, 150 1, 151 1, 152 1, 153 1, 154 1, 155 1, 156 1, 157 1, 158 1, 159 1, 160 1, 161 1, 162 1, 163 1, 164 1, 165 1, 166 1, 167 1, 168 1, 169 1, 170 1, 171 1, 172 1, 173 1, 174 1, 175 1, 176 1, 177 1, 178 1, 179 1, 180 1, 181 1, 182 1, 183 1, 184 1, 185 1, 186 1, 187 1, 188 1, 189 1, 190 1, 191 1, 192 1, 193 1, 194 1, 195 1, 196 1, 197 1, 198 1, 199 1, 200 1, 201 1, 202 1, 203 1, 204 1, 205 1, 206 1, 207 1, 208 1, 209 1, 210 1, 211 1, 212 1, 213 1, 214 1, 215 1, 216 1, 217 1, 218 1, 219 1, 220 1, 221 1, 222 1, 223 1, 224 1, 225 1, 226 1, 227 1, 228 1, 229 1, 230 1, 231 1, 232 1, 233 1, 234 1, 235 1, 236 1, 237 1, 238 1, 239 1, 240 1, 241 1, 242 1, 243 1, 244 1, 245 1, 246 1, 247 1, 248 1, 249 1, 250 1, 251 1, 252 1, 253 1, 254 1, 255 0.5]",
        lut_range=[0, 255],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="ASIST",
        lut_color="[0 0 0 0,1 0.015686 0 0,2 0.031373 0 0.011765,3 0.047059 0 0.023529,4 0.062745 0 0.035294,5 0.078431 0 0.047059,6 0.094118 0 0.058824,7 0.109804 0 0.070588,8 0.12549 0 0.082353,9 0.141176 0 0.094118,10 0.156863 0 0.105882,11 0.172549 0 0.117647,12 0.188235 0 0.129412,13 0.203922 0 0.141176,14 0.219608 0 0.152941,15 0.235294 0 0.164706,16 0.25098 0 0.176471,17 0.266667 0 0.188235,18 0.282353 0 0.2,19 0.298039 0 0.211765,20 0.313725 0 0.223529,21 0.309804 0 0.235294,22 0.305882 0 0.247059,23 0.301961 0 0.258824,24 0.298039 0 0.270588,25 0.294118 0 0.282353,26 0.290196 0 0.294118,27 0.286275 0 0.305882,28 0.282353 0 0.317647,29 0.278431 0 0.329412,30 0.27451 0 0.341176,31 0.270588 0 0.352941,32 0.266667 0 0.364706,33 0.262745 0 0.376471,34 0.258824 0 0.388235,35 0.254902 0 0.4,36 0.25098 0 0.411765,37 0.247059 0 0.423529,38 0.243137 0 0.435294,39 0.239216 0 0.447059,40 0.235294 0 0.458824,41 0.231373 0 0.470588,42 0.227451 0 0.482353,43 0.223529 0 0.494118,44 0.219608 0 0.505882,45 0.215686 0 0.517647,46 0.211765 0.011765 0.529412,47 0.207843 0.023529 0.541176,48 0.203922 0.035294 0.552941,49 0.2 0.047059 0.564706,50 0.196078 0.058824 0.576471,51 0.192157 0.070588 0.588235,52 0.188235 0.082353 0.6,53 0.184314 0.094118 0.611765,54 0.180392 0.105882 0.623529,55 0.176471 0.117647 0.635294,56 0.172549 0.129412 0.647059,57 0.168627 0.141176 0.658824,58 0.164706 0.152941 0.670588,59 0.160784 0.164706 0.682353,60 0.156863 0.176471 0.694118,61 0.152941 0.188235 0.705882,62 0.14902 0.2 0.717647,63 0.145098 0.211765 0.729412,64 0.141176 0.223529 0.741176,65 0.137255 0.235294 0.752941,66 0.133333 0.247059 0.764706,67 0.129412 0.258824 0.776471,68 0.12549 0.270588 0.788235,69 0.121569 0.282353 0.8,70 0.117647 0.294118 0.811765,71 0.113725 0.305882 0.823529,72 0.109804 0.317647 0.835294,73 0.105882 0.329412 0.847059,74 0.101961 0.341176 0.858824,75 0.098039 0.352941 0.870588,76 0.094118 0.364706 0.882353,77 0.090196 0.376471 0.894118,78 0.086275 0.388235 0.905882,79 0.082353 0.4 0.917647,80 0.078431 0.411765 0.929412,81 0.07451 0.423529 0.941176,82 0.070588 0.435294 0.952941,83 0.066667 0.447059 0.964706,84 0.062745 0.458824 0.976471,85 0.058824 0.470588 0.988235,86 0.054902 0.482353 1,87 0.05098 0.494118 0.980392,88 0.047059 0.505882 0.960784,89 0.043137 0.517647 0.941176,90 0.039216 0.529412 0.921569,91 0.035294 0.541176 0.901961,92 0.031373 0.552941 0.882353,93 0.027451 0.564706 0.862745,94 0.023529 0.576471 0.843137,95 0.019608 0.588235 0.823529,96 0.015686 0.6 0.803922,97 0.011765 0.611765 0.784314,98 0.007843 0.623529 0.764706,99 0.003922 0.635294 0.745098,100 0 0.647059 0.72549,101 0 0.658824 0.705882,102 0 0.670588 0.686275,103 0 0.682353 0.666667,104 0 0.694118 0.647059,105 0 0.705882 0.627451,106 0 0.717647 0.607843,107 0 0.729412 0.588235,108 0 0.741176 0.568627,109 0 0.752941 0.54902,110 0 0.764706 0.529412,111 0 0.776471 0.509804,112 0 0.788235 0.490196,113 0 0.8 0.470588,114 0 0.811765 0.45098,115 0 0.823529 0.431373,116 0 0.835294 0.411765,117 0 0.847059 0.392157,118 0 0.858824 0.372549,119 0 0.870588 0.352941,120 0 0.882353 0.333333,121 0 0.894118 0.313725,122 0 0.905882 0.294118,123 0 0.917647 0.27451,124 0 0.929412 0.254902,125 0 0.941176 0.235294,126 0 0.952941 0.215686,127 0 0.964706 0.196078,128 0 0.976471 0.176471,129 0.015686 0.988235 0.156863,130 0.031373 1 0.137255,131 0.047059 1 0.117647,132 0.062745 1 0.098039,133 0.078431 1 0.078431,134 0.094118 1 0.058824,135 0.109804 1 0.039216,136 0.12549 1 0.019608,137 0.141176 1 0,138 0.156863 1 0,139 0.172549 1 0,140 0.188235 1 0,141 0.203922 1 0,142 0.219608 1 0,143 0.235294 1 0,144 0.25098 1 0,145 0.266667 1 0,146 0.282353 1 0,147 0.298039 1 0,148 0.313725 1 0,149 0.329412 1 0,150 0.345098 1 0,151 0.360784 1 0,152 0.376471 1 0,153 0.392157 1 0,154 0.407843 1 0,155 0.423529 1 0,156 0.439216 1 0,157 0.454902 1 0,158 0.470588 1 0,159 0.486275 1 0,160 0.501961 1 0,161 0.517647 1 0,162 0.533333 1 0,163 0.54902 1 0,164 0.564706 1 0,165 0.580392 1 0,166 0.596078 1 0,167 0.611765 1 0,168 0.627451 1 0,169 0.643137 1 0,170 0.658824 1 0,171 0.67451 1 0,172 0.690196 1 0,173 0.705882 1 0,174 0.721569 1 0,175 0.737255 1 0,176 0.752941 1 0,177 0.768627 1 0,178 0.784314 1 0,179 0.8 1 0,180 0.815686 1 0,181 0.831373 1 0,182 0.847059 1 0,183 0.862745 1 0,184 0.878431 1 0,185 0.894118 1 0,186 0.909804 1 0,187 0.92549 1 0,188 0.941176 1 0,189 0.956863 1 0,190 0.972549 1 0,191 0.988235 1 0,192 1 0.988235 0,193 1 0.972549 0,194 1 0.956863 0,195 1 0.941176 0,196 1 0.92549 0,197 1 0.909804 0,198 1 0.894118 0,199 1 0.878431 0,200 1 0.862745 0,201 1 0.847059 0,202 1 0.831373 0,203 1 0.815686 0,204 1 0.8 0,205 1 0.784314 0,206 1 0.768627 0,207 1 0.752941 0,208 1 0.737255 0,209 1 0.721569 0,210 1 0.705882 0,211 1 0.690196 0,212 1 0.67451 0,213 1 0.658824 0,214 1 0.643137 0,215 1 0.627451 0,216 1 0.611765 0,217 1 0.596078 0,218 1 0.580392 0,219 1 0.564706 0,220 1 0.54902 0,221 1 0.533333 0,222 1 0.517647 0,223 1 0.501961 0,224 1 0.486275 0,225 1 0.470588 0,226 1 0.454902 0,227 1 0.439216 0,228 1 0.423529 0,229 1 0.407843 0,230 1 0.392157 0,231 1 0.376471 0,232 1 0.360784 0,233 1 0.345098 0,234 1 0.329412 0,235 1 0.313725 0,236 1 0.298039 0,237 1 0.282353 0,238 1 0.266667 0,239 1 0.25098 0,240 1 0.235294 0,241 1 0.219608 0,242 1 0.203922 0,243 1 0.188235 0,244 1 0.172549 0,245 1 0.156863 0,246 1 0.141176 0,247 1 0.12549 0,248 1 0.109804 0,249 1 0.094118 0,250 1 0.078431 0,251 1 0.062745 0,252 1 0.047059 0,253 1 0.031373 0,254 1 0.015686 0,255 1 0 0, 256 1 0 0]",
        lut_alpha="[0 0, 1 1, 2 1, 3 1, 4 1, 5 1, 6 1, 7 1, 8 1, 9 1, 10 1, 11 1, 12 1, 13 1, 14 1, 15 1, 16 1, 17 1, 18 1, 19 1, 20 1, 21 1, 22 1, 23 1, 24 1, 25 1, 26 1, 27 1, 28 1, 29 1, 30 1, 31 1, 32 1, 33 1, 34 1, 35 1, 36 1, 37 1, 38 1, 39 1, 40 1, 41 1, 42 1, 43 1, 44 1, 45 1, 46 1, 47 1, 48 1, 49 1, 50 1, 51 1, 52 1, 53 1, 54 1, 55 1, 56 1, 57 1, 58 1, 59 1, 60 1, 61 1, 62 1, 63 1, 64 1, 65 1, 66 1, 67 1, 68 1, 69 1, 70 1, 71 1, 72 1, 73 1, 74 1, 75 1, 76 1, 77 1, 78 1, 79 1, 80 1, 81 1, 82 1, 83 1, 84 1, 85 1, 86 1, 87 1, 88 1, 89 1, 90 1, 91 1, 92 1, 93 1, 94 1, 95 1, 96 1, 97 1, 98 1, 99 1, 100 1, 101 1, 102 1, 103 1, 104 1, 105 1, 106 1, 107 1, 108 1, 109 1, 110 1, 111 1, 112 1, 113 1, 114 1, 115 1, 116 1, 117 1, 118 1, 119 1, 120 1, 121 1, 122 1, 123 1, 124 1, 125 1, 126 1, 127 1, 128 1, 129 1, 130 1, 131 1, 132 1, 133 1, 134 1, 135 1, 136 1, 137 1, 138 1, 139 1, 140 1, 141 1, 142 1, 143 1, 144 1, 145 1, 146 1, 147 1, 148 1, 149 1, 150 1, 151 1, 152 1, 153 1, 154 1, 155 1, 156 1, 157 1, 158 1, 159 1, 160 1, 161 1, 162 1, 163 1, 164 1, 165 1, 166 1, 167 1, 168 1, 169 1, 170 1, 171 1, 172 1, 173 1, 174 1, 175 1, 176 1, 177 1, 178 1, 179 1, 180 1, 181 1, 182 1, 183 1, 184 1, 185 1, 186 1, 187 1, 188 1, 189 1, 190 1, 191 1, 192 1, 193 1, 194 1, 195 1, 196 1, 197 1, 198 1, 199 1, 200 1, 201 1, 202 1, 203 1, 204 1, 205 1, 206 1, 207 1, 208 1, 209 1, 210 1, 211 1, 212 1, 213 1, 214 1, 215 1, 216 1, 217 1, 218 1, 219 1, 220 1, 221 1, 222 1, 223 1, 224 1, 225 1, 226 1, 227 1, 228 1, 229 1, 230 1, 231 1, 232 1, 233 1, 234 1, 235 1, 236 1, 237 1, 238 1, 239 1, 240 1, 241 1, 242 1, 243 1, 244 1, 245 1, 246 1, 247 1, 248 1, 249 1, 250 1, 251 1, 252 1, 253 1, 254 1, 255 1,256 1]",
        lut_range=[0, 255],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="VESSEL12",
        lut_color="[ 0 0.25098 0 1, 79.3568 0.145147 0 1, 125.913 0.0745912 0 1, 255 1 0 0 ]",
        lut_alpha="[ 0 0, 79.3568 0, 125.913 0.202454, 255 0.453988 ]",
        lut_range=[0, 255],
        color_interpolation="InterpolateHLSneg",
        lut_relative=True,
    ),
    LegacyLUT(
        title="AIRWAY1",
        lut_color="[ 0 0 0 0, 1 1 0.5 0]",
        lut_alpha="[ 0 0, 1 0.55]",
        lut_range=[0, 1],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="AIRWAY2",
        lut_color="[ 0 0 0 0, 1 0 1 1]",
        lut_alpha="[ 0 0, 1 0.99]",
        lut_range=[0, 1],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="AIRWAY3",
        lut_color="[ 0 0 0 0, 1 0 0 1]",
        lut_alpha="[ 0 0, 1 0.75]",
        lut_range=[0, 1],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
    LegacyLUT(
        title="VITREA",
        lut_color="[ -10 0 0 1, 0 0.484375 0 1, 21.7427 1 0 0.0470588, 54.9793 1 0.682353 0, 68.7967 1 0.882353 0, 80 1 0.968627 0.764706 ]",
        lut_alpha="[ -10 0.343558, 0 0, 21.7427 0.5, 54.9793 0.6, 68.7967 0.7, 80 0.8 ]",
        color_interpolation="InterpolateHLS",
        color_interpolation_invert="InterpolateHLS",
        lut_color_invert="[ 0 0 0 0, 144.425 0 0 0, 144.429 1 0 0, 4095 0 0 1]",
        lut_alpha_invert="[ 4095 0, 3950.429 0, 3950.425 1, 0 1 ]",
        lut_range=[-10, 80],
        lut_relative=False,
    ),
    LegacyLUT(
        title="VITREA2",
        lut_color="[ -11 0 0 1, -10 0.0392157 0 0.917647, -0.888889 0.484375 0 1, 21.0954 1 0 0.0470588, 54.7013 1 0.682353 0, 68.6722 1 0.882353 0, 80 1 0.968627 0.764706 ]",
        lut_alpha="[ -11 0, -10 0.337423, -0.888889 0, 21.0954 0.5, 54.7013 0.6, 68.6722 0.7, 80 0.8 ]",
        color_interpolation="InterpolateHLS",
        color_interpolation_invert="InterpolateHLS",
        lut_color_invert="[ 0 0 0 0, 144.425 0 0 0, 144.429 1 0 0, 4095 0 0 1]",
        lut_alpha_invert="[ 4095 0, 3950.429 0, 3950.425 1, 0 1 ]",
        lut_range=[-11, 80],
        lut_relative=False,
    ),
    LegacyLUT(
        title="TOSHIBA",
        lut_color="[-10.00 0.00 0.00 0.00, -9.65 0.00 0.00 0.00, -9.29 0.00 0.00 0.14, -8.94 0.00 0.00 0.14, -8.59 0.00 0.00 0.18, -8.24 0.00 0.00 0.18, -7.88 0.00 0.00 0.21, -7.53 0.00 0.00 0.21, -7.18 0.00 0.00 0.24, -6.82 0.00 0.00 0.24, -6.47 0.00 0.00 0.27, -6.12 0.00 0.00 0.27, -5.76 0.00 0.00 0.31, -5.41 0.00 0.00 0.31, -5.06 0.00 0.00 0.34, -4.71 0.00 0.00 0.34, -4.35 0.00 0.00 0.38, -4.00 0.00 0.00 0.38, -3.65 0.02 0.00 0.41, -3.29 0.02 0.00 0.41, -2.94 0.05 0.00 0.44, -2.59 0.05 0.00 0.44, -2.24 0.07 0.00 0.47, -1.88 0.07 0.00 0.47, -1.53 0.09 0.00 0.51, -1.18 0.09 0.00 0.51, -0.82 0.12 0.00 0.54, -0.47 0.12 0.00 0.54, -0.12 0.14 0.00 0.58, 0.24 0.14 0.00 0.58, 0.59 0.16 0.00 0.61, 0.94 0.16 0.00 0.61, 1.29 0.19 0.00 0.64, 1.65 0.19 0.00 0.64, 2.00 0.21 0.00 0.67, 2.35 0.21 0.00 0.67, 2.71 0.24 0.00 0.70, 3.06 0.24 0.00 0.70, 3.41 0.26 0.00 0.73, 3.76 0.26 0.00 0.73, 4.12 0.28 0.00 0.75, 4.47 0.28 0.00 0.75, 4.82 0.31 0.00 0.78, 5.18 0.31 0.00 0.78, 5.53 0.33 0.00 0.80, 5.88 0.33 0.00 0.80, 6.24 0.35 0.00 0.83, 6.59 0.35 0.00 0.83, 6.94 0.38 0.00 0.86, 7.29 0.38 0.00 0.86, 7.65 0.40 0.00 0.88, 8.00 0.40 0.00 0.88, 8.35 0.42 0.00 0.91, 8.71 0.42 0.00 0.91, 9.06 0.45 0.00 0.90, 9.41 0.45 0.00 0.90, 9.76 0.47 0.00 0.89, 10.12 0.47 0.00 0.89, 10.47 0.49 0.00 0.89, 10.82 0.49 0.00 0.89, 11.18 0.52 0.00 0.88, 11.53 0.52 0.00 0.88, 11.88 0.54 0.00 0.86, 12.24 0.54 0.00 0.86, 12.59 0.56 0.00 0.84, 12.94 0.56 0.00 0.84, 13.29 0.59 0.00 0.81, 13.65 0.59 0.00 0.81, 14.00 0.61 0.00 0.78, 14.35 0.61 0.00 0.78, 14.71 0.62 0.00 0.75, 15.06 0.62 0.00 0.75, 15.41 0.64 0.00 0.72, 15.76 0.64 0.00 0.72, 16.12 0.64 0.00 0.69, 16.47 0.64 0.00 0.69, 16.82 0.65 0.00 0.67, 17.18 0.65 0.00 0.67, 17.53 0.67 0.00 0.64, 17.88 0.67 0.00 0.64, 18.24 0.68 0.00 0.61, 18.59 0.68 0.00 0.61, 18.94 0.69 0.00 0.58, 19.29 0.69 0.00 0.58, 19.65 0.70 0.00 0.55, 20.00 0.70 0.00 0.55, 20.35 0.71 0.00 0.53, 20.71 0.71 0.00 0.53, 21.06 0.72 0.00 0.49, 21.41 0.72 0.00 0.49, 21.76 0.73 0.00 0.47, 22.12 0.73 0.00 0.47, 22.47 0.74 0.00 0.44, 22.82 0.74 0.00 0.44, 23.18 0.75 0.00 0.41, 23.53 0.75 0.00 0.41, 23.88 0.76 0.00 0.38, 24.24 0.76 0.00 0.38, 24.59 0.77 0.00 0.35, 24.94 0.77 0.00 0.35, 25.29 0.78 0.00 0.33, 25.65 0.78 0.00 0.33, 26.00 0.80 0.01 0.30, 26.35 0.80 0.01 0.30, 26.71 0.81 0.03 0.27, 27.06 0.81 0.03 0.27, 27.41 0.82 0.05 0.24, 27.76 0.82 0.05 0.24, 28.12 0.83 0.07 0.21, 28.47 0.83 0.07 0.21, 28.82 0.84 0.10 0.18, 29.18 0.84 0.10 0.18, 29.53 0.85 0.12 0.16, 29.88 0.85 0.12 0.16, 30.24 0.86 0.14 0.13, 30.59 0.86 0.14 0.13, 30.94 0.87 0.16 0.10, 31.29 0.87 0.16 0.10, 31.65 0.88 0.18 0.07, 32.00 0.88 0.18 0.07, 32.35 0.89 0.21 0.04, 32.71 0.89 0.21 0.04, 33.06 0.90 0.23 0.02, 33.41 0.90 0.23 0.02, 33.76 0.91 0.25 0.00, 34.12 0.91 0.25 0.00, 34.47 0.93 0.27 0.00, 34.82 0.93 0.27 0.00, 35.18 0.94 0.29 0.00, 35.53 0.94 0.29 0.00, 35.88 0.95 0.31 0.00, 36.24 0.95 0.31 0.00, 36.59 0.96 0.34 0.00, 36.94 0.96 0.34 0.00, 37.29 0.97 0.36 0.00, 37.65 0.97 0.36 0.00, 38.00 0.98 0.38 0.00, 38.35 0.98 0.38 0.00, 38.71 0.99 0.40 0.00, 39.06 0.99 0.40 0.00, 39.41 1.00 0.42 0.00, 39.76 1.00 0.42 0.00, 40.12 1.00 0.44 0.00, 40.47 1.00 0.44 0.00, 40.82 1.00 0.45 0.00, 41.18 1.00 0.45 0.00, 41.53 1.00 0.47 0.00, 41.88 1.00 0.47 0.00, 42.24 1.00 0.48 0.00, 42.59 1.00 0.48 0.00, 42.94 1.00 0.49 0.00, 43.29 1.00 0.49 0.00, 43.65 1.00 0.51 0.00, 44.00 1.00 0.51 0.00, 44.35 1.00 0.52 0.00, 44.71 1.00 0.52 0.00, 45.06 1.00 0.53 0.00, 45.41 1.00 0.53 0.00, 45.76 1.00 0.55 0.00, 46.12 1.00 0.55 0.00, 46.47 1.00 0.56 0.00, 46.82 1.00 0.56 0.00, 47.18 1.00 0.58 0.00, 47.53 1.00 0.58 0.00, 47.88 1.00 0.59 0.00, 48.24 1.00 0.59 0.00, 48.59 1.00 0.60 0.00, 48.94 1.00 0.60 0.00, 49.29 1.00 0.62 0.00, 49.65 1.00 0.62 0.00, 50.00 1.00 0.63 0.00, 50.35 1.00 0.63 0.00, 50.71 1.00 0.65 0.00, 51.06 1.00 0.65 0.00, 51.41 1.00 0.66 0.00, 51.76 1.00 0.66 0.00, 52.12 1.00 0.67 0.00, 52.47 1.00 0.67 0.00, 52.82 1.00 0.69 0.00, 53.18 1.00 0.69 0.00, 53.53 1.00 0.70 0.00, 53.88 1.00 0.70 0.00, 54.24 1.00 0.71 0.00, 54.59 1.00 0.71 0.00, 54.94 1.00 0.73 0.00, 55.29 1.00 0.73 0.00, 55.65 1.00 0.74 0.00, 56.00 1.00 0.74 0.00, 56.35 1.00 0.76 0.00, 56.71 1.00 0.76 0.00, 57.06 1.00 0.77 0.00, 57.41 1.00 0.77 0.00, 57.76 1.00 0.78 0.00, 58.12 1.00 0.78 0.00, 58.47 1.00 0.80 0.00, 58.82 1.00 0.80 0.00, 59.18 1.00 0.81 0.00, 59.53 1.00 0.81 0.00, 59.88 1.00 0.83 0.00, 60.24 1.00 0.83 0.00, 60.59 1.00 0.84 0.00, 60.94 1.00 0.84 0.00, 61.29 1.00 0.85 0.00, 61.65 1.00 0.85 0.00, 62.00 1.00 0.87 0.00, 62.35 1.00 0.87 0.00, 62.71 1.00 0.88 0.00, 63.06 1.00 0.88 0.00, 63.41 1.00 0.89 0.00, 63.76 1.00 0.89 0.00, 64.12 1.00 0.91 0.00, 64.47 1.00 0.91 0.00, 64.82 1.00 0.93 0.00, 65.18 1.00 0.93 0.00, 65.53 1.00 0.94 0.00, 65.88 1.00 0.94 0.00, 66.24 1.00 0.95 0.02, 66.59 1.00 0.95 0.02, 66.94 1.00 0.96 0.07, 67.29 1.00 0.96 0.07, 67.65 1.00 0.98 0.13, 68.00 1.00 0.98 0.13, 68.35 1.00 0.99 0.18, 68.71 1.00 0.99 0.18, 69.06 1.00 1.00 0.22, 69.41 1.00 1.00 0.22, 69.76 1.00 1.00 0.27, 70.12 1.00 1.00 0.27, 70.47 1.00 1.00 0.33, 70.82 1.00 1.00 0.33, 71.18 1.00 1.00 0.38, 71.53 1.00 1.00 0.38, 71.88 1.00 1.00 0.42, 72.24 1.00 1.00 0.42, 72.59 1.00 1.00 0.47, 72.94 1.00 1.00 0.47, 73.29 1.00 1.00 0.53, 73.65 1.00 1.00 0.53, 74.00 1.00 1.00 0.58, 74.35 1.00 1.00 0.58, 74.71 1.00 1.00 0.62, 75.06 1.00 1.00 0.62, 75.41 1.00 1.00 0.67, 75.76 1.00 1.00 0.67, 76.12 1.00 1.00 0.73, 76.47 1.00 1.00 0.73, 76.82 1.00 1.00 0.78, 77.18 1.00 1.00 0.78, 77.53 1.00 1.00 0.82, 77.88 1.00 1.00 0.82, 78.24 1.00 1.00 0.87, 78.59 1.00 1.00 0.87, 78.94 1.00 1.00 0.93, 79.29 1.00 1.00 0.93, 79.65 1.00 1.00 0.98, 80.00 1.00 1.00 0.98]",
        lut_alpha="[-10.00 0, -9.65 0.75, -9.29 0.75, -8.94 0.75, -8.59 0.75, -8.24 0.75, -7.88 0.75, -7.53 0.75, -7.18 0.75, -6.82 0.75, -6.47 0.75, -6.12 0.75, -5.76 0.75, -5.41 0.75, -5.06 0.75, -4.71 0.75, -4.35 0.75, -4.00 0.75, -3.65 0.75, -3.29 0.75, -2.94 0.75, -2.59 0.75, -2.24 0.75, -1.88 0.75, -1.53 0.75, -1.18 0.75, -0.82 0.75, -0.47 0.75, -0.12 0.75, 0.24 0.75, 0.759 0.75, 0.94 0.75, 1.29 0.75, 1.65 0.75, 2.00 0.75, 2.35 0.75, 2.71 0.75, 3.06 0.75, 3.41 0.75, 3.76 0.75, 4.12 0.75, 4.47 0.75, 4.82 0.75, 5.18 0.75, 5.53 0.75, 5.88 0.75, 6.24 0.75, 6.59 0.75, 6.94 0.75, 7.29 0.75, 7.65 0.75, 8.00 0.75, 8.35 0.75, 8.71 0.75, 9.06 0.75, 9.41 0.75, 9.76 0.75, 10.12 0.75, 10.47 0.75, 10.82 0.75, 11.18 0.75, 11.53 0.75, 11.88 0.75, 12.24 0.75, 12.59 0.75, 12.94 0.75, 13.29 0.75, 13.65 0.75, 14.00 0.75, 14.35 0.75, 14.71 0.75, 15.06 0.75, 15.41 0.75, 15.76 0.75, 16.12 0.75, 16.47 0.75, 16.82 0.75, 17.18 0.75, 17.53 0.75, 17.88 0.75, 18.24 0.75, 18.59 0.75, 18.94 0.75, 19.29 0.75, 19.65 0.75, 20.00 0.75, 20.35 0.75, 20.71 0.75, 21.06 0.75, 21.41 0.75, 21.76 0.75, 22.12 0.75, 22.47 0.75, 22.82 0.75, 23.18 0.75, 23.53 0.75, 23.88 0.75, 24.24 0.75, 24.59 0.75, 24.94 0.75, 25.29 0.75, 25.65 0.75, 26.00 0.75, 26.35 0.75, 26.71 0.75, 27.06 0.75, 27.41 0.75, 27.76 0.75, 28.12 0.75, 28.47 0.75, 28.82 0.75, 29.18 0.75, 29.53 0.75, 29.88 0.75, 30.24 0.75, 30.59 0.75, 30.94 0.75, 31.29 0.75, 31.65 0.75, 32.00 0.75, 32.35 0.75, 32.71 0.75, 33.06 0.75, 33.41 0.75, 33.76 0.75, 34.12 0.75, 34.47 0.75, 34.82 0.75, 35.18 0.75, 35.53 0.75, 35.88 0.75, 36.24 0.75, 36.59 0.75, 36.94 0.75, 37.29 0.75, 37.65 0.75, 38.00 0.75, 38.35 0.75, 38.71 0.75, 39.06 0.75, 39.41 0.75, 39.76 0.75, 40.12 0.75, 40.47 0.75, 40.82 0.75, 41.18 0.75, 41.53 0.75, 41.88 0.75, 42.24 0.75, 42.59 0.75, 42.94 0.75, 43.29 0.75, 43.65 0.75, 44.00 0.75, 44.35 0.75, 44.71 0.75, 45.06 0.75, 45.41 0.75, 45.76 0.75, 46.12 0.75, 46.47 0.75, 46.82 0.75, 47.18 0.75, 47.53 0.75, 47.88 0.75, 48.24 0.75, 48.59 0.75, 48.94 0.75, 49.29 0.75, 49.65 0.75, 50.00 0.75, 50.35 0.75, 50.71 0.75, 51.06 0.75, 51.41 0.75, 51.76 0.75, 52.12 0.75, 52.47 0.75, 52.82 0.75, 53.18 0.75, 53.53 0.75, 53.88 0.75, 54.24 0.75, 54.59 0.75, 54.94 0.75, 55.29 0.75, 55.65 0.75, 56.00 0.75, 56.35 0.75, 56.71 0.75, 57.06 0.75, 57.41 0.75, 57.76 0.75, 58.12 0.75, 58.47 0.75, 58.82 0.75, 59.18 0.75, 59.53 0.75, 59.88 0.75, 60.24 0.75, 60.59 0.75, 60.94 0.75, 61.29 0.75, 61.65 0.75, 62.00 0.75, 62.35 0.75, 62.71 0.75, 63.06 0.75, 63.41 0.75, 63.76 0.75, 64.12 0.75, 64.47 0.75, 64.82 0.75, 65.18 0.75, 65.53 0.75, 65.88 0.75, 66.24 0.75, 66.59 0.75, 66.94 0.75, 67.29 0.75, 67.65 0.75, 68.00 0.75, 68.35 0.75, 68.71 0.75, 69.06 0.75, 69.41 0.75, 69.76 0.75, 70.12 0.75, 70.47 0.75, 70.82 0.75, 71.18 0.75, 71.53 0.75, 71.88 0.75, 72.24 0.75, 72.59 0.75, 72.94 0.75, 73.29 0.75, 73.65 0.75, 74.00 0.75, 74.35 0.75, 74.71 0.75, 75.06 0.75, 75.41 0.75, 75.76 0.75, 76.12 0.75, 76.47 0.75, 76.82 0.75, 77.18 0.75, 77.53 0.75, 77.88 0.75, 78.24 0.75, 78.59 0.75, 78.94 0.75, 79.29 0.75, 79.65 0.75, 80.00 0.75]",
        lut_range=[0, 100],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
        color_interpolation_invert="InterpolateHLS",
        lut_color_invert="[ 0 0 0 0, 144.425 0 0 0, 144.429 1 0 0, 4095 0 0 1]",
        lut_alpha_invert="[ 4095 0, 3950.429 0, 3950.425 1, 0 1 ]",
    ),
    LegacyLUT(
        title="TRANSPARENT_WINDOW",
        lut_color="[0 1 0 0, 33.3333 1 0 0, 33.3333 0 0 1, 100 0 0 1]",
        lut_alpha="[0 1, 33.3333 0, 33.3333 0, 100 1]",
        lut_range=[0, 100],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
        color_interpolation_invert="InterpolateRGB",
        lut_color_invert="[0 0 0 1, 33.3333 0 0 1, 33.3333 1 0 0, 100 1 0 0]",
        lut_alpha_invert="[0 1, 33.3333 0, 33.3333 0, 100 1]",
    ),
    LegacyLUT(
        title="VERTEBRA",
        lut_color="[0 0 0 0, 1 1 0 0, 2 0.88 1 0, 3 0 1 0.23, 4 0 0.65 1, 5 0.47 0 1, 6 1 0 0.42, 7 0.71 0.497 0, 8 0.1278 0.71 0, 9 0 0.71 0.6603, 10 0.0355 0 0.71, 11 0.71 0 0.5893, 12 0.5 0.14 0, 13 0.3 0.5 0, 14 0 0.5 0.26, 15 0 0.185 0.5, 16 0.375 0 0.5, 17 0.5 0 0.065, 18 1 0.99 0.34, 19 0.34 1 0.41, 20 0.34 0.85 1, 21 0.56 0.34 1, 22 1 0.34 0.7, 23 0.71 0.5112 0.2414, 24 0.3905 0.71 0.2414, 25 0.2414 0.71 0.6177, 26 0.2414 0.2769 0.71, 27 0.71 0.2414 0.6958, 28 0.5 0.22 0.17, 29 0.41 0.5 0.17, 30 0.17 0.5 0.295, 31 0.17 0.335 0.5, 32 0.375 0.17 0.5, 33 0.5 0.17 0.26, 34 1 0.94 0.57, 35 0.58 1 0.57, 36 0.57 0.97 1, 37 0.66 0.57 1, 38 1 0.57 0.86, 39 0.71 0.5325 0.4047, 40 0.5396 0.71 0.4047, 41 0.4047 0.71 0.6106, 42 0.4047 0.4686 0.71, 43 0.6816 0.4047 0.71, 44 0.5 0.29 0.285, 45 0.47 0.5 0.285, 46 0.285 0.5 0.34, 47 0.285 0.42 0.5, 48 0.39 0.285 0.5, 49 0.5 0.285 0.37, 50 1 0.72 0, 51 0.17 1 0, 52 0 1 0.95, 53 0.07 0 1, 54 1 0 0.82, 55 0.71 0.213 0, 56 0.4118 0.71 0, 57 0 0.71 0.3763, 58 0 0.2485 0.71, 59 0.5467 0 0.71, 60 0.71 0 0.0852, 61 0.5 0.5 0, 62 0 0.5 0.06, 63 0 0.385 0.5, 64 0.175 0 0.5, 65 0.5 0 0.265, 66 1 0.72 0.34, 67 0.54 1 0.34, 68 0.34 1 0.88, 69 0.34 0.39 1, 70 1 0.34 0.97, 71 0.71 0.3195 0.2414, 72 0.5822 0.71 0.2414, 73 0.2414 0.71 0.426, 74 0.2414 0.4686 0.71, 75 0.5396 0.2414 0.71, 76 0.71 0.2414 0.3621, 77 0.5 0.455 0.17, 78 0.175 0.5 0.17, 79 0.17 0.465 0.5, 80 0.245 0.17 0.5, 81 0.5 0.17 0.39, 82 1 0.76 0.57, 83 0.75 1 0.57, 84 0.57 1 0.86, 85 0.57 0.66 1, 86 0.97 0.57 1, 87 0.71 0.4118 0.4047, 88 0.6674 0.71 0.4047, 89 0.4047 0.71 0.4899, 90 0.4047 0.5964 0.71, 91 0.5538 0.4047 0.71, 92 0.71 0.4047 0.5183, 93 0.5 0.44 0.285, 94 0.315 0.5 0.285, 95 0.285 0.5 0.495, 96 0.3 0.285 0.5, 97 0.5 0.285 0.455, 98 1 0.32 0, 99 0.57 1 0, 100 0 1 0.55, 101 1 0 0, 102 0.88 1 0, 103 0 1 0.23, 104 0 0.65 1, 105 0.47 0 1, 106 1 0 0.42, 107 0.71 0.497 0, 108 0.1278 0.71 0, 109 0 0.71 0.6603, 110 0.0355 0 0.71, 111 0.71 0 0.5893, 112 0.5 0.14 0, 113 0.3 0.5 0, 114 0 0.5 0.26, 115 0 0.185 0.5, 116 0.375 0 0.5, 117 0.5 0 0.065, 118 1 0.99 0.34, 119 0.34 1 0.41, 120 0.34 0.85 1, 121 0.56 0.34 1, 122 1 0.34 0.7, 123 0.71 0.5112 0.2414, 124 0.3905 0.71 0.2414, 125 0.2414 0.71 0.6177, 126 0.2414 0.2769 0.71, 127 0.71 0.2414 0.6958, 128 0.5 0.22 0.17, 129 0.41 0.5 0.17, 130 0.17 0.5 0.295, 131 0.17 0.335 0.5, 132 0.375 0.17 0.5, 133 0.5 0.17 0.26, 134 1 0.94 0.57, 135 0.58 1 0.57, 136 0.57 0.97 1, 137 0.66 0.57 1, 138 1 0.57 0.86, 139 0.71 0.5325 0.4047, 140 0.5396 0.71 0.4047, 141 0.4047 0.71 0.6106, 142 0.4047 0.4686 0.71, 143 0.6816 0.4047 0.71, 144 0.5 0.29 0.285, 145 0.47 0.5 0.285, 146 0.285 0.5 0.34, 147 0.285 0.42 0.5, 148 0.39 0.285 0.5, 149 0.5 0.285 0.37, 150 1 0.72 0, 151 0.17 1 0, 152 0 1 0.95, 153 0.07 0 1, 154 1 0 0.82, 155 0.71 0.213 0, 156 0.4118 0.71 0, 157 0 0.71 0.3763, 158 0 0.2485 0.71, 159 0.5467 0 0.71, 160 0.71 0 0.0852, 161 0.5 0.5 0, 162 0 0.5 0.06, 163 0 0.385 0.5, 164 0.175 0 0.5, 165 0.5 0 0.265, 166 1 0.72 0.34, 167 0.54 1 0.34, 168 0.34 1 0.88, 169 0.34 0.39 1, 170 1 0.34 0.97, 171 0.71 0.3195 0.2414, 172 0.5822 0.71 0.2414, 173 0.2414 0.71 0.426, 174 0.2414 0.4686 0.71, 175 0.5396 0.2414 0.71, 176 0.71 0.2414 0.3621, 177 0.5 0.455 0.17, 178 0.175 0.5 0.17, 179 0.17 0.465 0.5, 180 0.245 0.17 0.5, 181 0.5 0.17 0.39, 182 1 0.76 0.57, 183 0.75 1 0.57, 184 0.57 1 0.86, 185 0.57 0.66 1, 186 0.97 0.57 1, 187 0.71 0.4118 0.4047, 188 0.6674 0.71 0.4047, 189 0.4047 0.71 0.4899, 190 0.4047 0.5964 0.71, 191 0.5538 0.4047 0.71, 192 0.71 0.4047 0.5183, 193 0.5 0.44 0.285, 194 0.315 0.5 0.285, 195 0.285 0.5 0.495, 196 0.3 0.285 0.5, 197 0.5 0.285 0.455, 198 1 0.32 0, 199 0.57 1 0, 200 0 1 0.55]",
        lut_alpha="[0 0, 1 1, 2 1, 3 1, 4 1, 5 1, 6 1, 7 1, 8 1, 9 1, 10 1, 11 1, 12 1, 13 1, 14 1, 15 1, 16 1, 17 1, 18 1, 19 1, 20 1, 21 1, 22 1, 23 1, 24 1, 25 1, 26 1, 27 1, 28 1, 29 1, 30 1, 31 1, 32 1, 33 1, 34 1, 35 1, 36 1, 37 1, 38 1, 39 1, 40 1, 41 1, 42 1, 43 1, 44 1, 45 1, 46 1, 47 1, 48 1, 49 1, 50 1, 51 1, 52 1, 53 1, 54 1, 55 1, 56 1, 57 1, 58 1, 59 1, 60 1, 61 1, 62 1, 63 1, 64 1, 65 1, 66 1, 67 1, 68 1, 69 1, 70 1, 71 1, 72 1, 73 1, 74 1, 75 1, 76 1, 77 1, 78 1, 79 1, 80 1, 81 1, 82 1, 83 1, 84 1, 85 1, 86 1, 87 1, 88 1, 89 1, 90 1, 91 1, 92 1, 93 1, 94 1, 95 1, 96 1, 97 1, 98 1, 99 1, 100 1, 101 0.5, 102 0.5, 103 0.5, 104 0.5, 105 0.5, 106 0.5, 107 0.5, 108 0.5, 109 0.5, 110 0.5, 111 0.5, 112 0.5, 113 0.5, 114 0.5, 115 0.5, 116 0.5, 117 0.5, 118 0.5, 119 0.5, 120 0.5, 121 0.5, 122 0.5, 123 0.5, 124 0.5, 125 0.5, 126 0.5, 127 0.5, 128 0.5, 129 0.5, 130 0.5, 131 0.5, 132 0.5, 133 0.5, 134 0.5, 135 0.5, 136 0.5, 137 0.5, 138 0.5, 139 0.5, 140 0.5, 141 0.5, 142 0.5, 143 0.5, 144 0.5, 145 0.5, 146 0.5, 147 0.5, 148 0.5, 149 0.5, 150 0.5, 151 0.5, 152 0.5, 153 0.5, 154 0.5, 155 0.5, 156 0.5, 157 0.5, 158 0.5, 159 0.5, 160 0.5, 161 0.5, 162 0.5, 163 0.5, 164 0.5, 165 0.5, 166 0.5, 167 0.5, 168 0.5, 169 0.5, 170 0.5, 171 0.5, 172 0.5, 173 0.5, 174 0.5, 175 0.5, 176 0.5, 177 0.5, 178 0.5, 179 0.5, 180 0.5, 181 0.5, 182 0.5, 183 0.5, 184 0.5, 185 0.5, 186 0.5, 187 0.5, 188 0.5, 189 0.5, 190 0.5, 191 0.5, 192 0.5, 193 0.5, 194 0.5, 195 0.5, 196 0.5, 197 0.5, 198 0.5, 199 0.5, 200 0.5]",
        lut_range=[0, 200],
        color_interpolation="InterpolateRGB",
        lut_relative=False,
    ),
]
