from django.apps import AppConfig
from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.db.models.signals import post_migrate


def init_users_groups(sender, **kwargs):
    from django.contrib.auth.models import Group
    from guardian.management import create_anonymous_user
    from guardian.utils import get_anonymous_user

    try:
        anon = get_anonymous_user()
    except ObjectDoesNotExist:
        create_anonymous_user(sender, **kwargs)
        anon = get_anonymous_user()

    g_reg_anon, _ = Group.objects.get_or_create(
        name=settings.REGISTERED_AND_ANON_USERS_GROUP_NAME
    )
    anon.groups.add(g_reg_anon)

    g_reg, _ = Group.objects.get_or_create(
        name=settings.REGISTERED_USERS_GROUP_NAME
    )
    anon.groups.remove(g_reg)


def rename_site(sender, **kwargs):
    from django.contrib.sites.models import Site

    desired_domain = settings.SESSION_COOKIE_DOMAIN.lstrip(".")

    if settings.SITE_SERVER_PORT is not None:
        desired_domain += f":{settings.SITE_SERVER_PORT}"

    if not Site.objects.filter(domain=desired_domain).exists():
        # Set the domain of the main site id if one doesn't already exist
        s = Site.objects.get(pk=settings.SITE_ID)
        s.domain = desired_domain
        s.name = desired_domain.split(".")[0].replace("-", " ").title()
        s.save()


class CoreConfig(AppConfig):
    name = "grandchallenge.core"

    def ready(self):
        post_migrate.connect(init_users_groups, sender=self)
        post_migrate.connect(rename_site, sender=self)

        # noinspection PyUnresolvedReferences
        import grandchallenge.core.signals  # noqa: F401
