## ----globalOpts, include=FALSE------------------------------------------
library(knitr)
opts_chunk$set(concordance = TRUE,
               fig.align = "center",
               out.width = "1.2\\linewidth", ## default is 1 
               tidy = FALSE,
               comment = NA,
               fig.path = "Rmd_files/Rmd_Figures/Modelling/", 
               cache.path = "Rmd_files/Cache/Modelling/",
               # dev = "png",
               fig.pos = "H",
               crop = hook_pdfcrop) ## changes mar defaults...
##
options(digits = 8)


## ----Checkpoint, message=FALSE------------------------------------------
## (messages are omitted in this chunk)
##
# library(checkpoint)
# checkpoint(snapshot_date = "2022-11-15") 


## ----loadPackages, message=FALSE----------------------------------------
## (messages are omitted from this chunk)
##
library(dplyr)
library(kableExtra)
library(ggplot2)
library(tibble) ## function rownames_to_column()
library(multcomp)
library(mgcv)


## ----Settings-----------------------------------------------------------
Sys.setenv(lang = "en_US")
theme_set(theme_bw())

if (!dir.exists("Prepared_data_and_models")) {
  dir.create("Prepared_data_and_models")
}


## ----getData------------------------------------------------------------
d.maize <- readRDS(file = paste0("Prepared_data_and_models/", 
                                "d.maize_PreparedData.RDS"))


## ----OverviewHeadStr----------------------------------------------------
dim(d.maize)
head(d.maize)[1:min(ncol(d.maize), 30)]
str(d.maize)


## ----createWellsMatrix, echo=FALSE--------------------------------------
## (this chunk is not echoed)
##
M.wells <- matrix(data = letters[1:6], 
                  byrow = TRUE,
                  nrow = 3, ncol = 2)[c(3:1), ]
M.wells


## ----createPotsOnTableMatrix, echo=FALSE--------------------------------
## (this chunk is not echoed)
##
M.pots_on_table <- matrix(data = paste0(rep(LETTERS[1:3], each = 6),
                                        1:6), 
                          byrow = TRUE,
                          nrow = 3, ncol = 6)[c(3:1), ]
M.pots_on_table


## ----surv.days.to.germ--------------------------------------------------
gam.cob_weight.num <- gam(cob_weight.num ~ 
                            s(position_field_x_cm, 
                              position_field_y_cm) +
                            soil.fac +
                            depth +
                            seed.weight,
                          data = d.maize)
##
summary(gam.cob_weight.num)
plot(gam.cob_weight.num)


## ----surv.days.to.germ2-------------------------------------------------
# The response variable is amount, thus we log transform it and refit the model
gam.cob_weight.num.log <- gam(log(cob_weight.num) ~ s(position_field_x_cm,
                                                      position_field_y_cm) +
                                soil.fac +
                                depth +
                                seed.weight,
                              data = d.maize)
##
summary(gam.cob_weight.num.log)
plot(gam.cob_weight.num.log)


## -----------------------------------------------------------------------
gam.check(gam.cob_weight.num)


## -----------------------------------------------------------------------
gam.check(gam.cob_weight.num.log)


## ----additiveSmooth-----------------------------------------------------
k.x <- d.maize %>% 
  pull(position_field_x_cm) %>%  
  n_distinct()
k.y <- d.maize %>% 
  pull(position_field_y_cm) %>% 
  n_distinct()
gam.cob_weight.num.add <- gam(cob_weight.num ~ s(position_field_x_cm, k = k.x) + 
                                s(position_field_y_cm, k = k.y) +
                                soil.fac +
                                depth +
                                seed.weight,
                              data = d.maize)
summary(gam.cob_weight.num.add)
plot(gam.cob_weight.num.add)


## ----linear-------------------------------------------------------------
lm.cob_weight.num <- lm(cob_weight.num ~ position_field_x_cm + 
                          position_field_y_cm +
                          soil.fac +
                          depth +
                          seed.weight,
                        data = d.maize)
##
summary(lm.cob_weight.num)


## -----------------------------------------------------------------------
AIC(gam.cob_weight.num.add, 
    gam.cob_weight.num, 
    lm.cob_weight.num)
##
BIC(gam.cob_weight.num.add, 
    gam.cob_weight.num, 
    lm.cob_weight.num)


## -----------------------------------------------------------------------
drop1(lm.cob_weight.num, test = "F")


## -----------------------------------------------------------------------
plot(lm.cob_weight.num)


## ----lm.cob_weight_CI---------------------------------------------------
( CI.cob_weight <- confint(lm.cob_weight.num) )


## ----CIcovariates-------------------------------------------------------
## Store the estimated values as dataframe
( d.coef.cob_weight <- data.frame(coef.cob_weight = coef(lm.cob_weight.num)) )
##
d.CI.cob_weight <- as.data.frame(CI.cob_weight)

## Join the two dataframe by rowname
d.est.cob_weight <- left_join(rownames_to_column(d.coef.cob_weight), 
                              rownames_to_column(d.CI.cob_weight),
                              by = c("rowname" = "rowname"))  
##
## visualise the dataframe
d.est.cob_weight %>% 
  kable(caption = paste0("Estimates and 95\\% CI."),
        label = "tab_coef_cob_weight",
        booktabs = TRUE,
        longtable = TRUE,
        linesep = c("")) %>%
  # landscape() %>%
  kable_styling(font_size = 7,
                latex_options = c("striped", "repeat_header", "hold_position"))


## ----plot_CI------------------------------------------------------------
d.est.cob_weight %>% 
  filter(rowname != "(Intercept)") %>% 
  ggplot(mapping = aes(y = rowname, x = coef.cob_weight)) +
  geom_point() +
  geom_errorbar(mapping = aes(xmin = `2.5 %`, xmax = `97.5 %`)) +
  xlab("Confidence intervals") +
  theme(axis.title.y = element_blank()) +
  geom_vline(xintercept = 0, color = "darkgrey")


## ----soil.fac_contrast--------------------------------------------------
test.soil <- glht(lm.cob_weight.num,
                  linfct = mcp(soil.fac = "Tukey"))


## ----soil.fac_contrast2-------------------------------------------------
par("mar") ## the second value refers to the left margin (to be enlarged)
par(mar = c(5.1, 11, 4.1, 2.1))
plot(test.soil)


## ----citations----------------------------------------------------------
citation()
citation("mgcv")


## ----sessionInfo--------------------------------------------------------
sessionInfo()

