## ----globalOpts, include=FALSE------------------------------------------
library(knitr)
opts_chunk$set(concordance = TRUE,
               fig.align = "center",
               out.width = "1.2\\linewidth", ## default is 1 
               tidy = FALSE,
               comment = NA,
               fig.path = "Rmd_files/Rmd_Figures/Modelling/", 
               cache.path = "Rmd_files/Cache/Modelling/",
               # dev = "png",
               fig.pos = "H",
               crop = hook_pdfcrop) ## changes mar defaults...
##
options(digits = 8)


## ----Checkpoint, message=FALSE------------------------------------------
## (messages are omitted in this chunk)
##
# library(checkpoint)
# checkpoint(snapshot_date = "2022-11-15") 


## ----loadPackages, message=FALSE----------------------------------------
## (messages are omitted from this chunk)
##
library(dplyr)
library(kableExtra)
library(ggplot2)
library(tibble) ## function rownames_to_column()
library(multcomp) ## function glht
library(lme4) ## function glmer


## ----Settings-----------------------------------------------------------
Sys.setenv(lang = "en_US")
theme_set(theme_bw())

if (!dir.exists("Prepared_data_and_models")) {
  dir.create("Prepared_data_and_models")
}


## ----getData------------------------------------------------------------
d.maize <- readRDS(file = paste0("Prepared_data_and_models/", 
                                "d.maize_PreparedData.RDS"))


## ----OverviewHeadStr----------------------------------------------------
dim(d.maize)
head(d.maize)[1:min(ncol(d.maize), 30)]
str(d.maize)


## ----createWellsMatrix, echo=FALSE--------------------------------------
## (this chunk is not echoed)
##
M.wells <- matrix(data = letters[1:6], 
                  byrow = TRUE,
                  nrow = 3, ncol = 2)[c(3:1), ]
M.wells


## ----createPotsOnTableMatrix, echo=FALSE--------------------------------
## (this chunk is not echoed)
##
M.pots_on_table <- matrix(data = paste0(rep(LETTERS[1:3], each = 6),
                                        1:6), 
                          byrow = TRUE,
                          nrow = 3, ncol = 6)[c(3:1), ]
M.pots_on_table


## ----surv.days.to.germ--------------------------------------------------
bin.germinated.in.lab <- glmer(germinated.in.lab ~ 
                                 soil.fac +
                                 depth +
                                 seed.weight +
                                 (1 | pot.fac) +
                                 (1 | well.fac),
                               family = "binomial",
                               data = d.maize)

##
summary(bin.germinated.in.lab)


## ----ranef1-------------------------------------------------------------
(ranef.germinated.in.lab <- ranef(bin.germinated.in.lab))


## ----ranef2-------------------------------------------------------------
(st.ranef <- VarCorr(bin.germinated.in.lab))


## ----plot_pot_well------------------------------------------------------
ggplot(ranef.germinated.in.lab$pot.fac, mapping = aes(x = `(Intercept)`)) +
  geom_density() +
  geom_rug(alpha = 0.4)
##
ggplot(ranef.germinated.in.lab$well.fac, mapping = aes(x = `(Intercept)`)) +
  geom_density() +
  geom_rug(alpha = 0.4)


## ----range_ranef--------------------------------------------------------
range(ranef.germinated.in.lab$pot.fac)
range(ranef.germinated.in.lab$well.fac)


## ----CI_ranef, error=TRUE-----------------------------------------------
( CI.ranef.germinated.in.lab <- confint(bin.germinated.in.lab, parm = "theta_", 
                                        method = "profile") )
## error
( CI.ranef.germinated.in.lab <- confint(bin.germinated.in.lab, parm = "theta_", 
                                        method = "Wald") )
## Only NAs
set.seed(2023)
( CI.ranef.germinated.in.lab <- confint(bin.germinated.in.lab, parm = "theta_", 
                                        method = "boot") )


## -----------------------------------------------------------------------
drop1(bin.germinated.in.lab, test = "Chisq")


## ----lm.germinated.in.lab_CI--------------------------------------------
( CI.germinated.in.lab <- confint(bin.germinated.in.lab, parm = "beta_") )


## ----CIcovariates-------------------------------------------------------
## Store the estimated values as dataframe
( d.coef.germinated.in.lab <- 
    data.frame(fixef.germinated.in.lab = fixef(bin.germinated.in.lab)) )
##
d.CI.germinated.in.lab <- as.data.frame(CI.germinated.in.lab)

## Join the two dataframe by rowname
d.est.germinated.in.lab <- left_join(rownames_to_column(d.coef.germinated.in.lab), 
                                     rownames_to_column(d.CI.germinated.in.lab),
                                     by = c("rowname" = "rowname"))  
##
## visualise the dataframe
d.est.germinated.in.lab %>% 
  kable(caption = paste0("Estimates and 95\\% CI."),
        label = "tab_coef_germinated.in.lab",
        booktabs = TRUE,
        longtable = TRUE,
        linesep = c("")) %>%
  # landscape() %>%
  kable_styling(font_size = 7,
                latex_options = c("striped", "repeat_header", "hold_position"))


## ----CI_exp-------------------------------------------------------------
## exponential
d.est.germinated.in.lab.exp <- data.frame(d.est.germinated.in.lab$rowname,
                                          exp(d.est.germinated.in.lab$fixef.germinated.in.lab),
                                          exp(d.est.germinated.in.lab$`2.5 %`),
                                          exp(d.est.germinated.in.lab$`97.5 %`)) 
##
colnames(d.est.germinated.in.lab.exp) <- c("rowname",
                                           "exp.fixef.germinated.in.lab", 
                                           "2.5 %", 
                                           "97.5 %")
d.est.germinated.in.lab.exp %>% 
  kable(caption = paste0("Exponential of the estimates and 95\\% CI."),
        label = "tab_exp_fixef_germinated_in_lab",
        booktabs = TRUE,
        longtable = TRUE,
        linesep = c("")) %>%
  # landscape() %>%
  kable_styling(font_size = 7,
                latex_options = c("striped", "repeat_header", "hold_position"))


## ----plot_exp-----------------------------------------------------------
## Plot
d.est.germinated.in.lab.exp %>% 
  filter( rowname != "(Intercept)" ) %>% 
  ggplot(mapping = aes(y = rowname, x = exp.fixef.germinated.in.lab)) +
  geom_point() +
  geom_errorbar(mapping = aes(xmin = `2.5 %`, xmax = `97.5 %`)) +
  xlab("Confidence intervals") +
  theme(axis.title.y = element_blank()) +
  geom_vline(xintercept = 1, color = "darkgrey")


## ----soil.fac_contrast--------------------------------------------------
test.soil <- multcomp::glht(bin.germinated.in.lab,
                            linfct = mcp(soil.fac = "Tukey"))
##
summary(test.soil)


## ----soil.fac_contrast2-------------------------------------------------
par("mar") ## the second value refers to the left margin (to be enlarged)
par(mar = c(5.1, 11, 4.1, 2.1))
plot(test.soil)


## ----citations----------------------------------------------------------
citation()
citation("lme4")


## ----sessionInfo--------------------------------------------------------
sessionInfo()

