## ----globalOpts, include=FALSE------------------------------------------
library(knitr)
opts_chunk$set(concordance = TRUE,
               fig.align = "center",
               out.width = "1.2\\linewidth", ## default is 1 
               tidy = FALSE,
               comment = NA,
               fig.path = "Rmd_files/Rmd_Figures/GraphAnalysis/", 
               cache.path = "Rmd_files/Cache/GraphAnalysis/",
               # dev = "png",
               fig.pos = "H",
               crop = hook_pdfcrop) ## changes mar defaults...
##
options(digits = 2)


## ----Checkpoint, message=FALSE------------------------------------------
## (messages are omitted in this chunk)
##
library(checkpoint)
checkpoint(snapshot_date = "2022-11-15") 


## ----loadPackages, message=FALSE----------------------------------------
## (messages are omitted from this chunk)
##
library(dplyr)
library(kableExtra)
library(ggplot2)
library(lubridate)
library(stringr)
library(ggbeeswarm)
library(tableone) 
library(tidyr)
library(mgcv)
library(survival)
library(survminer)


## ----Settings-----------------------------------------------------------
Sys.setenv(lang = "en_US")
theme_set(theme_bw())

if (!dir.exists("Prepared_data_and_models")) {
  dir.create("Prepared_data_and_models")
}


## ----getData------------------------------------------------------------
d.maize <- readRDS(file = paste0("Prepared_data_and_models/", 
                                "d.maize_PreparedData.RDS"))


## ----OverviewHeadStr----------------------------------------------------
dim(d.maize)
head(d.maize)[1:min(ncol(d.maize), 30)]
str(d.maize)


## ----NonTable1Vars------------------------------------------------------
d.maize %>% 
  summarize(across(.cols = everything(), .fns = class)) %>% 
  pivot_longer(cols = everything()) %>% 
  filter( ! (value %in% c("numeric", "integer", "factor", "character"))) %>% 
  pull(name)


## ----tab1_cat_calc------------------------------------------------------
## get categorical variables
vars.cat <- lapply(X = d.maize, FUN = function(x){
  class(x) %in% c("character", "factor")
}) %>% 
  unlist()
## get variables with <= 10 different categories
vars.cat.10 <- d.maize %>%
  select(all_of(names(vars.cat)[vars.cat])) %>% 
  apply(MARGIN = 2, FUN = function(x){
    n_distinct(x, na.rm = TRUE) <= 10 
  })
vars.cat.10 <- names(vars.cat.10)[vars.cat.10]


## ----tab1_cat_calc_unshown----------------------------------------------
## categorical variables not presented in Table 1 (more than 10 levels):
vars.cat.not10 <- setdiff(names(vars.cat)[vars.cat], vars.cat.10)
vars.cat.not10


## ----Table1VariableSelection--------------------------------------------
vars.cont <- d.maize %>% 
  select(where(is.numeric)) %>% 
  colnames()
vars.tab1 <- vars.cat.10
vars.tab1 <- colnames(d.maize)[colnames(d.maize) %in% vars.tab1]


## ----tab1_cat_calc2_invisible}, warning=FALSE, echo=TRUE, include=FALSE----
## (output and warnings are omitted in this chunk
##
## calculate table 1
tab1 <- CreateTableOne(
  vars = as.character(vars.tab1), 
  data = d.maize[, colnames(d.maize) %in% vars.tab1], 
  test = FALSE) 
tab1.print <- tableone:::print.TableOne(x = tab1,
                                        showAllLevels = FALSE, 
                                        nonnormal = TRUE, 
                                        dropEqual = FALSE, 
                                        minMax = FALSE, 
                                        explain = TRUE, 
                                        missing = TRUE) 
tab1.res_indent <- seq_along(rownames(tab1.print))
tab1.res_indent <- tab1.res_indent[str_detect(string = rownames(tab1.print),
                                              pattern = "   ")]


## ----tab1_cat_calc2}, warning=FALSE, echo=TRUE, include=TRUE, eval = FALSE----
## ## (output and warnings are omitted in this chunk
## ##
## ## calculate table 1
## tab1 <- CreateTableOne(
##   vars = as.character(vars.tab1),
##   data = d.maize[, colnames(d.maize) %in% vars.tab1],
##   test = FALSE)
## tab1.print <- tableone:::print.TableOne(x = tab1,
##                                         showAllLevels = FALSE,
##                                         nonnormal = TRUE,
##                                         dropEqual = FALSE,
##                                         minMax = FALSE,
##                                         explain = TRUE,
##                                         missing = TRUE)
## tab1.print <- tableone:::print.TableOne(x = tab1,
##                                         showAllLevels = FALSE,
##                                         nonnormal = TRUE,
##                                         dropEqual = FALSE,
##                                         minMax = FALSE,
##                                         explain = TRUE,
##                                         missing = TRUE)
## tab1.res_indent <- seq_along(rownames(tab1.print))
## tab1.res_indent <- tab1.res_indent[str_detect(string = rownames(tab1.print),
##                                               pattern = "   ")]


## ----tab1_cat_print-----------------------------------------------------
## print table
tab1.print %>% 
  as_tibble(rownames = " ") %>% 
  mutate(` ` = str_sub(string = ` `, start = 1, end = 70)) %>% 
  kable(caption = paste0("Descriptive statistics for categorical variables: ",
                         "frequencies and percentages in brackets are displayed. ",
                         "Only one category is shown for binary variables."),
        label = "tab1_categorical", 
        booktabs=TRUE,
        longtable = TRUE, 
        linesep = c("")) %>%
  # landscape() %>% 
  #column_spec(column = 1, width = "5in") %>%
  kable_styling(font_size = 7,
                latex_options = c("striped", "repeat_header", "hold_position")) %>% 
  add_indent(tab1.res_indent)
write.csv2(x = tab1.print,
           file = paste0("./Prepared_data_and_models/","d.maize_table1.csv"),
           fileEncoding = "UTF-8",
           quote = TRUE,
           row.names = TRUE)


## ----tab1_cont_calc-----------------------------------------------------
## calculate summary statistics
tab1.print.cont <- d.maize %>% 
  select(where(is.numeric)) %>% 
  tidyr::pivot_longer(cols = select(.data = d.maize,
                                    where(is.numeric), 
                                    -any_of("")) %>% 
                        colnames()) %>%  
  mutate(name = factor(x = name, levels = unique(name))) %>% 
  
  ## group_by (variable and strata)
  group_by_at(c("name",
                select(.data = d.maize,
                       any_of(c("name", ""))) %>% 
                  colnames())) %>%  
  summarize(
    mean = mean(value, na.rm = TRUE),
    sd = sd(value, na.rm = TRUE),
    median = median(value, na.rm = TRUE),
    quartiles.25 = quantile(x = value, probs = 0.25, na.rm = TRUE),
    quartiles.75 = quantile(x = value, probs = 0.75, na.rm = TRUE),
    IQR = IQR(value, na.rm = TRUE),
    n = sum(is.na(value)),
    n.perc = n / n() * 100, .groups = "keep") %>% 
  ungroup() %>% 
  
  ## rounding
  mutate(n = format(x = round(x = n, digits = 1),
                    nsmall = 0)) %>% 
  mutate(n.perc = format(x = round(x = n.perc, digits = 1),
                         nsmall = 1)) %>% 
  mutate(across(.cols = c(mean, sd, median, quartiles.25, quartiles.75, IQR), .fns = function(x){
    signif(x = x, digits = 3) %>% 
      as.character()
  })) %>% 
  mutate("Mean (SD)" = paste0(mean, " (", sd, ")")) %>% 
  mutate("Median [IQR]" = paste0(median, " [", quartiles.25, ", ", quartiles.75, "]")) %>% 
  mutate("Missings" = paste0(n, " (", n.perc, "%)")) %>% 
  select(name, any_of(c("", "Mean (SD)", "Median [IQR]", "Missings")))


## ----tab1_cont_print----------------------------------------------------
## print table
tab1.print.cont %>% 
  mutate(`name` = str_sub(string = `name`, start = 1, end = 70)) %>% 
  kable(caption = paste0("Descriptive statistics for continuous variables: ",
                         "mean and standard deviation (SD) as well as median ", 
                         "and interquartile range (IQR) are presented."),
        label = "tab1_continuous", 
        booktabs=TRUE,
        longtable = TRUE, 
        linesep = c("")) %>%
  # landscape() %>% 
  # column_spec(column = 1, width = "2.8in") %>% 
  kable_styling(font_size = 7,
                latex_options = c("striped", "repeat_header", "hold_position"))
write.csv2(x = tab1.print.cont,
           file = paste0("./Prepared_data_and_models/",
                         "d.maize_table1.continuous.csv"),
           fileEncoding = "UTF-8",
           quote = TRUE,
           row.names = FALSE)


## ----proto_gg_scatter---------------------------------------------------
f.gg.scatter <- function(data, x, y, group = NULL, 
                         fill = NULL, color = NULL,
                         plot.alpha = 1,
                         ...){
  
  if(!missing(x)){x <- enquo(x)}else{x = NULL}
  if(!missing(y)){y <- enquo(y)}else{y = NULL}
  if(!missing(group)){group <- enquo(group)}else{group = NULL}
  if(!missing(fill)){fill <- enquo(fill)}else{fill = NULL}
  if(!missing(color)){color <- enquo(color)}else{color = NULL}
  
  
  ## create ggplot object
  gg.ret <- data %>% 
    ggplot() +
    aes(x = !!x, y = !!y, color = !!color, fill = !!fill, group = !!group) + 
    
    geom_point(alpha = plot.alpha) + 
    geom_smooth(method = "loess", formula = "y ~ x", se = TRUE)
  
  ## project-specific tailoring
  gg.ret <- gg.ret +
    labs()
  
  return(gg.ret)
}


## ----proto_gg_density---------------------------------------------------
f.gg.density <- function(data, x, y, group = NULL, 
                         fill = NULL, color = NULL,
                         plot.alpha = 1,
                         ...){
  
  if(!missing(x)){x <- enquo(x)}else{x = NULL}
  if(!missing(y)){y <- enquo(y)}else{y = NULL}
  if(!missing(group)){group <- enquo(group)}else{group = NULL}
  if(!missing(fill)){fill <- enquo(fill)}else{fill = NULL}
  if(!missing(color)){color <- enquo(color)}else{color = NULL}
  
  ## create ggplot object
  gg.ret <- data %>%
    ggplot() +
    
    ## to not drop empty factor levels
    scale_color_discrete(drop = FALSE) +
    scale_fill_discrete(drop = FALSE) +
    
    aes(y = !!y, color = !!color, fill = !!fill, group = !!group) + 
    geom_density(alpha = plot.alpha) +
    geom_rug(alpha = plot.alpha) +
    coord_flip()
  
  ## project-specific tailoring
  gg.ret <- gg.ret +
    labs()
  
  return(gg.ret)
}


## ----proto_gg_boxplot---------------------------------------------------
f.gg.boxplot <- function(data, x, y, 
                         fill = NULL, color = NULL,
                         group = NULL,
                         plot.alpha = 1,
                         plot.beeswarm = FALSE,
                         ...){
  
  
  if(!missing(x)){x <- enquo(x)}else{x = NULL}
  if(!missing(y)){y <- enquo(y)}else{y = NULL}
  if(!missing(group)){group <- enquo(group)}else{group = NULL}
  if(!missing(fill)){fill <- enquo(fill)}else{fill = NULL}
  if(!missing(color)){color <- enquo(color)}else{color = NULL}
  
  ## create ggplot object
  gg.ret <- data %>%
    ggplot() +
    
    ## to not drop empty factor levels
    scale_color_discrete(drop = FALSE) +
    scale_fill_discrete(drop = FALSE) +
    
    aes(x = !!x, y = !!y, color = !!color, fill = !!fill, group = !!group)
  
  ## use beeswarm or not
  if(plot.beeswarm){
    gg.ret <- gg.ret +
      geom_boxplot(outlier.alpha = 0) + 
      geom_beeswarm(groupOnX = FALSE, alpha = plot.alpha)
  }else{
    gg.ret <- gg.ret +
      geom_boxplot(outlier.alpha = plot.alpha, alpha = plot.alpha)
  }
  
  ## project-specific tailoring
  gg.ret <- gg.ret +
    labs()
  
  return(gg.ret)
}


## ----gg_proto_function_builder------------------------------------------
f.gg.proto.builder <- function(data, x, y = NULL,
                               type, 
                               na.rm.pairwise = TRUE,
                               ...){
  
  ## needed for filter "na.rm.pairwise"
  if(!missing(x)){x <- enquo(x)}else{x = NULL}
  if(!missing(y)){y <- enquo(y)}else{y = NULL}
  
  ## remove missing values pairwise
  if(na.rm.pairwise & !is.null(x)){
    removed.rows <- data %>% 
      filter(is.na(!!x) | is.na((!!y))) %>% 
      nrow()
    if(removed.rows > 0){
      message(paste0("(m): Deleted ", removed.rows, 
                     " row(s) due to pairwise missing data."))
      data <- data %>% 
        filter(!is.na(!!x) & !is.na((!!y)))
    }
  }
  
  ## plot type
  if(type == "scatter"){
    gg.out <- f.gg.scatter(data = data, x = !!x, y = !!y, ...)
  }
  if(type == "density"){
    gg.out <- f.gg.density(data = data, y = !!y, ...)
  }
  if(type == "boxplot"){
    gg.out <- f.gg.boxplot(data = data, x = !!x, y = !!y, ...)
  }
  
  return(gg.out)
}


## -----------------------------------------------------------------------
## setting defaults
f.gg.proto <- function(...){
  f.gg.proto.builder(data = d.maize, 
                     plot.beeswarm = TRUE, 
                     na.rm.pairwise = TRUE,
                     plot.alpha = 0.3,
                     ... )
}


## ----gg_boxplot_cob_weight.num_versus_pot.fac---------------------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `pot.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_cob_weight.num_versus_soil.fac--------------------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `soil.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_cob_weight.num_versus_well.fac--------------------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `well.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_scatter_cob_weight.num_versus_depth-----------------------------
f.gg.proto(type = "scatter", y = `cob_weight.num`, x = `depth`)


## ----gg_scatter_cob_weight.num_versus_seed.weight.grams-----------------
f.gg.proto(type = "scatter", 
           y = `cob_weight.num`, 
           x = `seed.weight.grams`)


## -----------------------------------------------------------------------
table(d.maize$broken, useNA = "ifany")


## ----gg_boxplot_cob_weight.num_versus_broken----------------------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `broken`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_scatter_cob_weight.num_versus_date.germinated.asDate------------
f.gg.proto(type = "scatter", 
           y = `cob_weight.num`, 
           x = `date.germinated.asDate`)


## ----gg_boxplot_cob_weight.num_versus_germinated.in.lab-----------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `germinated.in.lab`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_cob_weight.num_versus_germinated.in.field---------------
f.gg.proto(type = "boxplot", y = `cob_weight.num`, x = `germinated.in.field`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = seed_coord_x, y = seed_coord_y)) +
  geom_point(mapping = aes(size = cob_weight.num, 
                           colour = cob_weight.num))


## -----------------------------------------------------------------------
gam.cob_weight.coord <- gam(cob_weight.num ~ s(seed_coord_x, 
                                               seed_coord_y,
                                               k = 30),
                            data = d.maize)
plot(gam.cob_weight.coord)


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = position_field_x_cm, y = position_field_y_cm)) +
  geom_point(mapping = aes(size = cob_weight.num, 
                           colour = cob_weight.num))


## -----------------------------------------------------------------------
gam.cob_weight.field <- gam(cob_weight.num ~ s(position_field_x_cm, 
                                               position_field_y_cm,
                                               k = 30),
                            data = d.maize)
plot(gam.cob_weight.field)


## ----gg_boxplot_height_2022_07_05.num_versus_pot.fac--------------------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `pot.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_height_2022_07_05.num_versus_soil.fac-------------------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `soil.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_height_2022_07_05.num_versus_well.fac-------------------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `well.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_scatter_height_versus_depth-------------------------------------
f.gg.proto(type = "scatter", y = `height_2022_07_05.num`, x = `depth`)


## ----gg_scatter_height_2022_07_05.num_versus_seed.weight.grams----------
f.gg.proto(type = "scatter", y = `height_2022_07_05.num`, x = `seed.weight.grams`)


## ----gg_boxplot_height_2022_07_05.num_versus_broken---------------------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `broken`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_height_2022_07_05.num_versus_germinated.in.lab----------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `germinated.in.lab`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_height_2022_07_05.num_versus_germinated.in.field--------
f.gg.proto(type = "boxplot", y = `height_2022_07_05.num`, x = `germinated.in.field`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = seed_coord_x, y = seed_coord_y)) +
  geom_point(mapping = aes(size = height_2022_07_05.num, 
                           colour = height_2022_07_05.num))


## -----------------------------------------------------------------------
gam.height.coord <- gam(height_2022_07_05.num ~ s(seed_coord_x, 
                                                  seed_coord_y,
                                                  k = 30),
                        data = d.maize)
plot(gam.height.coord)


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = position_field_x_cm, y = position_field_y_cm)) +
  geom_point(mapping = aes(size = height_2022_07_05.num, 
                           colour = height_2022_07_05.num))


## -----------------------------------------------------------------------
gam.height.field <- gam(height_2022_07_05.num ~ s(position_field_x_cm, 
                                                  position_field_y_cm,
                                                  k = 30),
                        data = d.maize)
plot(gam.height.field)


## ----tab_germinated.yes_versus_pot.fac----------------------------------
tab <- xtabs(~`germinated.yes` + `pot.fac`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_pot.fac----------------------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_pot.fac----------------------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----tab_germinated.yes_versus_soil.fac---------------------------------
tab <- xtabs(~`germinated.yes` + `soil.fac`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_soil.fac---------------------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_soil.fac---------------------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----tab_germinated.yes_versus_well.fac---------------------------------
tab <- xtabs(~`germinated.yes` + `well.fac`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_well.fac---------------------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_well.fac---------------------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----gg_boxplot_germinated.yes_versus_depth-----------------------------
f.gg.proto(type = "boxplot", y = `germinated.yes`, x = `depth`) + 
  scale_y_discrete(drop = FALSE)


## ----gg_boxplot_germinated.yes_versus_seed.weight.grams-----------------
f.gg.proto(type = "boxplot", y = `germinated.yes`, x = `seed.weight.grams`) + 
  scale_y_discrete(drop = FALSE)


## ----tab_germinated.yes_versus_broken-----------------------------------
tab <- xtabs(~`germinated.yes` + `broken`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_broken-----------------------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_broken-----------------------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----tab_germinated.yes_versus_germinated.in.lab------------------------
tab <- xtabs(~`germinated.yes` + `germinated.in.lab`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_germinated.in.lab------------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_germinated.in.lab------------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----tab_germinated.yes_versus_germinated.in.field----------------------
tab <- xtabs(~`germinated.yes` + `germinated.in.field`, 
             addNA = TRUE, 
             data = d.maize)
tab


## ----tab_prop1_germinated.yes_versus_germinated.in.field----------------
tab <- tab[!is.na(rownames(tab)), !is.na(colnames(tab))]
round(x = prop.table(x = tab, margin = 1) * 100, digits = 2)


## ----tab_plot1_germinated.yes_versus_germinated.in.field----------------
plot(x = tab, las = 1, main = "")
rm(tab)


## ----messages=FALSE-----------------------------------------------------
ggplot(d.maize, mapping = aes(x = seed_coord_x, y = seed_coord_y)) +
  geom_point(mapping = aes(size = germinated.yes, 
                           colour = germinated.yes))


## -----------------------------------------------------------------------
gam.germinated.yes.coord <- gam(germinated.yes ~ s(seed_coord_x, 
                                                   seed_coord_y,
                                                   k = 30),
                                data = d.maize)
plot(gam.germinated.yes.coord)


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = position_field_x_cm, y = position_field_y_cm)) +
  geom_point(mapping = aes(size = germinated.yes, 
                           colour = germinated.yes))


## -----------------------------------------------------------------------
gam.germinated.yes.field <- gam(germinated.yes ~ s(position_field_x_cm, 
                                                   position_field_y_cm,
                                                   k = 30),
                                data = d.maize)
plot(gam.germinated.yes.field)


## ----gg_boxplot_days.to.germination_versus_pot.fac----------------------
f.gg.proto(type = "boxplot", y = `days.to.germination`, x = `pot.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_days.to.germination_versus_soil.fac---------------------
f.gg.proto(type = "boxplot", y = `days.to.germination`, x = `soil.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_boxplot_days.to.germination_versus_well.fac---------------------
f.gg.proto(type = "boxplot", y = `days.to.germination`, x = `well.fac`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## ----gg_scatter_days.to.germination_versus_depth------------------------
f.gg.proto(type = "scatter", y = `days.to.germination`, x = `depth`)


## ----gg_scatter_days.to.germination_versus_seed.weight.grams------------
f.gg.proto(type = "scatter", y = `days.to.germination`, x = `seed.weight.grams`)


## ----gg_boxplot_days.to.germination_versus_broken-----------------------
f.gg.proto(type = "boxplot", y = `days.to.germination`, x = `broken`) +
  scale_x_discrete(drop = FALSE) +
  theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust = 1))


## -----------------------------------------------------------------------
ggplot(d.maize, mapping = aes(x = seed_coord_x, y = seed_coord_y)) +
  geom_point(mapping = aes(size = days.to.germination, 
                           colour = days.to.germination))


## -----------------------------------------------------------------------
gam.days.to.germination.coord <- gam(days.to.germination ~ s(seed_coord_x, 
                                                             seed_coord_y,
                                                             k = 30),
                                     data = d.maize)
plot(gam.days.to.germination.coord)


## ----sessionInfo--------------------------------------------------------
sessionInfo()

